/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * SpreadSheetConverterArchive.java
 * Copyright (C) 2013-2014 University of Waikato, Hamilton, New Zealand
 */
package adams.data.io.input;

import java.util.HashMap;

import adams.data.spreadsheet.SpreadSheet;
import adams.data.twitter.SpreadSheetConverter;
import adams.data.twitter.TwitterField;

/**
 <!-- globalinfo-start -->
 * Replays tweets stored in a CSV file (can be compressed with gzip) generated by the adams.data.twitter.SpreadSheetConverter converter.<br>
 * Extracts hashtags ('#...') and usermentions ('&#64;...') automatically.
 * <br><br>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * <pre>-logging-level &lt;OFF|SEVERE|WARNING|INFO|CONFIG|FINE|FINER|FINEST&gt; (property: loggingLevel)
 * &nbsp;&nbsp;&nbsp;The logging level for outputting errors and debugging output.
 * &nbsp;&nbsp;&nbsp;default: WARNING
 * </pre>
 * 
 * <pre>-archive &lt;adams.core.io.PlaceholderFile&gt; (property: archive)
 * &nbsp;&nbsp;&nbsp;The ADAMS CSV archive to load the tweets from.
 * &nbsp;&nbsp;&nbsp;default: ${CWD}
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision$
 */
public class SpreadSheetConverterArchive
  extends AbstractCsvArchive {

  /** for serialization. */
  private static final long serialVersionUID = -3432288686771377759L;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return
	"Replays tweets stored in a CSV file (can be compressed with gzip) "
	+ "generated by the " + SpreadSheetConverter.class.getName()
	+ " converter.\n"
	+ "Extracts hashtags ('#...') and usermentions ('@...') automatically.";
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  @Override
  public String archiveTipText() {
    return "The ADAMS CSV archive to load the tweets from.";
  }

  /**
   * Configures the mapping between column names and twitter fields.
   *
   * @param chunk	the spreadsheet to use as template
   * @return		the generated mapping
   */
  @Override
  protected HashMap<TwitterField,Integer> configureColumns(SpreadSheet chunk) {
    HashMap<TwitterField,Integer>	result;
    int					index;

    result = new HashMap<TwitterField,Integer>();
    for (TwitterField field: TwitterField.values()) {
      index = chunk.getHeaderRow().indexOfContent(field.toString());
      if (index > -1)
	result.put(field, index);
    }

    return result;
  }
}
