/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Exponential.java
 * Copyright (C) 2011-2012 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.stats.paintlet;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.util.Arrays;

import org.apache.commons.math.MathException;
import org.apache.commons.math.distribution.ExponentialDistributionImpl;

import adams.data.statistics.StatUtils;
import adams.gui.event.PaintEvent.PaintMoment;
import adams.gui.visualization.core.axis.Type;
import adams.gui.visualization.core.plot.Axis;

/**
 <!-- globalinfo-start -->
 * Paints the transformed exponential distribution
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-stroke-thickness &lt;float&gt; (property: strokeThickness)
 * &nbsp;&nbsp;&nbsp;The thickness of the stroke.
 * &nbsp;&nbsp;&nbsp;default: 1.0
 * &nbsp;&nbsp;&nbsp;minimum: 0.01
 * </pre>
 *
 * <pre>-color &lt;java.awt.Color&gt; (property: color)
 * &nbsp;&nbsp;&nbsp;Stroke color for the paintlet
 * &nbsp;&nbsp;&nbsp;default: #000000
 * </pre>
 *
 * <pre>-size &lt;int&gt; (property: size)
 * &nbsp;&nbsp;&nbsp;Size of the data points
 * &nbsp;&nbsp;&nbsp;default: 5
 * </pre>
 *
 * <pre>-fill-point (property: fillPoint)
 * &nbsp;&nbsp;&nbsp;Whether to fill the data point with solid color
 * </pre>
 *
 * <pre>-fill-color &lt;java.awt.Color&gt; (property: fillColor)
 * &nbsp;&nbsp;&nbsp;Color for filling data point
 * &nbsp;&nbsp;&nbsp;default: #ff0000
 * </pre>
 *
 <!-- options-end -->
 *
 * @author msf8
 * @version $Revision: 5341 $
 */
public class Exponential
extends AbstractProbabilityPaintlet{

  /** for serialization */
  private static final long serialVersionUID = -3000060668170218515L;

  public void setAxis() {
    m_AxisBottom = getPanel().getPlot().getAxis(Axis.BOTTOM);
    m_AxisLeft = getPanel().getPlot().getAxis(Axis.LEFT);

    m_AxisBottom.setType(Type.LOG_ABSOLUTE);
    m_AxisLeft.setType(Type.LOG_ABSOLUTE);
    m_AxisLeft.setNumberFormat("#.##");
  }
  /**
   * The paint routine of the paintlet.
   *
   * @param g		the graphics context to use for painting
   * @param moment	what {@link PaintMoment} is currently being painted
   */
  public void performPaint(Graphics g, PaintMoment moment) {
    if(m_Instances != null) {
      if (m_AntiAliasingEnabled)
	((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
      else
	((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_OFF);

      double median;
      double[] data = m_Instances.attributeToDoubleArray(m_Index);
      m_TransformedY = new double[data.length];
      Arrays.sort(data);
      for(int i = 0; i< data.length; i++) {
	median = ((i+1)-0.3)/(data.length +0.4);
	//calculate the transformed y values using inverse exponential
	m_TransformedY[i] = -1*(Math.log10(1-median));
      }
      //If the axis can handle the data
      if(m_AxisBottom.getType().canHandle(data[0], data[data.length-1])) {
	m_AxisBottom.setMinimum(data[0]);
	m_AxisBottom.setMaximum(data[data.length-1]);
      }
      else
      {
	System.err.println("errors in plotting");
      }
      if(m_AxisLeft.getType().canHandle(m_TransformedY[0], m_TransformedY[m_TransformedY.length -1]))
      {
	m_AxisLeft.setMinimum(m_TransformedY[0]);
	m_AxisLeft.setMaximum(m_TransformedY[m_TransformedY.length -1]);
      }
      else
      {
	System.err.println("errors in plotting");
      }
      m_AxisBottom.setAxisName(m_Instances.attribute(m_Index).name() + ")");
      m_AxisLeft.setAxisName("Inverse Exponential");

      for(int i = 0; i< data.length; i++) {
	Graphics2D g2d = (Graphics2D)g;
	//If data points are to be filled
	if(m_Fill) {
	  g2d.setColor(m_FillColor);
	  g2d.setStroke(new BasicStroke(0));
	  g2d.fillOval(m_AxisBottom.valueToPos(data[i])-m_Size/2, m_AxisLeft.valueToPos(m_TransformedY[i])-m_Size/2, m_Size, m_Size);
	}
	//outline of data point
	g2d.setStroke(new BasicStroke(m_StrokeThickness));
	g2d.setColor(m_Color);
	g2d.drawOval(m_AxisBottom.valueToPos(data[i])-m_Size/2, m_AxisLeft.valueToPos(m_TransformedY[i])-m_Size/2, m_Size, m_Size);
      }
      //if drawing regression fit diagonal
      if(m_RegressionLine) {
	g.setColor(Color.BLACK);
	double[] newData = new double[data.length];
	for(int i = 0; i < data.length; i++) {
	  newData[i] = Math.log(data[i]);
	}
	ExponentialDistributionImpl ex = new ExponentialDistributionImpl(StatUtils.mean(newData));
	//draw the expected diagonal line using the exponential distribution
	for(int i = 0; i< data.length-1; i++) {
	  double prob1;
	  try {
	    prob1 = ex.cumulativeProbability(newData[i]);
	  } catch (MathException e) {
	    prob1 = 0;
	  }
	  double prob2;
	  try {
	    prob2 = ex.cumulativeProbability(newData[i+1]);
	  } catch (MathException e) {
	    prob2 = 0;
	  }
	  double p1 = -Math.log(1-prob1);
	  double p2 = -Math.log(1-prob2);
	  g.drawLine(m_AxisBottom.valueToPos(data[i]), m_AxisLeft.valueToPos(p1), m_AxisBottom.valueToPos(data[i+1]), m_AxisLeft.valueToPos(p2));
	}
      }
    }
  }

  public String globalInfo() {
    return "Paints the transformed exponential distribution";
  }

  public boolean hasFitLine() {
    return false;

  }
}