/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * HistogramPaintlet.java
 * Copyright (C) 2011-2012 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.stats.paintlet;

import java.awt.Color;
import java.awt.Graphics;

import adams.gui.event.PaintEvent.PaintMoment;
import adams.gui.visualization.core.AxisPanel;
import adams.gui.visualization.core.plot.Axis;

/**
 <!-- globalinfo-start -->
 * Paints the histogram
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-stroke-thickness &lt;float&gt; (property: strokeThickness)
 * &nbsp;&nbsp;&nbsp;The thickness of the stroke.
 * &nbsp;&nbsp;&nbsp;default: 1.0
 * &nbsp;&nbsp;&nbsp;minimum: 0.01
 * </pre>
 *
 * <pre>-color &lt;java.awt.Color&gt; (property: color)
 * &nbsp;&nbsp;&nbsp;Stroke color for the paintlet
 * &nbsp;&nbsp;&nbsp;default: #000000
 * </pre>
 *
 * <pre>-fill-bins (property: fillBins)
 * &nbsp;&nbsp;&nbsp;Fill the bins with color
 * </pre>
 *
 * <pre>-fill-color &lt;java.awt.Color&gt; (property: fillColor)
 * &nbsp;&nbsp;&nbsp;color for filling the bins
 * &nbsp;&nbsp;&nbsp;default: #ff0000
 * </pre>
 *
 <!-- options-end -->
 *
 * @author msf8
 * @version $Revision: 5341 $
 */
public class HistogramPaintlet
extends AbstractColorPaintlet{

  /** for serialization */
  private static final long serialVersionUID = -3474738819482043957L;

  /** Y axis of the plot */
  protected AxisPanel m_AxisLeft;

  /**x axis of the plot */
  protected AxisPanel m_AxisBottom;

  /**Number of bins in the histogram */
  protected int m_NumBins;

  /** Width of each bin in the histogram */
  protected double m_BinWidth;

  /** 2d array containing bin min values and frequency values */
  protected double[][] m_Data;

  /** Name for x axis */
  protected String m_Name;

  /** Whether to fill the bins with color */
  protected boolean m_Fill;

  /** Color to fill the bins with if fill is chosen */
  protected Color m_FillColor;

  public PaintMoment getPaintMoment() {
    return PaintMoment.PAINT;
  }

  public void defineOptions() {
    super.defineOptions();

    //Fill bins
    m_OptionManager.add(
	"fill-bins", "fillBins", true);

    //color for fill
    m_OptionManager.add(
	"fill-color", "fillColor", Color.RED);
  }

  /**
   * Set whether the bins should be filled with color
   * @param val			True if bins filled
   */
  public void setFillBins(boolean val) {
    m_Fill = val;
    memberChanged();
  }

  /**
   * get whether the bins should be filled with color
   * @return			true if bins filled
   */
  public boolean getFillBins() {
    return m_Fill;
  }

  /**
   * Tip text for the fill bins property
   * @return				String describing the property
   */
  public String fillBinsTipText() {
    return "Fill the bins with color";
  }

  /**
   * Set the color for filling the bins
   * @param val			Color for fill
   */
  public void setFillColor(Color val) {
    m_FillColor = val;
    memberChanged();
  }

  /**
   * Get the color for filling the bins
   * @return			Color for fill
   */
  public Color getFillColor() {
    return m_FillColor;
  }

  /**
   * Tip text for the color property
   * @return			Color describing the property
   */
  public String fillColorTipText() {
    return "color for filling the bins";
  }
  /**
   * The paint routine of the paintlet.
   *
   * @param g		the graphics context to use for painting
   * @param moment	what {@link PaintMoment} is currently being painted
   */
  public void performPaint(Graphics g, PaintMoment moment) {
    if(m_Data != null) {
      m_AxisBottom = getPanel().getPlot().getAxis(Axis.BOTTOM);
      m_AxisLeft = getPanel().getPlot().getAxis(Axis.LEFT);
      //Display 5 values, may end up 6 with rounding
      m_AxisBottom.setNthValueToShow((int)(m_Data.length/5));
      m_AxisBottom.setMinimum(m_Data[0][0]);
      m_AxisBottom.setMaximum(m_Data[m_Data.length-1][0] + m_BinWidth);
      m_AxisBottom.setAxisName(m_Name);
      m_AxisBottom.setNumTicks(m_Data.length);
      m_AxisLeft.setMinimum(0);

      //find the maximum frequency for a bin
      double max = m_Data[1][0];
      for(int i = 1; i< m_Data.length; i++) {
	if(m_Data[i][1]> max) {
	  max = m_Data[i][1];
	}
      }
      //y axis shows number in bin/ width of bin
      m_AxisLeft.setMaximum(max/m_BinWidth);
      m_AxisLeft.setAxisName("Frequency");
      for(int i = 0; i< m_Data.length; i++) {
	double val = m_Data[i][1]/m_BinWidth;
	//If bin is filled with color
	if(m_Fill) {
	  g.setColor(m_FillColor);
	  g.fillRect(m_AxisBottom.valueToPos(m_Data[i][0]),
	      m_AxisLeft.valueToPos(val),
	      m_AxisBottom.valueToPos(m_Data[i][0]+ m_BinWidth)-m_AxisBottom.valueToPos(m_Data[i][0]),
	      m_AxisLeft.valueToPos(m_AxisLeft.getMinimum())-m_AxisLeft.valueToPos(val));
	}
	//Outline of bin
	g.setColor(m_Color);
	g.drawRect(m_AxisBottom.valueToPos(m_Data[i][0]),
	    m_AxisLeft.valueToPos(val),
	    m_AxisBottom.valueToPos(m_Data[i][0]+ m_BinWidth)-m_AxisBottom.valueToPos(m_Data[i][0]),
	    m_AxisLeft.valueToPos(m_AxisLeft.getMinimum())-m_AxisLeft.valueToPos(val));
      }
    }
  }

  public String globalInfo() {
    return "Paints the histogram";
  }

  /**
   * Set the number of bins in the histogram
   * @param val		Number of bins
   */
  public void setNumBins(int val) {
    m_NumBins = val;
    memberChanged();
  }

  /**
   * Set the data for plotting the histogram
   * @param data			2d array containing bin minimum values and frequencies
   * @param binWidth		Width of each bin
   * @param name			Name for x axis
   */
  public void setData(double[][] data, double binWidth, String name) {
    m_Data = data;
    m_BinWidth = binWidth;
    m_Name = name;
    memberChanged();
  }
}