/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    ClassificationViaRegression.java
 *    Copyright (C) 1999-2013 University of Waikato, Hamilton, New Zealand
 *
 */

package weka.classifiers.meta;

import weka.classifiers.AbstractClassifier;
import weka.classifiers.Classifier;
import weka.classifiers.SingleClassifierEnhancer;
import weka.classifiers.trees.M5P;
import weka.core.Capabilities;
import weka.core.Capabilities.Capability;
import weka.core.Instance;
import weka.core.Instances;
import weka.core.RevisionUtils;
import weka.core.TechnicalInformation;
import weka.core.TechnicalInformation.Field;
import weka.core.TechnicalInformation.Type;
import weka.core.TechnicalInformationHandler;
import weka.core.Utils;
import weka.filters.Filter;
import weka.filters.unsupervised.attribute.MakeIndicator;

/**
 <!-- globalinfo-start -->
 * Class for doing classification using regression methods. Class is binarized and one regression model is built for each class value. For more information, see, for example<br>
 * <br>
 * E. Frank, Y. Wang, S. Inglis, G. Holmes, I.H. Witten (1998). Using model trees for classification. Machine Learning. 32(1):63-76.
 * <br><br>
 <!-- globalinfo-end -->
 *
 <!-- technical-bibtex-start -->
 * BibTeX:
 * <pre>
 * &#64;article{Frank1998,
 *    author = {E. Frank and Y. Wang and S. Inglis and G. Holmes and I.H. Witten},
 *    journal = {Machine Learning},
 *    number = {1},
 *    pages = {63-76},
 *    title = {Using model trees for classification},
 *    volume = {32},
 *    year = {1998}
 * }
 * </pre>
 * <br><br>
 <!-- technical-bibtex-end -->
 *
 <!-- options-start -->
 * Valid options are: <br><br>
 *
 * <pre> -D
 *  If set, classifier is run in debug mode and
 *  may output additional info to the console</pre>
 *
 * <pre> -W
 *  Full name of base classifier.
 *  (default: weka.classifiers.trees.M5P)</pre>
 *
 * <pre>
 * Options specific to classifier weka.classifiers.trees.M5P:
 * </pre>
 *
 * <pre> -N
 *  Use unpruned tree/rules</pre>
 *
 * <pre> -U
 *  Use unsmoothed predictions</pre>
 *
 * <pre> -R
 *  Build regression tree/rule rather than a model tree/rule</pre>
 *
 * <pre> -M &lt;minimum number of instances&gt;
 *  Set minimum number of instances per leaf
 *  (default 4)</pre>
 *
 * <pre> -L
 *  Save instances at the nodes in
 *  the tree (for visualization purposes)</pre>
 *
 <!-- options-end -->
 *
 * @author Eibe Frank (eibe@cs.waikato.ac.nz)
 * @author Len Trigg (trigg@cs.waikato.ac.nz)
 * @author Dale Fletcher (dale at waikato dot ac dot nz)
 * @version $Revision$
*/
public class ClassificationViaRegressionD
  extends SingleClassifierEnhancer
  implements TechnicalInformationHandler {

  /** for serialization */
  static final long serialVersionUID = 4500023123618669859L;

  /** The classifiers. (One for each class.) */
  protected Classifier[] m_Classifiers;

  /** The filters used to transform the class. */
  protected MakeIndicator[] m_ClassFilters;

  /**
   * Default constructor.
   */
  public ClassificationViaRegressionD() {
    m_Classifier = new M5P();
  }

  /**
   * Returns a string describing classifier
   * @return a description suitable for
   * displaying in the explorer/experimenter gui
   */
  public String globalInfo() {
    return "Class for doing classification using regression methods. Class is "
      + "binarized and one regression model is built for each class value. For more "
      + "information, see, for example\n\n"
      + getTechnicalInformation().toString();
  }

  /**
   * Returns an instance of a TechnicalInformation object, containing
   * detailed information about the technical background of this class,
   * e.g., paper reference or book this class is based on.
   *
   * @return the technical information about this class
   */
  public TechnicalInformation getTechnicalInformation() {
    TechnicalInformation 	result;

    result = new TechnicalInformation(Type.ARTICLE);
    result.setValue(Field.AUTHOR, "E. Frank and Y. Wang and S. Inglis and G. Holmes and I.H. Witten");
    result.setValue(Field.YEAR, "1998");
    result.setValue(Field.TITLE, "Using model trees for classification");
    result.setValue(Field.JOURNAL, "Machine Learning");
    result.setValue(Field.VOLUME, "32");
    result.setValue(Field.NUMBER, "1");
    result.setValue(Field.PAGES, "63-76");

    return result;
  }

  /**
   * String describing default classifier.
   *
   * @return the default classifier classname
   */
  @Override
  protected String defaultClassifierString() {
    return M5P.class.getName();
  }

  /**
   * Returns default capabilities of the classifier.
   *
   * @return      the capabilities of this classifier
   */
  @Override
  public Capabilities getCapabilities() {
    Capabilities result = super.getCapabilities();

    // class
    result.disableAllClasses();
    result.disableAllClassDependencies();
    result.enable(Capability.NOMINAL_CLASS);

    return result;
  }

  /**
   * Builds the classifiers.
   *
   * @param insts the training data.
   * @throws Exception if a classifier can't be built
   */
  public void buildClassifier(Instances insts) throws Exception {
    Instances 	newInsts;
    int		i;

    // can classifier handle the data?
    getCapabilities().testWithFail(insts);

    // remove instances with missing class
    insts = new Instances(insts);
    insts.deleteWithMissingClass();

    m_Classifiers = AbstractClassifier.makeCopies(m_Classifier, insts.numClasses());
    m_ClassFilters = new MakeIndicator[insts.numClasses()];
    for (i = 0; i < insts.numClasses(); i++) {
      m_ClassFilters[i] = new MakeIndicator();
      m_ClassFilters[i].setAttributeIndex("" + (insts.classIndex() + 1));
      m_ClassFilters[i].setValueIndex(i);
      m_ClassFilters[i].setNumeric(true);
      m_ClassFilters[i].setInputFormat(insts);
      newInsts = Filter.useFilter(insts, m_ClassFilters[i]);
      m_Classifiers[i].buildClassifier(newInsts);
    }
  }

  /**
   * Returns the distribution for an instance.
   *
   * @param inst the instance to get the distribution for
   * @return the computed distribution
   * @throws Exception if the distribution can't be computed successfully
   */
  @Override
  public double[] distributionForInstance(Instance inst) throws Exception {
    double[] 	probs;
    Instance 	newInst;
    double 	sum;
    int		i;

    probs = new double[inst.numClasses()];
    sum   = 0;

    for (i = 0; i < inst.numClasses(); i++) {
      m_ClassFilters[i].input(inst);
      m_ClassFilters[i].batchFinished();
      newInst = m_ClassFilters[i].output();
      probs[i] = m_Classifiers[i].classifyInstance(newInst);
      if (probs[i] > 1)
        probs[i] = 1;
      if (probs[i] < 0 || Double.isNaN(probs[i]))
	probs[i] = 0;
      sum += probs[i];
    }
    
    if (sum != 0)
      Utils.normalize(probs, sum);
    
    return probs;
  }

  /**
   * Returns the classifier for the specified label index (0-based).
   * 
   * @param index	the index of the classifier to retrieve
   * @return		the classifier
   */
  public Classifier getClassifier(int index) {
    return m_Classifiers[index];
  }
  
  /**
   * Prints the classifiers.
   *
   * @return a string representation of the classifier
   */
  @Override
  public String toString() {
    StringBuilder 	text;
    
    if (m_Classifiers == null)
      return "Classification via Regression: No model built yet.";

    text = new StringBuilder();
    text.append("Classification via Regression\n\n");
    for (int i = 0; i < m_Classifiers.length; i++) {
      text.append("Classifier for class with index " + i + ":\n\n");
      text.append(m_Classifiers[i].toString() + "\n\n");
    }
    return text.toString();
  }

  /**
   * Returns the revision string.
   *
   * @return		the revision
   */
  @Override
  public String getRevision() {
    return RevisionUtils.extract("$Revision$");
  }

  /**
   * Main method for running this class.
   *
   * @param args	the options for the learner
   */
  public static void main(String[] args){
    runClassifier(new ClassificationViaRegressionD(), args);
  }
}
