/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * WekaEvaluationValues.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import weka.classifiers.Evaluation;
import adams.core.io.SpreadSheet;
import adams.core.io.SpreadSheet.Row;
import adams.flow.core.EvaluationHelper;
import adams.flow.core.EvaluationStatistic;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Generates a spreadsheet from statistics of an Evaluation object.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;weka.classifiers.Evaluation<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;adams.core.io.SpreadSheet<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 * &nbsp;&nbsp;&nbsp;If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: EvaluationValues
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-statistic &lt;adams.flow.core.EvaluationStatistic&gt; [-statistic ...] (property: statisticValues)
 * &nbsp;&nbsp;&nbsp;The evaluation values to extract and turn into CSV.
 * </pre>
 *
 * <pre>-index &lt;int&gt; (property: classIndex)
 * &nbsp;&nbsp;&nbsp;The class label index (eg used for AUC).
 * &nbsp;&nbsp;&nbsp;default: 1
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class WekaEvaluationValues
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = -1977976026411517458L;

  /** the comparison fields. */
  protected EvaluationStatistic[] m_StatisticValues;

  /** the index of the class label. */
  protected int m_ClassIndex;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Generates a spreadsheet from statistics of an Evaluation object.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "statistic", "statisticValues",
	    new EvaluationStatistic[]{
	      EvaluationStatistic.PERCENT_CORRECT,
	      EvaluationStatistic.ROOT_MEAN_SQUARED_ERROR,
	      EvaluationStatistic.ROOT_RELATIVE_SQUARED_ERROR});

    m_OptionManager.add(
	    "index", "classIndex",
	    1);
  }

  /**
   * Sets the values to extract.
   *
   * @param value	the value
   */
  public void setStatisticValues(EvaluationStatistic[] value) {
    m_StatisticValues = value;
    reset();
  }

  /**
   * Returns the values to extract.
   *
   * @return		the value
   */
  public EvaluationStatistic[] getStatisticValues() {
    return m_StatisticValues;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String statisticValuesTipText() {
    return "The evaluation values to extract and turn into CSV.";
  }

  /**
   * Sets the class label index (1-based).
   *
   * @param value	the label index
   */
  public void setClassIndex(int value) {
    m_ClassIndex = value;
    reset();
  }

  /**
   * Returns the current class label index (1-based).
   *
   * @return		the label index
   */
  public int getClassIndex() {
    return m_ClassIndex;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String classIndexTipText() {
    return "The class label index (eg used for AUC).";
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("classIndex");

    return ((variable == null) ? m_ClassIndex : variable) + ": " + m_StatisticValues.length + " values";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->weka.classifiers.Evaluation.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Evaluation.class};
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;
    double		value;
    Evaluation		eval;
    SpreadSheet		sheet;
    Row			row;

    result = null;

    // fill spreadsheet
    eval  = (Evaluation) m_InputToken.getPayload();
    sheet = new SpreadSheet();
    sheet.getHeaderRow().addCell("0").setContent("Statistic");
    sheet.getHeaderRow().addCell("1").setContent("Value");
    for (EvaluationStatistic statistic: m_StatisticValues) {
      try {
	value = EvaluationHelper.getValue(eval, statistic, m_ClassIndex);
	row   = sheet.addRow("" + sheet.getRowCount());
	row.addCell("0").setContent(statistic.toDisplay());
	row.addCell("1").setContent(Double.toString(value));
      }
      catch (Exception e) {
	getSystemErr().printStackTrace(e);
        result = "Error retrieving value for '" + statistic + "':\n" + e;
      }
    }

    // generate output token
    m_OutputToken = new Token(sheet);

    return result;
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->adams.core.io.SpreadSheet.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{SpreadSheet.class};
  }
}
