/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * WekaClassification.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.condition.bool;

import weka.classifiers.Classifier;
import weka.core.Capabilities;
import weka.core.Capabilities.Capability;
import weka.core.Instance;
import weka.core.SerializationHelper;
import weka.core.Utils;
import adams.core.io.PlaceholderFile;
import adams.flow.core.AbstractActor;
import adams.flow.core.GlobalActorHelper;
import adams.flow.core.GlobalActorReference;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Uses the index of the classification, i.e., the predicted label, as index of the switch
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-model &lt;adams.core.io.PlaceholderFile&gt; (property: modelFile)
 * &nbsp;&nbsp;&nbsp;The model file to load (when not pointing to a directory).
 * &nbsp;&nbsp;&nbsp;default: ${CWD}
 * </pre>
 * 
 * <pre>-model-actor &lt;adams.flow.core.GlobalActorReference&gt; (property: modelActor)
 * &nbsp;&nbsp;&nbsp;The global actor to use for obtaining the model in case serialized model 
 * &nbsp;&nbsp;&nbsp;file points to a directory.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-on-the-fly (property: onTheFly)
 * &nbsp;&nbsp;&nbsp;If set to true, the model file is not required to be present at set up time 
 * &nbsp;&nbsp;&nbsp;(eg if built on the fly), only at execution time.
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 5940 $
 */
public class WekaClassification
  extends AbstractBooleanCondition 
  implements IndexedBooleanCondition {

  /** for serialization. */
  private static final long serialVersionUID = 3278345095591806425L;

  /** the serialized model to load. */
  protected PlaceholderFile m_ModelFile;

  /** the global actor to get the model from. */
  protected GlobalActorReference m_ModelActor;

  /** the model that was loaded from the model file. */
  protected Classifier m_Model;

  /** whether the model gets built on the fly and might not be present at the start. */
  protected boolean m_OnTheFly;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  @Override
  public String globalInfo() {
    return
        "Uses the index of the classification, i.e., the predicted label, "
	+ "as index of the switch";
  }

  /**
   * Adds options to the internal list of options.
   */
  @Override
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "model", "modelFile",
	    new PlaceholderFile("."));

    m_OptionManager.add(
	    "model-actor", "modelActor",
	    new GlobalActorReference());

    m_OptionManager.add(
	    "on-the-fly", "onTheFly",
	    false);
  }

  /**
   * Sets the file to load the model from.
   *
   * @param value	the model file
   */
  public void setModelFile(PlaceholderFile value) {
    m_ModelFile = value;
    reset();
  }

  /**
   * Returns the file to load the model from.
   *
   * @return		the model file
   */
  public PlaceholderFile getModelFile() {
    return m_ModelFile;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String modelFileTipText() {
    return "The model file to load (when not pointing to a directory).";
  }

  /**
   * Sets the global actor to obtain the model from if model file is pointing
   * to a directory.
   *
   * @param value	the actor reference
   */
  public void setModelActor(GlobalActorReference value) {
    m_ModelActor = value;
    reset();
  }

  /**
   * Returns the global actor to obtain the model from if model file is pointing
   * to a directory.
   *
   * @return		the actor reference
   */
  public GlobalActorReference getModelActor() {
    return m_ModelActor;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String modelActorTipText() {
    return
        "The global actor to use for obtaining the model in case serialized "
      + "model file points to a directory.";
  }

  /**
   * Sets whether the model file gets built on the fly and might not be present
   * at start up time.
   *
   * @param value	if true then the model does not have to be present at
   * 			start up time
   */
  public void setOnTheFly(boolean value) {
    m_OnTheFly = value;
    reset();
  }

  /**
   * Returns whether the model file gets built on the fly and might not be present
   * at start up time.
   *
   * @return		true if the model is not necessarily present at start
   * 			up time
   */
  public boolean getOnTheFly() {
    return m_OnTheFly;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String onTheFlyTipText() {
    return
        "If set to true, the model file is not required to be present at "
      + "set up time (eg if built on the fly), only at execution time.";
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  @Override
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("modelFile");
    if (variable != null) {
      return variable;
    }
    else if (m_ModelFile != null) {
      if (m_ModelFile.isDirectory())
	return m_ModelActor.getValue();
      else
	return m_ModelFile.toString();
    }

    return null;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		Unknown
   */
  @Override
  public Class[] accepts() {
    return new Class[]{Instance.class};
  }

  /**
   * Loads the model from the model file.
   *
   * @param owner	the actor this condition belongs to
   * @return		null if everything worked, otherwise an error message
   */
  protected String setUpModel(AbstractActor owner) {
    String		result;
    String		msg;
    Capabilities	caps;

    result = null;

    if (m_ModelFile.isDirectory()) {
      // obtain model from global actor
      try {
	m_Model = (Classifier) GlobalActorHelper.getSetup(Classifier.class, m_ModelActor, owner);
      }
      catch (Exception e) {
	m_Model = null;
	msg     = "Failed to obtain model from global actor '" + m_ModelActor + "': ";
	result  = msg + e.toString();
	getSystemErr().println(msg);
	getSystemErr().printStackTrace(e);
      }
    }
    else {
      // load model
      try {
	m_Model = (Classifier) SerializationHelper.read(m_ModelFile.getAbsolutePath());
      }
      catch (Exception e) {
	m_Model = null;
	msg     = "Failed to load model from '" + m_ModelFile + "': ";
	result  = msg + e.toString();
	getSystemErr().println(msg);
	getSystemErr().printStackTrace(e);
      }
    }
    
    // can model handle nominal class attribute?
    if (m_Model != null) {
      caps = m_Model.getCapabilities();
      if (    !caps.handles(Capability.UNARY_CLASS) 
	   && !caps.handles(Capability.BINARY_CLASS) 
	   && !caps.handles(Capability.NOMINAL_CLASS) )
	result = "Model can neither handle unary, binary nor nominal class attribute!";
    }

    return result;
  }

  /**
   * Initializes the item for flow execution.
   *
   * @param owner	the actor this condition belongs to
   * @return		null if everything is fine, otherwise error message
   */
  @Override
  public String setUp(AbstractActor owner) {
    String	result;

    result = super.setUp(owner);

    if (result == null) {
      if (!m_OnTheFly)
	result = setUpModel(owner);
    }

    return result;
  }

  /**
   * Evaluates whether to executed the "then" or "else" branch.
   *
   * @param owner	the owning IfThenElse actor
   * @param token	the current token passing through the IfThenElse actor
   * @return		true if data matches capabilities
   */
  @Override
  protected boolean doEvaluate(AbstractActor owner, Token token) {
    return (getCaseIndex(owner, token) != -1);
  }

  /**
   * Returns the index of the case that should get executed.
   * 
   * @param owner	the owning actor
   * @param token	the current token passing through the actor
   * @return		the index, -1 if not available
   */
  public int getCaseIndex(AbstractActor owner, Token token) {
    int		result;
    double	classification;
    String	msg;
    Instance	inst;

    result = -1;

    if (m_OnTheFly) {
      msg = setUpModel(owner);
      if (msg != null) {
	getSystemErr().println(msg);
	return result;
      }
    }
    
    inst = ((Instance) token.getPayload());
    if (inst.classIndex() == -1) {
      getSystemErr().println("No class set!");
      return result;
    }
    if (!inst.classAttribute().isNominal()) {
      getSystemErr().println("Class attribute is not nominal!");
      return result;
    }

    try {
      classification = m_Model.classifyInstance(inst);
      if (Utils.isMissingValue(classification))
	result = -1;
      else
	result = (int) classification;
    }
    catch (Exception e) {
      getSystemErr().printStackTrace("Failed to obtain classification: ", e);
    }

    return result;
  }

  /**
   * Returns the index of the default case.
   * 
   * @param owner	the owning actor
   * @param token	the current token passing through the actor
   * @return		the index, -1 if not available
   */
  public int getDefaultCaseIndex(AbstractActor owner, Token token) {
    Instance	inst;
    
    inst = ((Instance) token.getPayload());
    
    return inst.classAttribute().numValues();
  }
}
