/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * ExplorerExt.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */
package weka.gui.explorer;

import java.awt.Container;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;

import javax.swing.JButton;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import weka.core.converters.AbstractFileLoader;
import weka.core.converters.AbstractFileSaver;
import weka.core.converters.ConverterUtils;
import weka.gui.ConverterFileChooser;
import weka.gui.sql.SqlViewerDialog;
import adams.gui.application.ChildFrame;
import adams.gui.application.ChildWindow;
import adams.gui.core.GUIHelper;
import adams.gui.core.MenuBarProvider;
import adams.gui.core.RecentFilesHandler;
import adams.gui.core.TitleGenerator;
import adams.gui.event.RecentFileEvent;
import adams.gui.event.RecentFileListener;

/**
 * An extended Explorer interface using menus instead of buttons, as well
 * as remembering recent files.
 * 
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 6178 $
 */
public class ExplorerExt
  extends Explorer
  implements MenuBarProvider {

  /** for serialization. */
  private static final long serialVersionUID = 173388459172775839L;

  /** the file to store the recent files in. */
  public final static String SESSION_FILE = "ExplorerSession.props";

  /** the menu bar, if used. */
  protected JMenuBar m_MenuBar;

  /** the "load recent" submenu. */
  protected JMenu m_MenuItemLoadRecent;

  /** the recent files handler. */
  protected RecentFilesHandler m_RecentFilesHandler;

  /** the save menu item. */
  protected JMenuItem m_MenuItemFileSave;

  /** the save as menu item. */
  protected JMenuItem m_MenuItemFileSaveAs;

  /** the load classifier menu item. */
  protected JMenuItem m_MenuItemFileLoadClassifier;

  /** the load clusterer menu item. */
  protected JMenuItem m_MenuItemFileLoadClusterer;

  /** the undo menu item. */
  protected JMenuItem m_MenuItemEditUndo;

  /** the edit data menu item. */
  protected JMenuItem m_MenuItemEditData;

  /** the undo button of the preprocess panel. */
  protected JButton m_ButtonUndo;

  /** the edit button of the preprocess panel. */
  protected JButton m_ButtonEdit;

  /** The file chooser for selecting data files */
  protected ConverterFileChooser m_FileChooser;
  
  /** the current file. */
  protected File m_CurrentFile;

  /** for generating the title. */
  protected TitleGenerator m_TitleGenerator;
  
  /**
   * Default constructor.
   */
  public ExplorerExt() {
    super();
    initialize();
    initGUI();
  }
  
  /**
   * Initializes the members.
   */
  protected void initialize() {
    m_RecentFilesHandler = null;
    m_FileChooser        = null;
    m_ButtonUndo         = null;
    m_ButtonEdit         = null;
    m_CurrentFile        = null;
    m_TitleGenerator     = new TitleGenerator("Explorer", true);
  }
  
  /**
   * Initializes the widgets.
   */
  protected void initGUI() {
    hideButtons(getPreprocessPanel());
    m_FileChooser = getPreprocessPanel().m_FileChooser;
  }

  /**
   * Hides the buttons of the preprocess panel.
   * 
   * @param cont	the container to search
   * @return		true if hidden
   */
  protected boolean hideButtons(Container cont) {
    boolean	result;
    int		i;
    
    result = false;
    
    for (i = 0; i < cont.getComponentCount(); i++) {
      if (cont.getComponent(i) instanceof JButton) {
	if (((JButton) cont.getComponent(i)).getText().equals("Edit...")) {
	  m_ButtonEdit = (JButton) cont.getComponent(i);
	  cont.setVisible(false);
	  result = true;
	}
	else if (((JButton) cont.getComponent(i)).getText().equals("Undo")) {
	  m_ButtonUndo = (JButton) cont.getComponent(i);
	}
      }
      else if (cont.getComponent(i) instanceof Container) {
	result = hideButtons((Container) cont.getComponent(i));
	if (result)
	  break;
      }
    }
    
    return result;
  }

  /**
   * Creates a menu bar (singleton per panel object). Can be used in frames.
   *
   * @return		the menu bar
   */
  public JMenuBar getMenuBar() {
    JMenuBar		result;
    JMenu		menu;
    JMenu		submenu;
    JMenuItem		menuitem;

    if (m_MenuBar == null) {
      result = new JMenuBar();

      // File
      menu = new JMenu("File");
      result.add(menu);
      menu.setMnemonic('F');
      menu.addChangeListener(new ChangeListener() {
	public void stateChanged(ChangeEvent e) {
	  updateMenu();
	}
      });

      // File/Open
      menuitem = new JMenuItem("Open...");
      menu.add(menuitem);
      menuitem.setMnemonic('o');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed O"));
      menuitem.setIcon(GUIHelper.getIcon("open.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  open();
	}
      });

      // File/Recent files
      submenu = new JMenu("Open recent");
      menu.add(submenu);
      m_RecentFilesHandler = new RecentFilesHandler(SESSION_FILE, 10, submenu);
      m_RecentFilesHandler.addRecentFileListener(new RecentFileListener() {
	public void recentFileAdded(RecentFileEvent e) {
	  // ignored
	}
	public void recentFileSelected(RecentFileEvent e) {
	  openRecent(e);
	}
      });
      m_MenuItemLoadRecent = submenu;

      // File/Load from URL
      menuitem = new JMenuItem("Load from URL...");
      menu.addSeparator();
      menu.add(menuitem);
      menuitem.setMnemonic('U');
      menuitem.setIcon(GUIHelper.getIcon("internet.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  loadFromURL();
	}
      });

      // File/Load from database
      menuitem = new JMenuItem("Load from database...");
      menu.add(menuitem);
      menuitem.setMnemonic('L');
      menuitem.setIcon(GUIHelper.getIcon("database.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  loadFromDatabase();
	}
      });

      // File/Generate
      menuitem = new JMenuItem("Generate...");
      menu.add(menuitem);
      menuitem.setMnemonic('G');
      menuitem.setIcon(GUIHelper.getEmptyIcon());
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  generate();
	}
      });

      // File/Save
      menuitem = new JMenuItem("Save");
      menu.addSeparator();
      menu.add(menuitem);
      menuitem.setMnemonic('S');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed S"));
      menuitem.setIcon(GUIHelper.getIcon("save.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  save();
	}
      });
      m_MenuItemFileSave = menuitem;

      // File/Save
      menuitem = new JMenuItem("Save as...");
      menu.add(menuitem);
      menuitem.setMnemonic('a');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl shift pressed S"));
      menuitem.setIcon(GUIHelper.getEmptyIcon());
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  saveAs();
	}
      });
      m_MenuItemFileSaveAs = menuitem;

      // File/Load classifier
      menuitem = new JMenuItem("Load classifier model...");
      menu.addSeparator();
      menu.add(menuitem);
      menuitem.setMnemonic('c');
      menuitem.setIcon(GUIHelper.getEmptyIcon());
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  loadClassifier();
	}
      });
      m_MenuItemFileLoadClassifier = menuitem;

      // File/Load classifier
      menuitem = new JMenuItem("Load clusterer model...");
      menu.add(menuitem);
      menuitem.setMnemonic('l');
      menuitem.setIcon(GUIHelper.getEmptyIcon());
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  loadClusterer();
	}
      });
      m_MenuItemFileLoadClusterer = menuitem;

      // File/Close
      menuitem = new JMenuItem("Close");
      menu.addSeparator();
      menu.add(menuitem);
      menuitem.setMnemonic('C');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed Q"));
      menuitem.setIcon(GUIHelper.getIcon("exit.png"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  close();
	}
      });

      // Edit
      menu = new JMenu("Edit");
      menu.setMnemonic('E');
      menu.addChangeListener(new ChangeListener() {
	public void stateChanged(ChangeEvent e) {
	  updateMenu();
	}
      });
      result.add(menu);

      // Edit/Undo
      menuitem = new JMenuItem("Undo");
      menuitem.setMnemonic('U');
      menuitem.setEnabled(canUndo());
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed Z"));
      menuitem.setIcon(GUIHelper.getIcon("undo.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  undo();
	}
      });
      menu.add(menuitem);
      m_MenuItemEditUndo = menuitem;

      // Edit/Data editor
      menuitem = new JMenuItem("Data editor");
      menuitem.setMnemonic('D');
      menuitem.setEnabled(canUndo());
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed E"));
      menuitem.setIcon(GUIHelper.getIcon("report.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  edit();
	}
      });
      menu.addSeparator();
      menu.add(menuitem);
      m_MenuItemEditData = menuitem;

      // Window
      menu = new JMenu("Window");
      menu.setMnemonic('W');
      menu.addChangeListener(new ChangeListener() {
	public void stateChanged(ChangeEvent e) {
	  updateMenu();
	}
      });
      result.add(menu);

      // Window/New window
      menuitem = new JMenuItem("New");
      menuitem.setMnemonic('N');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed N"));
      menuitem.setIcon(GUIHelper.getIcon("new.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  newWindow(false);
	}
      });
      menu.add(menuitem);

      // Window/New window
      menuitem = new JMenuItem("Duplicate");
      menuitem.setMnemonic('D');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl shift pressed N"));
      menuitem.setIcon(GUIHelper.getIcon("copy.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  newWindow(true);
	}
      });
      menu.add(menuitem);

      // update menu
      m_MenuBar = result;
      updateMenu();
    }
    else {
      result = m_MenuBar;
    }

    return result;
  }

  /**
   * Updates title and menu items.
   */
  protected void update() {
    updateTitle();
    updateMenu();
  }

  /**
   * Updates the title of the dialog.
   */
  protected void updateTitle() {
    String	title;

    title = m_TitleGenerator.generate(m_CurrentFile);
    setParentTitle(title);
  }

  /**
   * updates the enabled state of the menu items.
   */
  protected void updateMenu() {
    if (m_MenuBar == null)
      return;
    
    // file
    m_MenuItemFileSave.setEnabled(isDataLoaded() && (m_CurrentFile != null));
    m_MenuItemFileSaveAs.setEnabled(isDataLoaded());
    m_MenuItemLoadRecent.setEnabled(m_RecentFilesHandler.size() > 0);
    m_MenuItemFileLoadClassifier.setEnabled(isDataLoaded() && hasClassifyTab());
    m_MenuItemFileLoadClusterer.setEnabled(isDataLoaded() && hasClusterTab());
    
    // edit
    m_MenuItemEditUndo.setEnabled(canUndo());
    m_MenuItemEditData.setEnabled(canEdit());
  }
  
  /**
   * Returns the hashcode of the current dataset.
   * 
   * @return	the hashcode, null if no data loaded
   */
  protected Integer getDataHashcode() {
    Integer	result;
    
    result = null;
    
    if (getPreprocessPanel().getInstances() != null)
      result = getPreprocessPanel().getInstances().hashCode();
    
    return result;
  }
  
  /**
   * Lets the user choose a file.
   */
  public void open() {
    int 	retVal;
    
    retVal = m_FileChooser.showOpenDialog(ExplorerExt.this);
    if (retVal != ConverterFileChooser.APPROVE_OPTION)
      return;
    
    try {
      getPreprocessPanel().addUndoPoint();
    }
    catch (Exception ignored) {
      // ignored
    }

    if (m_FileChooser.getLoader() == null) {
      GUIHelper.showErrorMessage(ExplorerExt.this,
	  "Cannot determine file loader automatically!",
	  "Load Instances");
      return;
    }
    else {
      if (m_RecentFilesHandler != null)
	m_RecentFilesHandler.addRecentFile(m_FileChooser.getSelectedFile());
      getPreprocessPanel().setInstancesFromFile(m_FileChooser.getLoader());
      m_CurrentFile = m_FileChooser.getSelectedFile();
      update();
    }
  }
  
  /**
   * For opening a recently used file.
   * 
   * @param e		the event
   */
  public void openRecent(RecentFileEvent e) {
    AbstractFileLoader 	loader;
    
    loader = ConverterUtils.getLoaderForFile(e.getFile());
    if (loader == null) {
      GUIHelper.showErrorMessage(ExplorerExt.this, "Failed to determine file loader for the following file:\n" + e.getFile());
      return;
    }
    
    try {
      loader.setFile(e.getFile());
      getPreprocessPanel().setInstancesFromFile(loader);
      m_CurrentFile = e.getFile();
      update();
    }
    catch (Exception ex) {
      System.err.println("Failed to load file:\n" + e.getFile());
      ex.printStackTrace();
    }
  }
  
  /**
   * Lets the user load data from a database.
   */
  public void loadFromDatabase() {
    SqlViewerDialog 	dialog;
    
    dialog = new SqlViewerDialog(null);
    dialog.setVisible(true);
    if (dialog.getReturnValue() == JOptionPane.OK_OPTION) {
      getPreprocessPanel().setInstancesFromDBQ(dialog.getURL(), dialog.getUser(),
	  dialog.getPassword(), dialog.getQuery(),
	  dialog.getGenerateSparseData());
      m_CurrentFile = null;
      update();
    }
  }

  /**
   * Lets the user load data from a URL.
   */
  public void loadFromURL() {
    getPreprocessPanel().setInstancesFromURLQ();
    m_CurrentFile = null;
    update();
  }
  
  /**
   * Pops up a dialog that allows the user to generate data.
   */
  public void generate() {
    getPreprocessPanel().generateInstances();
    m_CurrentFile = null;
    update();
  }

  /**
   * Allows the user to save the file. Prompts user with dialog if no filename
   * set currently.
   */
  public void save() {
    AbstractFileSaver	saver;

    if (m_CurrentFile == null) {
      saveAs();
      return;
    }

    saver = ConverterUtils.getSaverForFile(m_CurrentFile);
    if (saver == null) {
      saveAs();
      return;
    }
    
    getPreprocessPanel().saveInstancesToFile(saver, getPreprocessPanel().getInstances());
  }

  /**
   * Allows the user to save the file. Prompts user with dialog.
   */
  public void saveAs() {
    getPreprocessPanel().saveWorkingInstancesToFileQ();
    m_CurrentFile = m_FileChooser.getSelectedFile();
  }

  /**
   * Returns the classify tab, if available.
   * 
   * @return		the tab, null if not available
   */
  public ClassifierPanel getClassifyTab() {
    ClassifierPanel	result;
    
    result = null;
    
    for (ExplorerPanel panel: getPanels()) {
      if (panel instanceof ClassifierPanel) {
	result = (ClassifierPanel) panel;
	break;
      }
    }
    
    return result;
  }
  
  /**
   * Returns whether the classify tab is present.
   * 
   * @return		true if available
   */
  public boolean hasClassifyTab() {
    return (getClassifyTab() != null);
  }
  
  /**
   * Loads a classifier in the classify tab.
   */
  public void loadClassifier() {
    if (!hasClassifyTab() || !isDataLoaded())
      return;
    getClassifyTab().loadClassifier();
  }

  /**
   * Returns the cluster tab, if available.
   * 
   * @return		the tab, null if not available
   */
  public ClassifierPanel getClusterTab() {
    ClassifierPanel	result;
    
    result = null;
    
    for (ExplorerPanel panel: getPanels()) {
      if (panel instanceof ClassifierPanel) {
	result = (ClassifierPanel) panel;
	break;
      }
    }
    
    return result;
  }
  
  /**
   * Returns whether the cluster tab is present.
   * 
   * @return		true if available
   */
  public boolean hasClusterTab() {
    return (getClusterTab() != null);
  }

  /**
   * Loads a clusterer in the cluster tab.
   */
  public void loadClusterer() {
    if (!hasClusterTab() || !isDataLoaded())
      return;
    getClusterTab().loadClassifier();
  }

  /**
   * Closes the dialog.
   */
  public void close() {
    if (GUIHelper.getParentDialog(this) != null)
      GUIHelper.getParentDialog(this).setVisible(false);
    else if (GUIHelper.getParentFrame(this) != null)
      GUIHelper.getParentFrame(this).setVisible(false);
  }
  
  /**
   * Checks whether data is currently loaded.
   * 
   * @return		true if data loaded
   */
  public boolean isDataLoaded() {
    return (getPreprocessPanel().getInstances() != null);
  }
  
  /**
   * Checks whether undo is possible.
   * 
   * @return		true if undo is possible
   */
  public boolean canUndo() {
    if (m_ButtonUndo == null)
      return false;
    return m_ButtonUndo.isEnabled();
  }
  
  /**
   * Performs an undo.
   */
  public void undo() {
    if ((m_ButtonUndo != null) && m_ButtonUndo.isEnabled())
      m_ButtonUndo.doClick();
  }
  
  /**
   * Checks whether editing the data is possible.
   * 
   * @return		true if edit is possible
   */
  public boolean canEdit() {
    if (m_ButtonEdit == null)
      return false;
    return m_ButtonEdit.isEnabled();
  }
  
  /**
   * Performs an undo.
   */
  public void edit() {
    if ((m_ButtonEdit != null) && m_ButtonEdit.isEnabled())
      m_ButtonEdit.doClick();
  }

  /**
   * Opens up a new window, optionally with the same data.
   * 
   * @param duplicate	whether to load the same in new window
   */
  protected ExplorerExt newWindow(boolean duplicate) {
    ExplorerExt 	result;
    ChildFrame 		oldFrame;
    ChildFrame 		newFrame;
    ChildWindow 	oldWindow;
    ChildWindow 	newWindow;

    result    = null;
    oldFrame = (ChildFrame) GUIHelper.getParent(this, ChildFrame.class);
    if (oldFrame != null) {
      newFrame = oldFrame.getNewWindow();
      newFrame.setVisible(true);
      result  = (ExplorerExt) newFrame.getContentPane().getComponent(0);
    }
    else {
      oldWindow = (ChildWindow) GUIHelper.getParent(this, ChildWindow.class);
      if (oldWindow != null) {
	newWindow = oldWindow.getNewWindow();
	newWindow.setVisible(true);
	result  = (ExplorerExt) newWindow.getContentPane().getComponent(0);
      }
    }

    // use same data
    if (result != null) {
      if (duplicate) {
	result.m_CurrentFile = m_CurrentFile;
	if (getPreprocessPanel().getInstances() != null) {
	  result.m_FileChooser.setCurrentDirectory(m_FileChooser.getCurrentDirectory());
	  result.getPreprocessPanel().setInstances(getPreprocessPanel().getInstances());
	}
      }
    }

    return result;
  }

  /**
   * Sets the base title to use for the title generator.
   * 
   * @param value	the title to use
   * @see		#m_TitleGenerator
   */
  public void setTitle(String value) {
    m_TitleGenerator.setTitle(value);
    update();
  }
  
  /**
   * Returns the base title in use by the title generator.
   * 
   * @return		the title in use
   * @see		#m_TitleGenerator
   */
  public String getTitle() {
    return m_TitleGenerator.getTitle();
  }

  /**
   * Sets the new title for the parent.
   *
   * @param value	the title to use
   */
  protected void setParentTitle(String value) {
    if (GUIHelper.getParentDialog(this) != null)
      GUIHelper.getParentDialog(this).setTitle(value);
    else if (GUIHelper.getParentFrame(this) != null)
      GUIHelper.getParentFrame(this).setTitle(value);
  }
}
