/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    IterativeHMMKernelizer.java
 *    Copyright (C) 2010 Stefan Mutter
 *
 */

package weka.classifiers.meta;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Enumeration;
import java.util.Vector;

import weka.classifiers.RandomizableSingleClassifierEnhancer;
import weka.classifiers.sequence.IterativeProfileHMMClassifierSingleHMM;
import weka.classifiers.sequence.SequenceLearner;
import weka.classifiers.sequence.core.IllegalSymbolException;
import weka.classifiers.sequence.core.ImpossibleStateProbabilityException;
import weka.classifiers.sequence.core.InvalidStructureException;
import weka.classifiers.sequence.core.InvalidViterbiPathException;
import weka.classifiers.sequence.core.NumericStabilityException;
import weka.core.Capabilities;
import weka.core.Capabilities.Capability;
import weka.core.Instance;
import weka.core.Instances;
import weka.core.Option;
import weka.core.OptionHandler;
import weka.core.Utils;
import weka.core.converters.ArffLoader;
import weka.filters.Filter;
import weka.filters.unsupervised.instance.RemoveWithValues;
import weka.utils.OptFileFilter;


/**
<!-- globalinfo-start -->
* Class that takes a one-class or binary PHMM and uses its sufficient statistics to propositionalise sequences. The resulting feature vector can be used with any propositional classifier.
*
<!-- globalinfo-end -->
*
<!-- options-start -->
* Valid options are: <p/>
*
* <pre> -H classpath;
*  the one-class or binary PHMM.</pre>
*
* <pre> -R
*  if set, the propositional arff files will be saved. Be sure to set the path right in the class variable before executing.</pre>
*
* <pre> -N
*  no propositionalisation, just basic PHMM classification</pre>
*
<!-- options-end -->
*
*
* @author Stefan Mutter (pHMM4weka@gmail.com)
* @version $Revision: 10 $
*/
public class IterativeHMMKernelizer
  extends RandomizableSingleClassifierEnhancer
  implements IterativeHMMClassifier<IterativeProfileHMMClassifierSingleHMM> {

	/**
	 * for serialization
	 */
	private static final long serialVersionUID = -5885510190237928451L;

	private static final String Model_DIR = "/set/appropriate/path";

	/** The HMM */
	private IterativeProfileHMMClassifierSingleHMM m_sequenceLearner = new weka.classifiers.sequence.IterativeProfileHMMClassifierSingleHMM();

	/** The instance structure of the aligned instances */
	protected Instances m_AlignedInstances;

	protected boolean m_reUseModels = false;

	protected int iterationStepSize;

	protected int iterationCount;

	protected boolean noKernelClassification;

	private int actualFold;

	private int numberOfFolds;

	private int modelNumber;

	protected boolean testSetAlreadyPropositionalised;

	private boolean firstTime;

	public int getActualFold() {
		return actualFold;
	}

	public int getNumberOfFolds() {
		return numberOfFolds;
	}

	public void setActualFold(int actualFold) {
		this.actualFold = actualFold;
	}

	public void setNumberOfFolds(int numberOfFolds) {
		this.numberOfFolds = numberOfFolds;
	}


	public String globalInfo() {
		return   "";
	}

	/**
	 * String describing default classifier.
	 *
	 * @return the default classifier classname
	 */
	protected String defaultClassifierString() {

		return "weka.classifiers.functions.SMO";
	}

	protected String defaultSequenceLearnerString() {

		return "weka.classifiers.sequence.ProfileHMMClassifierNumbers";
	}

	/**
	 * Default constructor.
	 */
	public IterativeHMMKernelizer() {

		m_Classifier = new weka.classifiers.functions.SMO();
		m_sequenceLearner = new weka.classifiers.sequence.IterativeProfileHMMClassifierSingleHMM();
		m_reUseModels = false;
		iterationCount = 0;
		noKernelClassification = false;
		testSetAlreadyPropositionalised = false;
		firstTime = false;
	}



	/**
	 * Returns an enumeration describing the available options.
	 *
	 * @return an enumeration of all the available options.
	 */
	public Enumeration listOptions() {

		Vector newVector = new Vector(7);
		newVector.addElement(new Option(
				"\tFull class name of HMM classification algorithm to use, followed\n"
				+ "\tby HMM options.\n"
				+ "\teg: \"weka.classifiers.sequence.ProfileHMMClassifier\"",
				"H", 1, "-H <HMM specification>"));

		newVector.addElement(new Option(
				"\tre-use already built HMM-models\n",
				"R", 0, "-R " ));


		newVector.addElement(new Option(
				"\tno Propositionalisation, just baseline HMM Classification\n",
				"N", 0, "-N " ));


		Enumeration enu = super.listOptions();
		while (enu.hasMoreElements()) {
			newVector.addElement(enu.nextElement());
		}

		return newVector.elements();
	}


	public void setOptions(String[] options) throws Exception {

		// Same for filter
		String hmmString = Utils.getOption('H', options);
		if (hmmString.length() > 0) {
			String [] hmmSpec = Utils.splitOptions(hmmString);
			if (hmmSpec.length == 0) {
				throw new IllegalArgumentException("Invalid HMM specification string");
			}
			String hmmName = hmmSpec[0];
			hmmSpec[0] = "";
			setSequenceLearner((IterativeProfileHMMClassifierSingleHMM)Utils.forName(IterativeProfileHMMClassifierSingleHMM.class, hmmName, hmmSpec));
		} else {
			setSequenceLearner(new weka.classifiers.sequence.IterativeProfileHMMClassifierSingleHMM());
		}

		m_reUseModels = Utils.getFlag('R', options);


		noKernelClassification = Utils.getFlag('N', options);


		super.setOptions(options);
	}

	/**
	 * Gets the current settings of the Classifier.
	 *
	 * @return an array of strings suitable for passing to setOptions
	 */
	public String [] getOptions() {

		String [] superOptions = super.getOptions();
		String [] options = new String [superOptions.length + 17];
		int current = 0;

		options[current++] = "-H";
		options[current++] = "" + getSequenceLearnerSpec();
		if(m_reUseModels)
			options[current++] = "-R";


		if(noKernelClassification)
			options[current++] = "-N";




		System.arraycopy(superOptions, 0, options, current,
				superOptions.length);
		current += superOptions.length;
		while (current < options.length) {
			options[current++] = "";
		}

		return options;
	}




	public void setSequenceLearner(IterativeProfileHMMClassifierSingleHMM hmm) {

		m_sequenceLearner = hmm;
	}


	public IterativeProfileHMMClassifierSingleHMM getSequenceLearner() {

		return m_sequenceLearner;
	}


	protected String getSequenceLearnerSpec() {

		SequenceLearner c = getSequenceLearner();
		if (c instanceof OptionHandler) {
			return c.getClass().getName() + " "
			+ Utils.joinOptions(((OptionHandler)c).getOptions());
		}
		return c.getClass().getName();
	}

	/**
	 * Returns default capabilities of the classifier.
	 *
	 * @return      the capabilities of this classifier
	 */
	public Capabilities getCapabilities() {
		Capabilities	result;

		if (getSequenceLearner() == null)
			result = super.getCapabilities();
		else
			result = getSequenceLearner().getCapabilities();


		// set dependencies
		for (Capability cap: Capability.values())
			result.enableDependency(cap);

		return result;
	}

	/**
	 * Build the classifier on the filtered data.
	 *
	 * @param data the training data
	 * @throws Exception if the classifier could not be built successfully
	 */
	public void buildClassifier(Instances data) throws Exception {

		//String modeString;

		if (m_Classifier == null) {
			throw new Exception("No base classifiers have been set!");
		}
		m_AlignedInstances = null;
		// remove instances with missing class

		testSetAlreadyPropositionalised = false;

		data = new Instances(data);
		data.deleteWithMissingClass();

		getSequenceLearner().getCapabilities().testWithFail(data);

		if(!m_reUseModels){
			System.out.println("buildHMM");
			doIterations(data);
		}
		else{
			modelNumber = reUseModels(data);
			System.out.println("arff and object saved");
			if(!noKernelClassification){

				reUseKernelArffs(data, true);

				System.out.println("prop arff saved");
			}
		}
		if(!noKernelClassification){

			System.out.println("use second classifier");
			getClassifier().getCapabilities().testWithFail(m_AlignedInstances);
			m_Classifier.buildClassifier(m_AlignedInstances);

		}

	}

	private void doIterations(Instances data) throws Exception, IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException, ImpossibleStateProbabilityException {
		if(!noKernelClassification){

			if(iterationCount == 0){
				m_sequenceLearner.initClassifier(data);

			}
			for(int i = 0; i < iterationStepSize; i++){
				iterationCount++;
				m_sequenceLearner.next(iterationCount);
			}
			System.out.println("do prop");
			m_AlignedInstances = m_sequenceLearner.extractSufficientStatistics(data);
		}
		else{
			if(iterationCount == 0){
				m_sequenceLearner.initClassifier(data);

			}
			for(int i = 0; i < iterationStepSize; i++){
				iterationCount++;
				m_sequenceLearner.next(iterationCount);
			}
		}
	}

	public Instances getPropositionalTestSet(Instances testData) throws IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException, ImpossibleStateProbabilityException, IOException, NumericStabilityException{

		if(!noKernelClassification){
			reUseKernelArffs(testData, false);
		}

		testSetAlreadyPropositionalised = true;
		if(noKernelClassification){
			return testData;
		}
		else{
			return m_AlignedInstances;
		}
	}

	private void reUseKernelArffs(Instances data, boolean train) throws IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException, ImpossibleStateProbabilityException, IOException, NumericStabilityException {
		File dir;
		String dirName = "", length, sample;
		int modelLength = m_sequenceLearner.getRestrictMatchColumns();
		int sampleMethod = m_sequenceLearner.getSampleMethod();

		if(modelLength == -1){
			length = "f";
		}
		else{
			length = modelLength+"";
		}

		if(sampleMethod == -1){
			sample=null;
		}
		else{
			if(sampleMethod == 1){
				sample = "sampleUNI";
			}
			else{
				sample = "sample"+sampleMethod;
			}
		}

		if(! (m_sequenceLearner instanceof IterativeProfileHMMClassifierSingleHMM)){
			if(sampleMethod == -1){
				dir = new File(Model_DIR+data.relationName()+"/bc"+"/l"+length+"/kernelArffs/");
				dirName = Model_DIR+data.relationName()+"/bc"+"/l"+length+"/kernelArffs/";
			}
			else{
				dir = new File(Model_DIR+data.relationName()+"/bc"+"/l"+length+"/"+sample+"/kernelArffs/");
				dirName = Model_DIR+data.relationName()+"/bc"+"/l"+length+"/"+sample+"/kernelArffs/";
			}
		}
		else{

			int classIndex = ((IterativeProfileHMMClassifierSingleHMM)m_sequenceLearner).getClassIndexToKeep();
			String classNumber;
			if(classIndex == -1){
				classNumber = "all";
			}
			else{
				classNumber = classIndex+"";
			}
			dir = new File(Model_DIR+data.relationName()+"/oneClass_"+classNumber+"/l"+length+"/kernelArffs/");
			dirName = Model_DIR+data.relationName()+"/oneClass_"+classNumber+"/l"+length+"/kernelArffs/";
		}

		if(!dir.exists()){
			dir.mkdirs();
			if(train){
				writePropInstances(data, dirName);
			}
			else{
				writePropInstances(data, dirName+"test_");
			}
		}
		else{
			if(!train){
				dirName = dirName+"test_";
				//System.out.println("here "+dirName);
			}
			File arff = new File(dirName+"model_prop_"+modelNumber+".arff");
			if(!arff.exists()){
				arff = new File(dirName+"model_prop_"+modelNumber+".arff.gz");
			}
			if(arff.exists()){
				System.out.println("re-use prop");

				ArffLoader loader = new ArffLoader();
				loader.setFile(arff);
				m_AlignedInstances = loader.getDataSet();

				m_AlignedInstances.setClassIndex(m_AlignedInstances.numAttributes()-1);

			}
			else{
				writePropInstances(data, dirName);
			}
		}

	}

	private void writePropInstances(Instances data, String dirName) throws IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException, ImpossibleStateProbabilityException, IOException, NumericStabilityException {
		System.out.println("do prop");
		//applyPropSettings();
		m_AlignedInstances = m_sequenceLearner.extractSufficientStatistics(data);
		File arff = new File(dirName+"model_prop_"+modelNumber+".arff.gz");
		if(arff.exists()){
			arff.delete();
		}
		arff = new File(dirName+"model_prop_"+modelNumber+".arff");
		if(!arff.exists())
			arff.createNewFile();
		BufferedWriter outSpec = new BufferedWriter(new FileWriter(dirName+"model_prop_"+modelNumber+".arff"));
		outSpec.write(m_AlignedInstances.toString());
		outSpec.flush();
		outSpec.close();
	}



	/**
	 * Classifies a given instance after propositionalisation.
	 *
	 * @param instance the instance to be classified
	 * @return the class distribution for the given instance
	 * @throws Exception if instance could not be classified
	 * successfully
	 */
	public double [] distributionForInstance(Instance instance)
	throws Exception {
		if(!noKernelClassification){

			if(!testSetAlreadyPropositionalised){
				return m_Classifier.distributionForInstance(m_sequenceLearner.extractSufficientStatisticsTestInstance(instance));
			}
			else{
				//System.out.println("here");
				return m_Classifier.distributionForInstance(instance);
			}

		}
		else{
			return m_sequenceLearner.distributionForInstance(instance);
		}
	}

	private int reUseModels(Instances actualInstances) throws Exception{
		String relationName = actualInstances.relationName();
		String dirName, length, sample;

		int modelLength = m_sequenceLearner.getRestrictMatchColumns();
		int sampleMethod = m_sequenceLearner.getSampleMethod();

		if(modelLength == -1){
			length = "f";
		}
		else{
			length = modelLength+"";
		}

		if(sampleMethod == -1){
			sample=null;
		}
		else{
			if(sampleMethod == 1){
				sample = "sampleUNI";
			}
			else{
				sample = "sample"+sampleMethod;
			}
		}
		if(! (m_sequenceLearner instanceof IterativeProfileHMMClassifierSingleHMM)){
			//dirName = Model_DIR+relationName+"/";
			if(sampleMethod == -1){
				dirName = Model_DIR+relationName+"/bc"+"/l"+length+"/";
			}
			else{
				dirName = Model_DIR+relationName+"/bc"+"/l"+length+"/"+sample+"/";
			}
		}
		else{
			int classIndex = ((IterativeProfileHMMClassifierSingleHMM)m_sequenceLearner).getClassIndexToKeep();
			String classNumber;
			if(modelLength == -1){
				length = "f";
			}
			else{
				length = modelLength+"";
			}
			if(classIndex == -1){
				classNumber = "all";
			}
			else{
				classNumber = classIndex+"";
			}
			dirName = Model_DIR+relationName+"/oneClass_"+classNumber+"/l"+length+"/";
			int index = ((IterativeProfileHMMClassifierSingleHMM)m_sequenceLearner).getClassIndexToKeep();
			if(index >= 0){
				RemoveWithValues filter  = new RemoveWithValues();
				String[] options = new String [5];
				options[0] = "-L";
				options[1] = (index+1)+"";
				options[2] = "-C";
				options[3] = (actualInstances.classIndex()+1)+"";
				options[4] = "-V";
				filter.setInputFormat(actualInstances);
				filter.setOptions(options);
				actualInstances = new Instances(Filter.useFilter(actualInstances, filter));
				actualInstances.setRelationName(relationName);
			}

		}


		File searchPath = new File(dirName);
		//first time -> learn and save
		if(!searchPath.exists()){
			System.out.println("buildHMM");
			this.doIterations(actualInstances);
			writeModels(actualInstances, 1);
			System.out.println("model written");
			return 1;
		}

		//search existing models and load if possible
		String hmmOptions = "";
		for (int i = 0; i < getOptions().length; i++) {
			if(getOptions()[i] != null && getOptions()[i].equals("-H")){
				hmmOptions = (getOptions()[++i]);
				break;
			}
		}
		hmmOptions = hmmOptions.replaceAll("-Y -?\\d","");

		File dir;

		if(! (m_sequenceLearner instanceof IterativeProfileHMMClassifierSingleHMM)){
			if(sampleMethod == -1){
				dir = new File(Model_DIR+relationName+"/bc"+"/l"+length+"/iterativeOpts/");
			}
			else{
				dir = new File(Model_DIR+relationName+"/bc"+"/l"+length+"/"+sample+"/iterativeOpts/");
			}
		}
		else{
			modelLength = m_sequenceLearner.getRestrictMatchColumns();
			int classIndex = ((IterativeProfileHMMClassifierSingleHMM)m_sequenceLearner).getClassIndexToKeep();
			String classNumber;
			if(modelLength == -1){
				length = "f";
			}
			else{
				length = modelLength+"";
			}
			if(classIndex == -1){
				classNumber = "all";
			}
			else{
				classNumber = classIndex+"";
			}
			dir = new File(Model_DIR+relationName+"/oneClass_"+classNumber+"/l"+length+"/iterativeOpts/");
		}


		OptFileFilter filter = new OptFileFilter();
		File[] files = dir.listFiles(filter);
		if(files != null){
			for (int i = 0; i < files.length; i++) {
				BufferedReader inNum = new BufferedReader(new FileReader(files[i]));
				String spec = inNum.readLine();
				//System.out.println(files[i]);
				String iterationSpec = inNum.readLine();
				int iterationNumberFile = Integer.parseInt(iterationSpec);
				String actualFoldSpec = inNum.readLine();
				int actualFoldFile = Integer.parseInt(actualFoldSpec);
				String numberOfFoldsSpec = inNum.readLine();
				int numberOfFoldsFile = Integer.parseInt(numberOfFoldsSpec);
				String seedSpec = inNum.readLine();
				int seedFile = Integer.parseInt(seedSpec);
				inNum.close();
				if(spec.equals(hmmOptions)
						&& iterationNumberFile == this.getIterationCount()+1
						&& actualFoldFile == this.getActualFold()
						&& numberOfFoldsFile == this.getNumberOfFolds()
						&& seedFile == this.getSeed()){

					String determineNumber = files[i].getName();
					determineNumber = determineNumber.substring(determineNumber.lastIndexOf("_")+1, determineNumber.lastIndexOf("."));

					File arff = new File(dirName+"model_"+determineNumber+".arff.gz");
					if(!arff.exists()){
						arff = new File(dirName+"model_"+determineNumber+".arff");
					}
					ArffLoader loader = new ArffLoader();
					loader.setFile(arff);
					Instances loadedInstances = loader.getDataSet();

					//inNum = new BufferedReader(new FileReader(dirName+"model_"+determineNumber+".arff"));
					//Instances loadedInstances = new Instances(inNum);
					//inNum.close();

					loadedInstances.setClassIndex(actualInstances.classIndex());
					if(loadedInstances.equalHeaders(actualInstances)){
						if(loadedInstances.numInstances() == actualInstances.numInstances()){
							int j = 0;
							int k =0;
							boolean equals = true;
							for(j = 0; j< loadedInstances.numInstances();j++){
								Instance loaded = loadedInstances.instance(j);
								Instance actual = actualInstances.instance(j);
								for(k = 0; k< loaded.numAttributes();k++){
									if(loaded.attribute(k).isNumeric()){
										if(loaded.value(k)!=actual.value(k)){
											equals = false;
											break;
										}
									}
									else{
										if(!loaded.stringValue(k).equals(actual.stringValue(k))){
											equals = false;
											break;
										}
									}
								}
								if(!equals)
									break;
							}
							if(equals){
								//load model
								System.out.println("re-use model");
								FileInputStream fis = new FileInputStream(dirName+"model_"+determineNumber+".obj");
								ObjectInputStream ois = new ObjectInputStream(fis);

								m_sequenceLearner = (IterativeProfileHMMClassifierSingleHMM)ois.readObject();

								ois.close();
								fis.close();
								System.gc();

								iterationCount = iterationNumberFile;
								return Integer.parseInt(determineNumber);
							}
						}
					}
				}
			}
		}

		//learn and save because no identical model could be found
		System.out.println("buildHMM");
		this.doIterations(actualInstances);
		BufferedReader inNum = new BufferedReader(new FileReader(dirName+"models"+".num"));
		int modelNumber = Integer.parseInt(inNum.readLine());
		modelNumber++;
		inNum.close();
		writeModels(actualInstances, modelNumber);
		return modelNumber;


	}


	private void writeModels(Instances actualInstances, int modelNumber)throws Exception{
		String relationName = actualInstances.relationName();
		String hmmOptions="";
		File dir;

		String dirName, length, sample;

		int modelLength = m_sequenceLearner.getRestrictMatchColumns();
		int sampleMethod = m_sequenceLearner.getSampleMethod();

		if(modelLength == -1){
			length = "f";
		}
		else{
			length = modelLength+"";
		}

		if(sampleMethod == -1){
			sample=null;
		}
		else{
			if(sampleMethod == 1){
				sample = "sampleUNI";
			}
			else{
				sample = "sample"+sampleMethod;
			}
		}

		if(! (m_sequenceLearner instanceof IterativeProfileHMMClassifierSingleHMM)){
			//dir = new File(Model_DIR+relationName+"/iterativeOpts/");
			if(sampleMethod == -1){
				dir = new File(Model_DIR+relationName+"/bc"+"/l"+length+"/iterativeOpts/");
			}
			else{
				dir = new File(Model_DIR+relationName+"/bc"+"/l"+length+"/"+sample+"/iterativeOpts/");
			}
		}
		else{
			modelLength = m_sequenceLearner.getRestrictMatchColumns();
			int classIndex = ((IterativeProfileHMMClassifierSingleHMM)m_sequenceLearner).getClassIndexToKeep();
			String classNumber;
			if(modelLength == -1){
				length = "f";
			}
			else{
				length = modelLength+"";
			}
			if(classIndex == -1){
				classNumber = "all";
			}
			else{
				classNumber = classIndex+"";
			}
			dir = new File(Model_DIR+relationName+"/oneClass_"+classNumber+"/l"+length+"/iterativeOpts/");
		}

		if(!dir.exists()){
			dir.mkdirs();
		}
		for (int i = 0; i < getOptions().length; i++) {
			if(getOptions()[i] != null && getOptions()[i].equals("-H")){
				hmmOptions = (getOptions()[++i]);
				break;
			}
		}
		hmmOptions = hmmOptions.replaceAll("-Y -?\\d","");

		//hmmOptions = hmmOptions.replaceAll("-J UNI\\s","");

		if(! (m_sequenceLearner instanceof IterativeProfileHMMClassifierSingleHMM)){
			//dirName = Model_DIR+relationName+"/";
			if(sampleMethod == -1){
				dirName = Model_DIR+relationName+"/bc"+"/l"+length+"/";
			}
			else{
				dirName = Model_DIR+relationName+"/bc"+"/l"+length+"/"+sample+"/";
			}
		}
		else{
			modelLength = m_sequenceLearner.getRestrictMatchColumns();
			int classIndex = ((IterativeProfileHMMClassifierSingleHMM)m_sequenceLearner).getClassIndexToKeep();
			String classNumber;
			if(modelLength == -1){
				length = "f";
			}
			else{
				length = modelLength+"";
			}
			if(classIndex == -1){
				classNumber = "all";
			}
			else{
				classNumber = classIndex+"";
			}
			dirName = Model_DIR+relationName+"/oneClass_"+classNumber+"/l"+length+"/";
		}

		File opt = new File(dirName+"iterativeOpts/model_"+modelNumber+".opt");
		if(!opt.exists())
			opt.createNewFile();
		BufferedWriter outSpec = new BufferedWriter(new FileWriter(dirName+"iterativeOpts/model_"+modelNumber+".opt"));
		outSpec.write(hmmOptions+"\n");
		outSpec.write(this.getIterationCount()+"\n");
		outSpec.write(this.getActualFold()+"\n");
		outSpec.write(this.getNumberOfFolds()+"\n");
		outSpec.write(this.getSeed()+"");
		outSpec.flush();
		outSpec.close();
		File num = new File(dirName+"models.num");
		if(!num.exists())
			num.createNewFile();
		outSpec = new BufferedWriter(new FileWriter(dirName+"models"+".num"));
		outSpec.write(modelNumber+"");
		outSpec.flush();
		outSpec.close();
		File arff = new File(dirName+"model_"+modelNumber+".arff.gz");
		if(arff.exists()){
			arff.delete();
		}
		arff = new File(dirName+"model_"+modelNumber+".arff");
		if(!arff.exists())
			arff.createNewFile();
		outSpec = new BufferedWriter(new FileWriter(dirName+"model_"+modelNumber+".arff"));
		if(! (m_sequenceLearner instanceof IterativeProfileHMMClassifierSingleHMM)){
			outSpec.write(actualInstances.toString());
		}
		else{
			outSpec.write((((IterativeProfileHMMClassifierSingleHMM)m_sequenceLearner).getOneClassClassificationArff()).toString());
		}
		outSpec.flush();
		outSpec.close();


		FileOutputStream fos = new FileOutputStream(dirName+"model_"+modelNumber+".obj");
		ObjectOutputStream oos = new ObjectOutputStream(fos);
		oos.writeObject(m_sequenceLearner);
		oos.flush();
		oos.close();


	}

	public boolean getReUse(){
		return m_reUseModels;
	}

	public void setReUse(boolean flag){
		m_reUseModels = flag;
	}
	public String reUseTipText() {
		return "Re-uses already built HMM models";
	}

	/**
	 * Output a representation of this classifier
	 *
	 * @return a representation of this classifier
	 */
	public String toString() {

		if (m_AlignedInstances == null) {
			return "No Propositionalisation\n"
			+"Classification using the PHMM(s):\n"
			+m_sequenceLearner.toString();
		}

		String result = "HMMPropositionalizer using "
			+ getClassifierSpec()
			+ " on data propositionalized through "
			+ getSequenceLearnerSpec()
			+ "\n\nPropositionalized Instances Summary:\n"
			+ m_AlignedInstances.toSummaryString()
			+ "\n\nClassifier Model\n"
			+ m_Classifier.toString();
		if(this.getDebug()){
			return result;
		}
		else{
			String addInfo = "";
			if (m_Classifier instanceof CVParameterSelectionAUC){
				addInfo = ((CVParameterSelectionAUC)m_Classifier).toSummaryString();
			}
			if (m_Classifier instanceof CVParameterSelectionAUCMulti){
				addInfo = ((CVParameterSelectionAUCMulti)m_Classifier).toSummaryString();
			}
			return "Propositionalized Instances Short Summary:\n"
			+"number of attributes:\t"
			+m_AlignedInstances.numAttributes()
			+"\nnumber of instances:\t"
			+m_AlignedInstances.numInstances()+"\n"
			+m_sequenceLearner.toString()+"\n"
			+addInfo;//+"\n"
			//+m_Classifier.toString();
		}
	}



	public String sequenceLearnerTipText(){
		return "The HMM model";
	}





	/**
	 * Main method for testing this class.
	 *
	 * @param argv should contain the following arguments:
	 * -t training file [-T test file] [-c class index]
	 */
	public static void main(String [] argv) {
		runClassifier(new IterativeHMMKernelizer(), argv);
	}

	public String getRevision() {
		// TODO Auto-generated method stub
		return null;
	}

	public int getIterationStepSize() {
		return iterationStepSize;
	}

	public void setIterationStepSize(int iterationStepSize) {
		this.iterationStepSize = iterationStepSize;
	}

	public boolean hmmsConverged(){
		return m_sequenceLearner.fullyConverged();
	}

	public int getIterationCount() {
		return iterationCount;
	}

	public boolean isNoPropositionalisation() {
		return noKernelClassification;
	}

	public void setNoPropositionalisation(boolean noPropositionalisation) {
		this.noKernelClassification = noPropositionalisation;
	}


	public boolean isFirstTime() {
		return firstTime;
	}

	public void setFirstTime(boolean firstTime) {
		this.firstTime = firstTime;
	}



}
