/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    IterativeHMMPropositionalizer.java
 *    Copyright (C) 2010 Stefan Mutter
 *
 */

package weka.classifiers.meta;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Enumeration;
import java.util.Vector;

import weka.classifiers.RandomizableSingleClassifierEnhancer;
import weka.classifiers.sequence.IterativeProfileHMMClassifier;
import weka.classifiers.sequence.IterativeProfileHMMClassifierSingleHMM;
import weka.classifiers.sequence.IterativeSequenceLearner;
import weka.classifiers.sequence.SequenceLearner;
import weka.classifiers.sequence.core.IllegalSymbolException;
import weka.classifiers.sequence.core.ImpossibleStateProbabilityException;
import weka.classifiers.sequence.core.InvalidStructureException;
import weka.classifiers.sequence.core.InvalidViterbiPathException;
import weka.core.Capabilities;
import weka.core.Capabilities.Capability;
import weka.core.Instance;
import weka.core.Instances;
import weka.core.Option;
import weka.core.OptionHandler;
import weka.core.Range;
import weka.core.Utils;
import weka.core.converters.ArffLoader;
import weka.filters.Filter;
import weka.filters.unsupervised.instance.RemoveWithValues;
import weka.utils.OptFileFilter;

/**
<!-- globalinfo-start -->
 * Class that takes a one-class or binary PHMM and uses it to propositionalise sequences. The resulting feature vector can be used with any propositional classifier.
 *
<!-- globalinfo-end -->
 *
<!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre> -H classpath;
 *  the one-class or binary PHMM.</pre>
 *
 * <pre> -R
 *  if set, the propositional arff files will be saved. Be sure to set the path right in the class variable before executing.</pre>
 *
 * <pre> -N
 *  no propositionalisation, just basic PHMM classification.</pre>
 *
 * <pre> -M num
 *  sets the mode opf propositionalisation. Possible modes 1,2,3,4.
 *  1 -> representation of the Viterbi path
 *  2 -> logarithm of the probability of the best path
 *  3 -> logarithm of the probability of the sequence given the model
 *  4 -> logarithms of the probability for all states given a sequence</pre>
 *
 * <pre> -U
 *  create propostionalisations for all specified modes separately.
 *  For example -R -U -M 2-3 saves propositional arff files for -M 2 and -M 3 separately (instead of 23).</pre>
<!-- options-end -->
 *
 *
 * @author Stefan Mutter (pHMM4weka@gmail.com)
 * @version $Revision: 10 $
 */
public class IterativeHMMPropositionalizer
  extends RandomizableSingleClassifierEnhancer
  implements IterativeHMMClassifier<IterativeSequenceLearner> {

	/**
	 * for serialization
	 */
	private static final long serialVersionUID = -5885510190037928451L;

	private static final String Model_DIR = "/research/repository/mlgroup/smm52/HMMProps_models/newModels/"; //"/research/smm52/HMMProps_models/";

	/** The HMM */
	private IterativeSequenceLearner m_sequenceLearner = new weka.classifiers.sequence.IterativeProfileHMMClassifier();

	/** The instance structure of the aligned instances */
	protected Instances m_AlignedInstances;

	protected Range modeRange;

	protected int [] selectedMode;

	protected boolean m_reUseModels = false;

	protected int iterationStepSize;

	protected int iterationCount;

	protected boolean noPropositionalisation;

	private int actualFold;

	private int numberOfFolds;

	private String modeString;

	private String modeStringCopy;

	private boolean saveAsXML;

	private int modelNumber;

	protected boolean testSetAlreadyPropositionalised;

	protected boolean splitUpPropArffs;

	private boolean firstTime;

	public int getActualFold() {
		return actualFold;
	}

	public int getNumberOfFolds() {
		return numberOfFolds;
	}

	public void setActualFold(int actualFold) {
		this.actualFold = actualFold;
	}

	public void setNumberOfFolds(int numberOfFolds) {
		this.numberOfFolds = numberOfFolds;
	}

	/**
	 * Returns a string describing this classifier
	 * @return a description of the classifier suitable for
	 * displaying in the explorer/experimenter gui
	 */
	public String globalInfo() {
		return   "";
	}

	/**
	 * String describing default classifier.
	 *
	 * @return the default classifier classname
	 */
	protected String defaultClassifierString() {

		return "weka.classifiers.trees.M5P";
	}

	protected String defaultSequenceLearnerString() {

		return "weka.classifiers.sequence.ProfileHMMClassifierNumbers";
	}

	/**
	 * Default constructor.
	 */
	public IterativeHMMPropositionalizer() {

		m_Classifier = new weka.classifiers.trees.M5P();
		m_sequenceLearner = new weka.classifiers.sequence.IterativeProfileHMMClassifier();
		m_reUseModels = false;
		modeRange = new Range();
		iterationCount = 0;
		noPropositionalisation = false;
		saveAsXML = false;
		testSetAlreadyPropositionalised = false;
		splitUpPropArffs = false;
		firstTime = false;
	}



	/**
	 * Returns an enumeration describing the available options.
	 *
	 * @return an enumeration of all the available options.
	 */
	public Enumeration listOptions() {

		Vector newVector = new Vector(7);
		newVector.addElement(new Option(
				"\tFull class name of HMM classification algorithm to use, followed\n"
				+ "\tby HMM options.\n"
				+ "\teg: \"weka.classifiers.sequence.ProfileHMMClassifier\"",
				"H", 1, "-H <HMM specification>"));

		newVector.addElement(new Option(
				"\tSplits up propositionalised arff files when saving. One arff file for each mode.\n",
				"U", 0, "-U " ));


		newVector.addElement(new Option(
				"\tre-use already built HMM-models\n",
				"R", 0, "-R " ));


		newVector.addElement(new Option(
				"\tmodeOfProp\n",
				"M", 1, "-M " ));

		newVector.addElement(new Option(
				"\tno Propositionalisation, just baseline HMM Classification\n",
				"N", 0, "-N " ));


		Enumeration enu = super.listOptions();
		while (enu.hasMoreElements()) {
			newVector.addElement(enu.nextElement());
		}

		return newVector.elements();
	}


	public void setOptions(String[] options) throws Exception {

		// Same for filter
		String hmmString = Utils.getOption('H', options);
		if (hmmString.length() > 0) {
			String [] hmmSpec = Utils.splitOptions(hmmString);
			if (hmmSpec.length == 0) {
				throw new IllegalArgumentException("Invalid HMM specification string");
			}
			String hmmName = hmmSpec[0];
			hmmSpec[0] = "";
			setSequenceLearner((IterativeSequenceLearner)Utils.forName(SequenceLearner.class, hmmName, hmmSpec));
		} else {
			setSequenceLearner(new weka.classifiers.sequence.IterativeProfileHMMClassifier());
		}

		m_reUseModels = Utils.getFlag('R', options);

		splitUpPropArffs = Utils.getFlag('U', options);

		noPropositionalisation = Utils.getFlag('N', options);

		String modeList = Utils.getOption('M', options);
		if (modeList.length() != 0) {
			setAttributeIndices(modeList);
		}
		else{
			throw new IllegalArgumentException("No propositionalisation mode provided");
		}

		super.setOptions(options);
	}

	/**
	 * Gets the current settings of the Classifier.
	 *
	 * @return an array of strings suitable for passing to setOptions
	 */
	public String [] getOptions() {

		String [] superOptions = super.getOptions();
		String [] options = new String [superOptions.length + 17];
		int current = 0;

		options[current++] = "-H";
		options[current++] = "" + getSequenceLearnerSpec();
		if(m_reUseModels)
			options[current++] = "-R";

		if(splitUpPropArffs)
			options[current++] = "-U";

		if(noPropositionalisation)
			options[current++] = "-N";

		if (!getAttributeIndices().equals("")) {
			options[current++] = "-M"; options[current++] = getAttributeIndices();
		}


		System.arraycopy(superOptions, 0, options, current,
				superOptions.length);
		current += superOptions.length;
		while (current < options.length) {
			options[current++] = "";
		}

		return options;
	}




	public void setSequenceLearner(IterativeSequenceLearner hmm) {

		m_sequenceLearner = hmm;
	}


	public IterativeSequenceLearner getSequenceLearner() {

		return m_sequenceLearner;
	}


	protected String getSequenceLearnerSpec() {

		SequenceLearner c = getSequenceLearner();
		if (c instanceof OptionHandler) {
			return c.getClass().getName() + " "
			+ Utils.joinOptions(((OptionHandler)c).getOptions());
		}
		return c.getClass().getName();
	}

	/**
	 * Returns default capabilities of the classifier.
	 *
	 * @return      the capabilities of this classifier
	 */
	public Capabilities getCapabilities() {
		Capabilities	result;

		if (getSequenceLearner() == null)
			result = super.getCapabilities();
		else
			result = getSequenceLearner().getCapabilities();


		// set dependencies
		for (Capability cap: Capability.values())
			result.enableDependency(cap);

		return result;
	}

	/**
	 * Build the classifier on the filtered data.
	 *
	 * @param data the training data
	 * @throws Exception if the classifier could not be built successfully
	 */
	public void buildClassifier(Instances data) throws Exception {

		if (m_Classifier == null) {
			throw new Exception("No base classifiers have been set!");
		}
		m_AlignedInstances = null;
		// remove instances with missing class

		testSetAlreadyPropositionalised = false;

		data = new Instances(data);
		data.deleteWithMissingClass();

		getSequenceLearner().getCapabilities().testWithFail(data);

		if(!m_reUseModels){
			System.out.println("buildHMM");
			doIterations(data);
		}
		else{
			modelNumber = reUseModels(data);
			System.out.println("arff and object saved");
			if(!noPropositionalisation){
				if(!splitUpPropArffs){
					reUsePropArffs(data, true);
				}
				else{
					modeStringCopy = new String(modeString);
					for(int i = 0; i < modeStringCopy.length(); i++){
						modeString = modeStringCopy.charAt(i)+"";
						resetPropSettings();
						setAttributeIndices(modeString);
						reUsePropArffs(data, true);
					}
					modeString = modeStringCopy;
				}
				System.out.println("prop arff saved");
			}
		}
		if(!noPropositionalisation){
			if(!splitUpPropArffs){
				System.out.println("use second classifier");
				getClassifier().getCapabilities().testWithFail(m_AlignedInstances);
				m_Classifier.buildClassifier(m_AlignedInstances);
			}
		}

	}

	private void doIterations(Instances data) throws Exception, IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException, ImpossibleStateProbabilityException {
		if(!noPropositionalisation){
			//m_sequenceLearner.buildClassifier(data);
			if(iterationCount == 0){
				m_sequenceLearner.initClassifier(data);
				applyPropSettings();
			}
			for(int i = 0; i < iterationStepSize; i++){
				iterationCount++;
				m_sequenceLearner.next(iterationCount);
			}
			System.out.println("do prop");
			m_AlignedInstances = m_sequenceLearner.propositionalise(data);
		}
		else{
			if(iterationCount == 0){
				m_sequenceLearner.initClassifier(data);
				applyPropSettings();
			}
			for(int i = 0; i < iterationStepSize; i++){
				iterationCount++;
				m_sequenceLearner.next(iterationCount);
			}
		}
	}

	public Instances getPropositionalTestSet(Instances testData) throws IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException, ImpossibleStateProbabilityException, IOException{
		//System.out.println("test: "+modeString);
		if(splitUpPropArffs){
			//String modeStringCopy = new String(modeString);
			for(int i = 0; i < modeStringCopy.length(); i++){
				modeString = modeStringCopy.charAt(i)+"";
				//modeRange = new Range();
				resetPropSettings();
				setAttributeIndices(modeString);
				reUsePropArffs(testData, false);
				resetPropSettings();
			}
			modeString = modeStringCopy;
		}
		else{
			if(!noPropositionalisation){
				reUsePropArffs(testData, false);
			}
		}
		testSetAlreadyPropositionalised = true;
		if(splitUpPropArffs || noPropositionalisation){
			return testData;
		}
		else{
			return m_AlignedInstances;
		}
	}

	private void reUsePropArffs(Instances data, boolean train) throws IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException, ImpossibleStateProbabilityException, IOException {
		File dir;
		String dirName = "", length, sample;
		int modelLength = m_sequenceLearner.getRestrictMatchColumns();
		int sampleMethod = m_sequenceLearner.getSampleMethod();

		if(modelLength == -1){
			length = "f";
		}
		else{
			length = modelLength+"";
		}

		if(sampleMethod == -1){
			sample=null;
		}
		else{
			if(sampleMethod == 1){
				sample = "sampleUNI";
			}
			else{
				sample = "sample"+sampleMethod;
			}
		}

		if(! (m_sequenceLearner instanceof IterativeProfileHMMClassifierSingleHMM)){
			if(sampleMethod == -1){
				dir = new File(Model_DIR+data.relationName()+"/bc"+"/l"+length+"/propArffs/"+modeString+"/");
				dirName = Model_DIR+data.relationName()+"/bc"+"/l"+length+"/propArffs/"+modeString+"/";
			}
			else{
				dir = new File(Model_DIR+data.relationName()+"/bc"+"/l"+length+"/"+sample+"/propArffs/"+modeString+"/");
				dirName = Model_DIR+data.relationName()+"/bc"+"/l"+length+"/"+sample+"/propArffs/"+modeString+"/";
			}
		}
		else{

			int classIndex = ((IterativeProfileHMMClassifierSingleHMM)m_sequenceLearner).getClassIndexToKeep();
			String classNumber;
			if(classIndex == -1){
				classNumber = "all";
			}
			else{
				classNumber = classIndex+"";
			}
			dir = new File(Model_DIR+data.relationName()+"/oneClass_"+classNumber+"/l"+length+"/propArffs/"+modeString+"/");
			dirName = Model_DIR+data.relationName()+"/oneClass_"+classNumber+"/l"+length+"/propArffs/"+modeString+"/";
		}

		if(!dir.exists()){
			dir.mkdirs();
			if(train){
				writePropInstances(modeString, data, dirName);
			}
			else{
				writePropInstances(modeString, data, dirName+"test_");
			}
		}
		else{
			if(!train){
				dirName = dirName+"test_";
				//System.out.println("here "+dirName);
			}
			File arff = new File(dirName+"model_prop_"+modelNumber+"_"+modeString+".arff");
			if(!arff.exists()){
				arff = new File(dirName+"model_prop_"+modelNumber+"_"+modeString+".arff.gz");
			}
			if(arff.exists()){
				System.out.println("re-use prop");
				ArffLoader loader = new ArffLoader();
				loader.setFile(arff);
				m_AlignedInstances = loader.getDataSet();

				m_AlignedInstances.setClassIndex(m_AlignedInstances.numAttributes()-1);
				applyPropSettings();
			}
			else{
				writePropInstances(modeString, data, dirName);
			}
		}

	}

	private void writePropInstances(String modeString, Instances data, String dirName) throws IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException, ImpossibleStateProbabilityException, IOException {
		System.out.println("do prop");
		applyPropSettings();
		m_AlignedInstances = m_sequenceLearner.propositionalise(data);
		m_AlignedInstances.setRelationName(m_AlignedInstances.relationName()+"-"+modeString);
		File arff = new File(dirName+"model_prop_"+modelNumber+"_"+modeString+".arff.gz");
		if(arff.exists()){
			arff.delete();
		}
		arff = new File(dirName+"model_prop_"+modelNumber+"_"+modeString+".arff");
		if(!arff.exists())
			arff.createNewFile();
		BufferedWriter outSpec = new BufferedWriter(new FileWriter(dirName+"model_prop_"+modelNumber+"_"+modeString+".arff"));
		outSpec.write(m_AlignedInstances.toString());
		outSpec.flush();
		outSpec.close();
	}

	private void applyPropSettings() {
		modeString = "";

		resetPropSettings();
		modeRange.setUpper(6);
		selectedMode = modeRange.getSelection();
		m_sequenceLearner.setNoBasic(true);
		for(int i =0; i< selectedMode.length; i++){
			int currentMode = selectedMode[i];
			modeString += currentMode+1;
			if(currentMode == 0){
				m_sequenceLearner.setNoBasic(false);
			}
			if(currentMode == 1){
				m_sequenceLearner.setViterbiProb(true);
			}
			if(currentMode == 2){
				m_sequenceLearner.setFwdProb(true);
			}
			if(currentMode == 3){
				if(m_sequenceLearner.isAllProb() == true || m_sequenceLearner.isAllProbOnly() == true){
					throw new IllegalArgumentException("Incompatible modes");
				}
				m_sequenceLearner.setAllProb(true);
				m_sequenceLearner.setAllProbOnly(false);
			}
			if(currentMode == 4){
				if(m_sequenceLearner.isAllProb() == true){
					throw new IllegalArgumentException("Incompatible modes");
				}
				m_sequenceLearner.setAllProb(true);
				m_sequenceLearner.setAllProbOnly(true);
			}
		}
		//return modeString;
	}

	private void resetPropSettings() {
		m_sequenceLearner.setAllProb(false);
		m_sequenceLearner.setAllProbOnly(false);
		m_sequenceLearner.setFwdProb(false);
		m_sequenceLearner.setNoBasic(false);
		m_sequenceLearner.setViterbiProb(false);
		m_sequenceLearner.setNoPathLogScores(false);

	}

	/**
	 * Classifies a given instance after propositionalisation.
	 *
	 * @param instance the instance to be classified
	 * @return the class distribution for the given instance
	 * @throws Exception if instance could not be classified
	 * successfully
	 */
	public double [] distributionForInstance(Instance instance)
	throws Exception {
		if(!noPropositionalisation){
			if(!splitUpPropArffs){
				if(!testSetAlreadyPropositionalised){
					return m_Classifier.distributionForInstance(m_sequenceLearner.propositionaliseTestInstance(instance));
				}
				else{
					//System.out.println("here");
					return m_Classifier.distributionForInstance(instance);
				}
			}
			else{
				return m_sequenceLearner.distributionForInstance(instance);
			}
		}
		else{
			return m_sequenceLearner.distributionForInstance(instance);
		}
	}

	private int reUseModels(Instances actualInstances) throws Exception{
		String relationName = actualInstances.relationName();
		String dirName, length, sample;

		int modelLength = m_sequenceLearner.getRestrictMatchColumns();
		int sampleMethod = m_sequenceLearner.getSampleMethod();

		if(modelLength == -1){
			length = "f";
		}
		else{
			length = modelLength+"";
		}

		if(sampleMethod == -1){
			sample=null;
		}
		else{
			if(sampleMethod == 1){
				sample = "sampleUNI";
			}
			else{
				sample = "sample"+sampleMethod;
			}
		}
		if(! (m_sequenceLearner instanceof IterativeProfileHMMClassifierSingleHMM)){
			//dirName = Model_DIR+relationName+"/";
			if(sampleMethod == -1){
				dirName = Model_DIR+relationName+"/bc"+"/l"+length+"/";
			}
			else{
				dirName = Model_DIR+relationName+"/bc"+"/l"+length+"/"+sample+"/";
			}
		}
		else{
			int classIndex = ((IterativeProfileHMMClassifierSingleHMM)m_sequenceLearner).getClassIndexToKeep();
			String classNumber;
			if(modelLength == -1){
				length = "f";
			}
			else{
				length = modelLength+"";
			}
			if(classIndex == -1){
				classNumber = "all";
			}
			else{
				classNumber = classIndex+"";
			}
			dirName = Model_DIR+relationName+"/oneClass_"+classNumber+"/l"+length+"/";
			int index = ((IterativeProfileHMMClassifierSingleHMM)m_sequenceLearner).getClassIndexToKeep();
			if(index >= 0){
				RemoveWithValues filter  = new RemoveWithValues();
				String[] options = new String [5];
				options[0] = "-L";
				options[1] = (index+1)+"";
				options[2] = "-C";
				options[3] = (actualInstances.classIndex()+1)+"";
				options[4] = "-V";
				filter.setInputFormat(actualInstances);
				filter.setOptions(options);
				actualInstances = new Instances(Filter.useFilter(actualInstances, filter));
				actualInstances.setRelationName(relationName);
			}

		}


		File searchPath = new File(dirName);
		//System.out.println(dirName);

		//first time -> learn and save
		if(!searchPath.exists()){
			System.out.println("buildHMM");
			this.doIterations(actualInstances);
			writeModels(actualInstances, 1);
			System.out.println("model written");
			return 1;
		}

		//search existing models and load if possible
		String hmmOptions = "";
		for (int i = 0; i < getOptions().length; i++) {
			if(getOptions()[i] != null && getOptions()[i].equals("-H")){
				hmmOptions = (getOptions()[++i]);
				break;
			}
		}

		File dir;

		if(! (m_sequenceLearner instanceof IterativeProfileHMMClassifierSingleHMM)){
			//dir = new File(Model_DIR+relationName+"/iterativeOpts/");
			if(sampleMethod == -1){
				dir = new File(Model_DIR+relationName+"/bc"+"/l"+length+"/iterativeOpts/");
			}
			else{
				dir = new File(Model_DIR+relationName+"/bc"+"/l"+length+"/"+sample+"/iterativeOpts/");
			}
		}
		else{
			modelLength = m_sequenceLearner.getRestrictMatchColumns();
			int classIndex = ((IterativeProfileHMMClassifierSingleHMM)m_sequenceLearner).getClassIndexToKeep();
			String classNumber;
			if(modelLength == -1){
				length = "f";
			}
			else{
				length = modelLength+"";
			}
			if(classIndex == -1){
				classNumber = "all";
			}
			else{
				classNumber = classIndex+"";
			}
			dir = new File(Model_DIR+relationName+"/oneClass_"+classNumber+"/l"+length+"/iterativeOpts/");
		}


		OptFileFilter filter = new OptFileFilter();
		File[] files = dir.listFiles(filter);
		if(files != null){
			for (int i = 0; i < files.length; i++) {
				BufferedReader inNum = new BufferedReader(new FileReader(files[i]));
				String spec = inNum.readLine();
				//System.out.println(files[i]);
				String iterationSpec = inNum.readLine();
				int iterationNumberFile = Integer.parseInt(iterationSpec);
				String actualFoldSpec = inNum.readLine();
				int actualFoldFile = Integer.parseInt(actualFoldSpec);
				String numberOfFoldsSpec = inNum.readLine();
				int numberOfFoldsFile = Integer.parseInt(numberOfFoldsSpec);
				String seedSpec = inNum.readLine();
				int seedFile = Integer.parseInt(seedSpec);
				inNum.close();
				//System.out.println(hmmOptions+" "+spec);
				//System.out.println("ACHTUNG "+this.getIterationCount()+" "+iterationNumberFile);
				if(spec.equals(hmmOptions)
						&& iterationNumberFile == this.getIterationCount()+1
						&& actualFoldFile == this.getActualFold()
						&& numberOfFoldsFile == this.getNumberOfFolds()
						&& seedFile == this.getSeed()){

					String determineNumber = files[i].getName();
					determineNumber = determineNumber.substring(determineNumber.lastIndexOf("_")+1, determineNumber.lastIndexOf("."));

					File arff = new File(dirName+"model_"+determineNumber+".arff.gz");
					if(!arff.exists()){
						arff = new File(dirName+"model_"+determineNumber+".arff");
					}
					ArffLoader loader = new ArffLoader();
					loader.setFile(arff);
					Instances loadedInstances = loader.getDataSet();

					loadedInstances.setClassIndex(actualInstances.classIndex());
					if(loadedInstances.equalHeaders(actualInstances)){
						if(loadedInstances.numInstances() == actualInstances.numInstances()){
							int j = 0;
							int k =0;
							boolean equals = true;
							for(j = 0; j< loadedInstances.numInstances();j++){
								Instance loaded = loadedInstances.instance(j);
								Instance actual = actualInstances.instance(j);
								for(k = 0; k< loaded.numAttributes();k++){
									if(loaded.attribute(k).isNumeric()){
										if(loaded.value(k)!=actual.value(k)){
											equals = false;
											break;
										}
									}
									else{
										if(!loaded.stringValue(k).equals(actual.stringValue(k))){
											equals = false;
											break;
										}
									}
								}
								if(!equals)
									break;
							}
							if(equals){
								//load model
								System.out.println("re-use model");
								FileInputStream fis = new FileInputStream(dirName+"model_"+determineNumber+".obj");
								ObjectInputStream ois = new ObjectInputStream(fis);

								m_sequenceLearner = (IterativeProfileHMMClassifier)ois.readObject();

								ois.close();
								fis.close();
								System.gc();
								if(iterationCount == 0){
									getPropSettings();
								}
								iterationCount = iterationNumberFile;
								return Integer.parseInt(determineNumber);
							}
						}
					}
				}
			}
		}

		//learn and save because no identical model could be found
		System.out.println("buildHMM");
		this.doIterations(actualInstances);
		BufferedReader inNum = new BufferedReader(new FileReader(dirName+"models"+".num"));
		int modelNumber = Integer.parseInt(inNum.readLine());
		modelNumber++;
		inNum.close();
		writeModels(actualInstances, modelNumber);
		return modelNumber;


	}
	private void getPropSettings() {
		modeString = "";

		modeRange.setUpper(6);
		selectedMode = modeRange.getSelection();
		//m_sequenceLearner.setNoBasic(true);
		for(int i =0; i< selectedMode.length; i++){
			int currentMode = selectedMode[i];
			modeString += currentMode+1;
		}

	}

	private void writeModels(Instances actualInstances, int modelNumber)throws Exception{
		String relationName = actualInstances.relationName();
		String hmmOptions="";
		File dir;

		String dirName, length, sample;

		int modelLength = m_sequenceLearner.getRestrictMatchColumns();
		int sampleMethod = m_sequenceLearner.getSampleMethod();

		if(modelLength == -1){
			length = "f";
		}
		else{
			length = modelLength+"";
		}

		if(sampleMethod == -1){
			sample=null;
		}
		else{
			if(sampleMethod == 1){
				sample = "sampleUNI";
			}
			else{
				sample = "sample"+sampleMethod;
			}
		}

		if(! (m_sequenceLearner instanceof IterativeProfileHMMClassifierSingleHMM)){
			//dir = new File(Model_DIR+relationName+"/iterativeOpts/");
			if(sampleMethod == -1){
				dir = new File(Model_DIR+relationName+"/bc"+"/l"+length+"/iterativeOpts/");
			}
			else{
				dir = new File(Model_DIR+relationName+"/bc"+"/l"+length+"/"+sample+"/iterativeOpts/");
			}
		}
		else{
			modelLength = m_sequenceLearner.getRestrictMatchColumns();
			int classIndex = ((IterativeProfileHMMClassifierSingleHMM)m_sequenceLearner).getClassIndexToKeep();
			String classNumber;
			if(modelLength == -1){
				length = "f";
			}
			else{
				length = modelLength+"";
			}
			if(classIndex == -1){
				classNumber = "all";
			}
			else{
				classNumber = classIndex+"";
			}
			dir = new File(Model_DIR+relationName+"/oneClass_"+classNumber+"/l"+length+"/iterativeOpts/");
		}

		if(!dir.exists()){
			dir.mkdirs();
		}
		for (int i = 0; i < getOptions().length; i++) {
			if(getOptions()[i] != null && getOptions()[i].equals("-H")){
				hmmOptions = (getOptions()[++i]);
				break;
			}
		}
		hmmOptions = hmmOptions.replaceAll("-Y -?\\d","");

		//hmmOptions = hmmOptions.replaceAll("-J UNI\\s","");

		if(! (m_sequenceLearner instanceof IterativeProfileHMMClassifierSingleHMM)){
			//dirName = Model_DIR+relationName+"/";
			if(sampleMethod == -1){
				dirName = Model_DIR+relationName+"/bc"+"/l"+length+"/";
			}
			else{
				dirName = Model_DIR+relationName+"/bc"+"/l"+length+"/"+sample+"/";
			}
		}
		else{
			modelLength = m_sequenceLearner.getRestrictMatchColumns();
			int classIndex = ((IterativeProfileHMMClassifierSingleHMM)m_sequenceLearner).getClassIndexToKeep();
			String classNumber;
			if(modelLength == -1){
				length = "f";
			}
			else{
				length = modelLength+"";
			}
			if(classIndex == -1){
				classNumber = "all";
			}
			else{
				classNumber = classIndex+"";
			}
			dirName = Model_DIR+relationName+"/oneClass_"+classNumber+"/l"+length+"/";
		}

		File opt = new File(dirName+"iterativeOpts/model_"+modelNumber+".opt");
		if(!opt.exists())
			opt.createNewFile();
		BufferedWriter outSpec = new BufferedWriter(new FileWriter(dirName+"iterativeOpts/model_"+modelNumber+".opt"));
		outSpec.write(hmmOptions+"\n");
		outSpec.write(this.getIterationCount()+"\n");
		outSpec.write(this.getActualFold()+"\n");
		outSpec.write(this.getNumberOfFolds()+"\n");
		outSpec.write(this.getSeed()+"");
		outSpec.flush();
		outSpec.close();
		File num = new File(dirName+"models.num");
		if(!num.exists())
			num.createNewFile();
		outSpec = new BufferedWriter(new FileWriter(dirName+"models"+".num"));
		outSpec.write(modelNumber+"");
		outSpec.flush();
		outSpec.close();
		File arff = new File(dirName+"model_"+modelNumber+".arff.gz");
		if(arff.exists()){
			arff.delete();
		}
		arff = new File(dirName+"model_"+modelNumber+".arff");
		if(!arff.exists())
			arff.createNewFile();
		outSpec = new BufferedWriter(new FileWriter(dirName+"model_"+modelNumber+".arff"));
		if(! (m_sequenceLearner instanceof IterativeProfileHMMClassifierSingleHMM)){
			outSpec.write(actualInstances.toString());
		}
		else{
			outSpec.write((((IterativeProfileHMMClassifierSingleHMM)m_sequenceLearner).getOneClassClassificationArff()).toString());
		}
		outSpec.flush();
		outSpec.close();


		FileOutputStream fos = new FileOutputStream(dirName+"model_"+modelNumber+".obj");
		ObjectOutputStream oos = new ObjectOutputStream(fos);
		oos.writeObject(m_sequenceLearner);
		oos.flush();
		oos.close();


	}

	public boolean getReUse(){
		return m_reUseModels;
	}

	public void setReUse(boolean flag){
		m_reUseModels = flag;
	}
	public String reUseTipText() {
		return "Re-uses already built HMM models";
	}

	/**
	 * Output a representation of this classifier
	 *
	 * @return a representation of this classifier
	 */
	public String toString() {

		if (m_AlignedInstances == null) {
			return "No Propositionalisation\n"
			+"Classification using the PHMM(s):\n"
			+m_sequenceLearner.toString();
		}

		String result = "HMMPropositionalizer using "
			+ getClassifierSpec()
			+ " on data propositionalized through "
			+ getSequenceLearnerSpec()
			+ "\n\nPropositionalized Instances Summary:\n"
			+ m_AlignedInstances.toSummaryString()
			+ "\n\nClassifier Model\n"
			+ m_Classifier.toString();
		if(this.getDebug()){
			return result;
		}
		else{
			String addInfo = "";
			if (m_Classifier instanceof CVParameterSelectionAUC){
				addInfo = ((CVParameterSelectionAUC)m_Classifier).toSummaryString();
			}
			if (m_Classifier instanceof CVParameterSelectionAUCMulti){
				addInfo = ((CVParameterSelectionAUCMulti)m_Classifier).toSummaryString();
			}
			return "Propositionalized Instances Short Summary:\n"
			+"number of attributes:\t"
			+m_AlignedInstances.numAttributes()
			+"\nnumber of instances:\t"
			+m_AlignedInstances.numInstances()+"\n"
			+m_sequenceLearner.toString()+"\n"
			+addInfo;//+"\n"
			//+m_Classifier.toString();
		}
	}



	public String sequenceLearnerTipText(){
		return "The HMM model";
	}


	public String getAttributeIndices() {

		return modeRange.getRanges();
	}


	public void setAttributeIndices(String rangeList) {

		modeRange.setRanges(rangeList);
	}


	public void setAttributeIndicesArray(int [] modes) {

		setAttributeIndices(Range.indicesToRangeList(modes));
	}


	/**
	 * Main method for testing this class.
	 *
	 * @param argv should contain the following arguments:
	 * -t training file [-T test file] [-c class index]
	 */
	public static void main(String [] argv) {
		runClassifier(new IterativeHMMPropositionalizer(), argv);
	}

	public String getRevision() {
		return "1.0";
	}

	public int getIterationStepSize() {
		return iterationStepSize;
	}

	public void setIterationStepSize(int iterationStepSize) {
		this.iterationStepSize = iterationStepSize;
	}

	public boolean hmmsConverged(){
		return m_sequenceLearner.fullyConverged();
	}

	public int getIterationCount() {
		return iterationCount;
	}

	public boolean isNoPropositionalisation() {
		return noPropositionalisation;
	}

	public void setNoPropositionalisation(boolean noPropositionalisation) {
		this.noPropositionalisation = noPropositionalisation;
	}

	public boolean isSaveAsXML() {
		return saveAsXML;
	}

	public void setSaveAsXML(boolean saveAsXML) {
		this.saveAsXML = saveAsXML;
	}

	public boolean isSplitUpPropArffs() {
		return splitUpPropArffs;
	}

	public void setSplitUpPropArffs(boolean splitUpPropArffs) {
		this.splitUpPropArffs = splitUpPropArffs;
	}

	public boolean isFirstTime() {
		return firstTime;
	}

	public void setFirstTime(boolean firstTime) {
		this.firstTime = firstTime;
	}



}
