/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    IterativeProfileHMMClassifier.java
 *    Copyright (C) 2010 Stefan Mutter
 *
 */
package weka.classifiers.sequence;

import java.util.Arrays;
import java.util.Enumeration;
import java.util.List;
import java.util.Random;
import java.util.TreeMap;
import java.util.Vector;

import weka.classifiers.Evaluation;
import weka.classifiers.sequence.core.Alphabet;
import weka.classifiers.sequence.core.BaumWelchLearner;
import weka.classifiers.sequence.core.ForwardAlgorithm;
import weka.classifiers.sequence.core.IllegalSymbolException;
import weka.classifiers.sequence.core.ImpossibleStateProbabilityException;
import weka.classifiers.sequence.core.InvalidStructureException;
import weka.classifiers.sequence.core.InvalidViterbiPathException;
import weka.classifiers.sequence.core.NullModel;
import weka.classifiers.sequence.core.NumericStabilityException;
import weka.classifiers.sequence.core.ProbabilityPerStateCalculator;
import weka.classifiers.sequence.core.ProfileHMM;
import weka.classifiers.sequence.core.SequenceGenerator;
import weka.classifiers.sequence.core.ViterbiAlgorithm;
import weka.core.Attribute;
import weka.core.DenseInstance;
import weka.core.FastVector;
import weka.core.Instance;
import weka.core.Instances;
import weka.core.Option;
import weka.core.SelectedTag;
import weka.core.Tag;
import weka.core.Utils;

/**
<!-- globalinfo-start -->
 * binary Profile Hidden Markov Model classifier; a PHMM for the positive and one for the negative class.
 * The dataset needs to be binary and the positive class needs to have index 0 in the arff header.
 * The positive class is to be expected smaller than the negative one, thus sampling the negative one, leads to as many positive as negative instances.
 * <p/>
<!-- globalinfo-end -->
 *
<!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre> -Z
 *  undersamples the negative class (expected to be the bigger one) to be as big as the positive one
 *  tags: "ALL", "no sampling",
 *  "UNI_ONCE","take a radom sample of neagtive instances before the first itartion and keep this sample",
 *  "HIGH","sample highest scoring negative examples in each iteration",
 *  "LOW","sample lowest scoring negative examples in each iteration","
 *  "UNI","sample negative examples uniformly in each iteration",
 *  "UNIPLUS","sample twice as many negative examples (as there are positive examples) uniformly in each iteration, score the negatives with the positive PHMM and take the high scoring",
 *  "STRAT", "sample sorted negative examples based on their scores uniformly in each iteration",
 *  "ART", "create artificial negative examples in first iteration and keep them",
 *  "ARTNEG", "create artificial negative examples from negative HMM"
 *  (default: no sampling, when -Z is not used)</pre>
 *
 * <pre> -P &lt;double&gt;
 *  when artifical sampling is used, percentage of overlapping examples according to forward score</pre>
 *
 *
<!-- options-end -->
 *
 * @author Stefan Mutter (pHMM4weka@gmail.com)
 * @version $Revision: 10 $
 */
public class IterativeProfileHMMClassifier extends IterativeSequenceLearner{

	private static final long serialVersionUID = 3134763907535654750L;

	private List<ProfileHMM> allHMMs;

	private double[] trainingClassDistribution;

	private List<Double> viterbiScore;

	private List<Double> fwdScore;

	private List<List<Double>> allScore;

	private int iteration;

	private List<Double> logLikelihoodOfPHMM;

	private List<Double> initiallogLikelihoodOfPHMM;

	private List<String[]> allTrainingSequences;

	private List<Double> oldLogLikelihoodOfPHMM;

	private List<Integer> numIterationPerClass;

	protected int positiveClassIndex;//positiveClassIndex

	private List<Instances> datasets;

	private int numPositiveInstances;

	private double artificialPercentage;

	private static final int SAMPLE_ALL = -1;

	private static final int SAMPLE_UNIFORM_ONCE = 0;

	private static final int SAMPLE_HIGHEST = 2;

	private static final int SAMPLE_LOWEST = 3;

	private static final int SAMPLE_UNIFORM = 1;

	private static final int SAMPLE_STRATIFIED = 4;

	private static final int SAMPLE_ARTIFICIAL = 5;

	private static final int SAMPLE_UNIFORM_PLUS = 6;

	private static final int SAMPLE_ARTIFICIAL_NEGATIVE = 7;

	public static final Tag [] TAGS_SAMPLE = {
		new Tag(SAMPLE_ALL, "ALL", "no sampling"),
		new Tag(SAMPLE_UNIFORM_ONCE, "UNI_ONCE","take a radom sample of neagtive instances before the first itartion and keep this sample"),
		new Tag(SAMPLE_HIGHEST, "HIGH","sample highest scoring negative examples in each iteration"),
		new Tag(SAMPLE_LOWEST, "LOW","sample lowest scoring negative examples in each iteration"),
		new Tag(SAMPLE_UNIFORM, "UNI","sample negative examples uniformly in each iteration"),
		new Tag(SAMPLE_UNIFORM_PLUS, "UNIPLUS","sample twice as many negative examples (as there are positive examples) uniformly in each iteration, score the negatives with the positive PHMM and take the high scoring"),
		new Tag(SAMPLE_STRATIFIED, "STRAT", "sample sorted negative examples based on their scores uniformly in each iteration"),
		new Tag(SAMPLE_ARTIFICIAL, "ART", "create artificial negative examples in first iteration and keep them"),
		new Tag(SAMPLE_ARTIFICIAL_NEGATIVE, "ARTNEG", "create artificial negative examples from negative HMM")
	};

	//private int sampleMethod = SAMPLE_ALL;

	private boolean loadPositiveModel;

	private IterativeProfileHMMClassifier loadPositiveModelForm;


	public IterativeProfileHMMClassifier() {
		super();
		this.allHMMs = new Vector<ProfileHMM>();
		this.logLikelihoodThreshold = 0.0001; //personal communication Prof A Krogh
		this.useNullModel = false;
		this.transitionsEmissionsNotInLog = false;
		this.learnInsertEmissions = false;
		setBaumWelchOption(2);
		this.viterbiProb = false;
		this.fwdProb = false;
		this.allProb = false;
		this.allProbOnly = false;
		this.noBasic = false;
		this. noPathLogScores = false;
		//this.iteration = new Vector<Integer>();
		this.logLikelihoodOfPHMM = new Vector<Double>();
		this.initiallogLikelihoodOfPHMM = new Vector<Double>();
		this.allTrainingSequences = new Vector<String[]>();
		this.converged = new Vector<Boolean>();
		this.oldLogLikelihoodOfPHMM = new Vector<Double>();
		this.numIterationPerClass = new Vector<Integer>();
		//this.setStopAfterIteration(Integer.MAX_VALUE);
		this.positiveClassIndex = -1;
		this.sampleMethod = SAMPLE_ALL;
		this.loadPositiveModel = false;
		this.loadPositiveModelForm = null;
		this.artificialPercentage = -1;
	}


	public Instances propositionalise(Instances oldInstances) throws IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException, ImpossibleStateProbabilityException {

		Instances transformed = createPropositionalisedInstancesFormat(oldInstances);

		for(int j = 0;j < oldInstances.numInstances();j++){
			viterbiScore = new Vector<Double>();
			fwdScore = new Vector<Double>();
			allScore = new Vector<List<Double>>();
			Instance oldInstance = oldInstances.instance(j);
			String[] alignment = doAlignmentForPropositionalisation(oldInstance);
			transformed = doPropositionalisation(alignment, transformed, oldInstance.classValue());
		}
		//System.out.println(isViterbiProb()+" "+isFwdProb()+" "+isAllProb()+" "+isNoBasic()+" "+isAllProbOnly());
		//System.out.println("finished "+transformed);
		return transformed;
	}


	public Instance propositionaliseTestInstance(Instance oldInstance) throws IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException, ImpossibleStateProbabilityException {

		Instances oldInstances = oldInstance.dataset();
		Instances transformed = createPropositionalisedInstancesFormat(oldInstances);
		viterbiScore = new Vector<Double>();
		fwdScore = new Vector<Double>();
		allScore = new Vector<List<Double>>();
		String[] alignment = doAlignmentForPropositionalisation(oldInstance);

		transformed = doPropositionalisation(alignment, transformed, oldInstance.classValue());
		//System.out.println(transformed.firstInstance());
		return transformed.firstInstance();
	}


	private String[] doAlignmentForPropositionalisation(Instance oldInstance) throws IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException, ImpossibleStateProbabilityException {
		ProfileHMM hmm;
		String[] alignment = new String[allHMMs.size()];
		String test = "";
		test += oldInstance.stringValue(oldInstance.attribute(sequenceIndex));
		if(this.getRestrictSequenceLength() != -1 && test.length()>this.getRestrictSequenceLength())
			test = test.substring(0,this.getRestrictSequenceLength());
		for (int i = 0; i < allHMMs.size(); i++) {
			hmm = allHMMs.get(i);
			if(hmm != null){
				if(isViterbiProb() || !isNoBasic()){
					ViterbiAlgorithm vitAlg= new ViterbiAlgorithm(allHMMs.get(i),test);
					alignment[i] = vitAlg.calculateViterbiPath();
					if(noBasic){
						alignment[i] = "";
					}
					viterbiScore.add(i,vitAlg.getScore());
					vitAlg = null;
				}
				else{
					alignment[i] = "";
				}
				if(isFwdProb()){
					ForwardAlgorithm fwd = new ForwardAlgorithm(allHMMs.get(i),test);
					fwd.calculateForward();
					fwdScore.add(i,fwd.getScore());
					fwd = null;
				}
				if(isAllProb()){
					ProbabilityPerStateCalculator calc = new ProbabilityPerStateCalculator(allHMMs.get(i),test);
					allScore.add(i,calc.getScores());
					calc = null;
				}
			}
			else{
				alignment[i] = "";
				viterbiScore.add(i,null);
				fwdScore.add(i,null);
				allScore.add(i,null);
			}
		}
		if(noPathLogScores){
			double[] path;
			int viterbiSize = viterbiScore.size();
			int fwdSize = fwdScore.size();

			if(viterbiSize > 0){
				path = new double [viterbiSize];
				for(int i = 0; i < viterbiSize; i++){
					path[i] = viterbiScore.get(i);
					//System.out.println("log viterbi for HMM "+i+" : "+viterbiScore.get(i));
				}
				path = Utils.logs2probs(path);
				Utils.normalize(path);
				for(int i = 0; i < viterbiSize; i++){
					viterbiScore.set(i, path[i]);
					//System.out.println("viterbi for HMM "+i+" : "+viterbiScore.get(i));
				}
				path = null;
			}
			if(fwdSize > 0){
				path = new double [fwdSize];
				for(int i = 0; i < fwdSize; i++){
					path[i] = fwdScore.get(i);
					//System.out.println("log fwd for HMM "+i+" : "+fwdScore.get(i));
				}
				path = Utils.logs2probs(path);
				Utils.normalize(path);
				for(int i = 0; i < fwdSize; i++){
					fwdScore.set(i, path[i]);
					//System.out.println("fwd for HMM "+i+" : "+fwdScore.get(i));
				}
				path = null;
				//}
			}
		}
		return alignment;
	}

	protected Instances doPropositionalisation(String[] allAlignment, Instances transformed, double classValue) throws IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException {

		Instance newInst = new DenseInstance(transformed.numAttributes());
		newInst.setDataset(transformed);
		newInst.setClassValue(classValue);

		int attributeCounter = 0;

		for(int k = 0; k < allAlignment.length; k++){
			String alignment = allAlignment[k];
			for(int i = 0; i < alignment.length(); i++){
				if(alignment.charAt(i)== '-'){
					newInst.setValue(attributeCounter, "-");
					attributeCounter++;
					newInst.setValue(attributeCounter,0);
					attributeCounter++;
				}
				else{
					if(Character.isUpperCase(alignment.charAt(i))){
						newInst.setValue(attributeCounter, alignment.charAt(i)+"");
						attributeCounter++;
					}
					if(i < alignment.length()-1 && Character.isLowerCase(alignment.charAt(i+1))){
						i++;
						int count = 0;
						while(i < alignment.length() && Character.isLowerCase(alignment.charAt(i))){
							i++;
							count++;
						}
						i--;
						newInst.setValue(attributeCounter,count);
						attributeCounter++;
					}
					else{
						if(i < alignment.length()-1){
							newInst.setValue(attributeCounter,0);
							attributeCounter++;
						}
					}
				}
			}
			if(isViterbiProb()){
				newInst.setValue(attributeCounter, viterbiScore.get(k));
				attributeCounter++;
			}
			if(isFwdProb()){
				newInst.setValue(attributeCounter, fwdScore.get(k));
				attributeCounter++;
			}
			if(isAllProb()){
				List<Double> scores = allScore.get(k);
				double[] probs = new double[scores.size()];
				if(isAllProbOnly()){
					for(int n = 0; n< scores.size(); n++){
						probs[n] = scores.get(n);
					}
					probs = Utils.logs2probs(probs);
					Utils.normalize(probs);
				}
				for(int j = 0; j < scores.size(); j++){
					if(!isAllProbOnly()){
						newInst.setValue(attributeCounter, scores.get(j));
					}
					else{
						newInst.setValue(attributeCounter, probs[j]);
					}
					attributeCounter++;
				}
			}
		}

		transformed.add(newInst);
		newInst = null;
		return transformed;
	}

	public Instances createPropositionalisedInstancesFormat(Instances oldInstances){

		ProfileHMM hmm;
		Alphabet usedAlphabet = null;
		int numNonClassAttributes = 0;

		for (int i = 0; i < allHMMs.size(); i++) {
			hmm = allHMMs.get(i);
			if(hmm != null){
				if(usedAlphabet == null){
					usedAlphabet = hmm.getAlphabet();
				}
				if(!noBasic){
					numNonClassAttributes += (hmm.getNumberMatchStates()*2)-1;
				}
				if(isViterbiProb())
					numNonClassAttributes++;
				if(isFwdProb())
					numNonClassAttributes++;
				if(isAllProb())
					numNonClassAttributes += (hmm.getNumberMatchStates()*3)-5;
			}
		}
		FastVector attInfo = new FastVector(numNonClassAttributes+1);
		FastVector my_nominal_values = new FastVector();
		if(!noBasic){
			for(int i = 0; i < usedAlphabet.alphabetSize();i++){
				my_nominal_values.addElement(usedAlphabet.getSymbolAtIndex(i));
			}
			my_nominal_values.addElement("-");
		}
		FastVector my_nominal_class_values = new FastVector();
		for (int i = 0; i < oldInstances.numClasses(); i++) {
			my_nominal_class_values.addElement(oldInstances.classAttribute().value(i));
		}

		int counter = 0;
		int index = 0;
		String tempAttributeName = "";
		Attribute tempAttribute;
		for (int i = 0; i < allHMMs.size(); i++) {
			hmm = allHMMs.get(i);
			if(!noBasic){
				int numberOfAttributes = 0;
				if(hmm != null){
					numberOfAttributes = (hmm.getNumberMatchStates()*2)-1;
				}
				for(int j = 0; j < numberOfAttributes; j++){
					if(j%2==0){
						if(j == 0 || j == numberOfAttributes -1){
							tempAttributeName = "Match"+counter+"_HMM"+(index+1);
						}
						else{
							tempAttributeName = "Match_Delete"+counter+"_HMM"+(index+1);
						}
						tempAttribute = new Attribute(tempAttributeName,my_nominal_values);
						attInfo.addElement(tempAttribute);
					}
					else{
						tempAttributeName = "Insert"+counter+"_HMM"+(index+1);
						tempAttribute = new Attribute(tempAttributeName);
						attInfo.addElement(tempAttribute);
						counter++;
					}
				}
			}
			if(isViterbiProb()){
				tempAttribute = new Attribute("ViterbiScore_HMM"+(index+1));
				attInfo.addElement(tempAttribute);
			}
			if(isFwdProb()){
				tempAttribute = new Attribute("ForwardScore_HMM"+(index+1));
				attInfo.addElement(tempAttribute);
			}
			if(isAllProb()){
				for(int j = 1; j < hmm.getNumberMatchStates()-1; j++){
					tempAttribute = new Attribute("Score4Match"+j+"_HMM"+(index+1));
					attInfo.addElement(tempAttribute);
				}
				for(int j = 0; j < hmm.getNumberMatchStates()-1; j++){
					tempAttribute = new Attribute("Score4Insert"+j+"_HMM"+(index+1));
					attInfo.addElement(tempAttribute);
				}
				for(int j = 0; j < hmm.getNumberMatchStates()-2; j++){
					tempAttribute = new Attribute("Score4Delete"+j+"_HMM"+(index+1));
					attInfo.addElement(tempAttribute);
				}
			}

			index++;
			counter = 0;
		}

		tempAttribute = new Attribute("class",my_nominal_class_values);
		attInfo.addElement(tempAttribute);
		Instances transformed = new Instances(oldInstances.relationName()+"_propositiopnalized",attInfo,1);
		transformed.setClassIndex(numNonClassAttributes);

		return transformed;
	}

	private int determineColumns(String[] sequences) {
		int average = 0;
		for(int i = 0;i< sequences.length;i++){
			average += sequences[i].length();
		}
		return Math.round(average/sequences.length);
	}


	public void buildClassifier(Instances data) throws Exception {
		initClassifier(data);
		int iteration = 1;
		do{
			next(iteration);
			iteration++;
		}while(!this.fullyConverged());

	}



	public double classifyInstance(Instance instance) throws Exception {
		double result;
		double [] dist;
		int index;
		dist = distributionForInstance(instance);
		index = Utils.maxIndex(dist);
		if (dist[index] == 0){
			result = Utils.missingValue();
		}
		else{
			result = index;
		}
		return result;
	}

	public double [] distributionForInstance(Instance instance) throws Exception {
		double[] distribution = new double[allHMMs.size()];
		double[] logDistribution = new double[allHMMs.size()];
		double sum = 0.0;
		double sumLogScore = Double.NEGATIVE_INFINITY ;
		double logProbNullModel = 0;

		String testSequence = instance.stringValue(instance.attribute(sequenceIndex));
		if(this.getRestrictSequenceLength() != -1 && testSequence.length() > this.getRestrictSequenceLength()){
			testSequence = testSequence.substring(0,this.getRestrictSequenceLength());
		}

		if(this.isUseNullModel()){
			NullModel nullmodel = new NullModel(true,(allHMMs.get(0)).getAlphabet());
			for(int i = 0; i < testSequence.length(); i++){
				logProbNullModel = logsum(logProbNullModel,nullmodel.getNullModelEmissionProbability(testSequence.charAt(i)+""));
			}
		}

		if(!this.isUseNullModel()){
			NullModel nullmodel = new NullModel(true,(allHMMs.get(0)).getAlphabet());
			for(int i = 0; i < testSequence.length(); i++){
				logProbNullModel = logsum(logProbNullModel,nullmodel.getNullModelEmissionProbability(testSequence.charAt(i)+""));
			}
			//System.out.println("nullmodel: "+logProbNullModel);
			logProbNullModel = 0;
		}

		for(int i = 0;i < allHMMs.size(); i++){
			if(allHMMs.get(i) == null){
				logDistribution[i] = Double.NEGATIVE_INFINITY;
			}
			else{
				ForwardAlgorithm fwd = new ForwardAlgorithm(allHMMs.get(i),testSequence);
				fwd.calculateForward();
				double logProbTestSequence;
				if(this.isUseNullModel()){
					//System.out.println(i+" fwd: "+fwd.getScore()+" nullscore: "+logProbNullModel);
					logProbTestSequence = logsum(fwd.getScore(),logProbNullModel);
				}
				else{
					logProbTestSequence = fwd.getScore();
					//System.out.println(i+" fwd: "+fwd.getScore());
				}

				sumLogScore = logplus(sumLogScore, logProbTestSequence);

				if(Double.isNaN(logProbTestSequence)){
					throw new NumericStabilityException("The HMM score is NaN for test sequence "+testSequence);
				}
				logDistribution[i] = logProbTestSequence;
			}
		}

		if(sampleMethod == SAMPLE_ARTIFICIAL_NEGATIVE){
			for(int i = 0; i < logDistribution.length; i++){
				System.out.print(logDistribution[i]+" ");
			}
			System.out.print(" | ");
		}

		for(int i = 0; i < distribution.length; i++){
			if(logDistribution[i] == Double.NEGATIVE_INFINITY || sumLogScore == Double.NEGATIVE_INFINITY){
				distribution[i] = 0.0;
			}
			else{
				distribution[i] = Math.exp((logDistribution[i] - sumLogScore));
			}
			sum += distribution[i];
		}

		if(sampleMethod == SAMPLE_ARTIFICIAL_NEGATIVE){
			for(int i = 0; i < distribution.length; i++){
				System.out.print(distribution[i]+" ");
			}
			System.out.print(" | ");
		}

		if(sum == 0){
			int index = Utils.maxIndex(logDistribution);
			if(logDistribution[index] == Double.NEGATIVE_INFINITY){
				for(int i = 0; i < distribution.length; i++){
					distribution[i] = trainingClassDistribution[i];
					System.out.println("used training Class Distribution");
				}
				Utils.normalize(distribution);
			}
			else{
				for(int i = 0; i < distribution.length; i++){
					distribution[i] = 0.0;
				}
				distribution[index] = 1.0;
				System.out.println("fwd prob all zero, but not log Score, normalized according to logScore");
			}
		}
		else{
			Utils.normalize(distribution);
		}



		int index = -1;
		for(int i = 0; i < distribution.length; i++){
			if(Double.isNaN(distribution[i]))
				throw new NumericStabilityException("The HMM score is NaN after normalisation.");
			if(distribution[i] == 1.0 && index == -1){
				index = i;
			}
			//if(index == -1)
			//System.out.print(distribution[i]+"   ");
		}
		if(index != -1){
			for(int i = 0; i < distribution.length; i++){
				if(i != index){
					distribution[i] = 0.0;
				}
				//System.out.print(distribution[i]+"  ");
			}
		}
		//System.out.println();
		if(sampleMethod == SAMPLE_ARTIFICIAL_NEGATIVE){
			for(int i = 0; i < distribution.length; i++){
				System.out.print(distribution[i]+" ");
			}
			System.out.print("\n");
		}
		return distribution;
	}

	private int determineColumns(Instances inst) {
		int average = 0;
		for(int i = 0;i< inst.numInstances();i++){
			//System.out.println((inst.instance(i).stringValue(inst.instance(i).attribute(sequenceIndex))).length());
			average += (((inst.instance(i).stringValue(inst.instance(i).attribute(sequenceIndex))).length()));
		}
		//System.out.println(average);
		return Math.round(average/inst.numInstances());
	}

	/**
	 * Returns an enumeration describing the available options.
	 *
	 * @return an enumeration of all the available options.
	 */
	public Enumeration listOptions() {
		Enumeration 	enm;
		Vector		result;

		result = new Vector();

		enm = super.listOptions();
		while (enm.hasMoreElements())
			result.addElement(enm.nextElement());


		result.addElement(new Option(
				"\tchooses a method for sampling (default: no sampling)",
				"Z", 1, "-Z "+ Tag.toOptionList(TAGS_SAMPLE) ));

		result.addElement(new Option(
				"\twhen artifical sampling is used, percentage of overlapping examples according to forward score",
				"P", 1, "-P " ));


		return result.elements();
	}

	/**
	 * Gets the current settings.
	 *
	 * @return an array of strings suitable for passing to setOptions()
	 */
	public String [] getOptions() {
		int       	i;
		Vector    	result;
		String[]  	options;

		result = new Vector();

		options = super.getOptions();
		for (i = 0; i < options.length; i++)
			result.add(options[i]);


		if(getSampleMethod() != -1){
			result.add("-Z");
			result.add("" + getSampleType());
		}

		if(getArtificialPercentage() != -1){
			result.add("-P");
			result.add("" + getArtificialPercentage());
		}

		return (String[]) result.toArray(new String[result.size()]);
	}

	/**
	 * Parses a given list of options. <p/>
	 *
	 * @param options the list of options as an array of strings
	 * @throws Exception if an option is not supported
	 */
	public void setOptions(String[] options) throws Exception {

		super.setOptions(options);

		//positive class needs to at index 0 in arff header
		positiveClassIndex = 0;


		String numberString = Utils.getOption('P', options);
		if (numberString.length() != 0) {
			artificialPercentage = Integer.parseInt(numberString);
		}
		else {
			artificialPercentage = -1;
		}

		numberString = Utils.getOption('Z', options);
		if (numberString.length() != 0)
			setSampleType(new SelectedTag(numberString, TAGS_SAMPLE));
		else
			setSampleType(new SelectedTag(SAMPLE_ALL, TAGS_SAMPLE));
	}

	public String toString() {

		String result = " positive class index:\t"+positiveClassIndex
		+ "\n sample method:\t" + getSampleType();
		if(sampleMethod == SAMPLE_ARTIFICIAL){
			result+= " precentage of positive examples with overlapping score: "+getArtificialPercentage();
		}
		result += "\n";
		if(allHMMs != null){
			for(int i = 0;i < allHMMs.size(); i++){
				ProfileHMM hmm = allHMMs.get(i);
				if(hmm != null){
					result += "\nProfileHMM for class "+i+":\n"
					+"  match states:    \t"+hmm.getNumberMatchStates()
					+"\n  iterations in BW:\t"+numIterationPerClass.get(i)
					+"\n  final score:    \t"+logLikelihoodOfPHMM.get(i)
					+"\n  initial score:\t"+initiallogLikelihoodOfPHMM.get(i)+"\n";
				}
			}
		}

		return result;
	}

	/**
	 * Main method for testing this class.
	 *
	 * @param argv the options
	 */
	public static void main(String [] argv) {

		try {
			System.out.println(Evaluation.evaluateModel(new IterativeProfileHMMClassifier(), argv));
		} catch (Exception e) {
			//System.err.println(e.getMessage());
			e.printStackTrace();
		}
	}


	public String getRevision() {

		return "1.0";
	}


	public void initClassifier(Instances data) throws Exception {
		//  can classifier handle the data?
		getCapabilities().testWithFail(data);
		data = new Instances(data);
		data.deleteWithMissingClass();

		if(data.numAttributes() > 2)
			throw new Exception ("Dataset has to consist of exactly one string attribute and a class attribute");
		if(data.classIndex() == 1)
			sequenceIndex = 0;
		else
			sequenceIndex = 1;



		iteration = 0;


		trainingClassDistribution = new double[data.numClasses()];

		Enumeration enist = data.enumerateInstances();
		datasets = new Vector<Instances>(data.numClasses());
		for(int j=0;j<data.numClasses();j++){
			datasets.add(j,new Instances(data,data.numInstances()));
		}
		while(enist.hasMoreElements()){
			Instance temp = (Instance)enist.nextElement();
			Instances instTemp = datasets.get((int)temp.classValue());
			datasets.remove((int)temp.classValue());
			instTemp.add(temp);
			datasets.add((int)temp.classValue(),instTemp);
		}

		if(sampleMethod == SAMPLE_ALL){
			initPHMMAndSequences(data);
		}
		else{
			initPHMMAndSequencesSampling(data);
		}
	}


	private void initPHMMAndSequencesSampling(Instances data) {
		int numInstances = data.numInstances();
		Instances positiveSet = datasets.get(positiveClassIndex);
		numPositiveInstances = positiveSet.numInstances();
		if(positiveSet.numInstances() == 0){
			trainingClassDistribution[0] = 0.0;
			allHMMs.add(0, null);
			converged.add(0,true);
			logLikelihoodOfPHMM.add(0, null);
			initiallogLikelihoodOfPHMM.add(0, null);
			oldLogLikelihoodOfPHMM.add(0,null);
			numIterationPerClass.add(0, null);
		}
		else{
			trainingClassDistribution[0] = positiveSet.numInstances()/numInstances;
			trainingClassDistribution[1] = (numInstances - positiveSet.numInstances())/numInstances;
			ProfileHMM hmm;
			if(this.getRestrictSequenceLength() != -1){
				hmm = new ProfileHMM(this.getRestrictSequenceLength(),this.getAlphabet(),useNullModel,!transitionsEmissionsNotInLog);
			}
			else{
				if(this.getRestrictMatchColumns() != -1){
					hmm = new ProfileHMM(this.getRestrictMatchColumns(),this.getAlphabet(),useNullModel,!transitionsEmissionsNotInLog);
				}
				else{
					int matchColumns = determineColumns(positiveSet);
					hmm = new ProfileHMM(matchColumns,this.getAlphabet(),useNullModel, !transitionsEmissionsNotInLog);
				}
			}
			allHMMs.add(0,hmm);
			converged.add(0,false);
			logLikelihoodOfPHMM.add(0, Double.NEGATIVE_INFINITY);
			initiallogLikelihoodOfPHMM.add(0, Double.NEGATIVE_INFINITY);
			oldLogLikelihoodOfPHMM.add(0, Double.NEGATIVE_INFINITY);
			numIterationPerClass.add(0, -1);
		}
		String[] allTrainingSequencesForClass = new String[positiveSet.numInstances()];
		for(int k =0;k<positiveSet.numInstances();k++){
			String sequence = positiveSet.instance(k).stringValue(positiveSet.instance(k).attribute(sequenceIndex));
			if(this.getRestrictSequenceLength() != -1 && sequence.length()>this.getRestrictSequenceLength()){
				sequence = sequence.substring(0,this.getRestrictSequenceLength());
			}
			allTrainingSequencesForClass[k]=sequence;
		}
		allTrainingSequences.add(0,allTrainingSequencesForClass);

		converged.add(1,false);
		logLikelihoodOfPHMM.add(1, Double.NEGATIVE_INFINITY);
		initiallogLikelihoodOfPHMM.add(1, Double.NEGATIVE_INFINITY);
		oldLogLikelihoodOfPHMM.add(1, Double.NEGATIVE_INFINITY);
		numIterationPerClass.add(1, -1);

		if(sampleMethod != SAMPLE_UNIFORM_ONCE){
			allHMMs.add(1,null);
			allTrainingSequences.add(1,null);
		}
		else{
			int negativeClassIndex = 1;
			if(positiveClassIndex == 1){
				negativeClassIndex =0;
			}
			Instances negativeSet = datasets.get(negativeClassIndex);

			allTrainingSequencesForClass = new String[positiveSet.numInstances()];
			Random rand = new Random(numPositiveInstances+1);//+1 to get identical results with SAMPLE_UNIFORM. iteration count starts with 1
			for(int i = 0; i < numPositiveInstances; i++){
				int index = rand.nextInt(negativeSet.numInstances());
				String sequence = negativeSet.instance(index).stringValue(negativeSet.instance(index).attribute(sequenceIndex));
				if(this.getRestrictSequenceLength() != -1 && sequence.length()>this.getRestrictSequenceLength()){
					sequence = sequence.substring(0,this.getRestrictSequenceLength());
				}
				allTrainingSequencesForClass[i] = sequence;
			}
			allTrainingSequences.add(1,allTrainingSequencesForClass);

			ProfileHMM hmm;
			if(this.getRestrictSequenceLength() != -1){
				hmm = new ProfileHMM(this.getRestrictSequenceLength(),this.getAlphabet(),useNullModel,!transitionsEmissionsNotInLog);
			}
			else{
				if(this.getRestrictMatchColumns() != -1){
					hmm = new ProfileHMM(this.getRestrictMatchColumns(),this.getAlphabet(),useNullModel,!transitionsEmissionsNotInLog);
				}
				else{
					int matchColumns = determineColumns(allTrainingSequencesForClass);
					hmm = new ProfileHMM(matchColumns,this.getAlphabet(),useNullModel, !transitionsEmissionsNotInLog);
				}
			}
			allHMMs.add(1,hmm);
		}
	}


	private void initPHMMAndSequences(Instances data) {
		int numInstances = data.numInstances();
		for (int i = 0; i < data.numClasses(); i++) {
			Instances trainingSetClassi = datasets.get(i);
			if(trainingSetClassi.numInstances() == 0){
				trainingClassDistribution[i] = 0.0;
				allHMMs.add(i, null);
				converged.add(i,true);
				logLikelihoodOfPHMM.add(i, null);
				initiallogLikelihoodOfPHMM.add(i, null);
				oldLogLikelihoodOfPHMM.add(i,null);
				numIterationPerClass.add(i, null);
			}
			else{
				trainingClassDistribution[i] = trainingSetClassi.numInstances()/numInstances;
				ProfileHMM hmm;
				if(this.getRestrictSequenceLength() != -1){
					hmm = new ProfileHMM(this.getRestrictSequenceLength(),this.getAlphabet(),useNullModel,!transitionsEmissionsNotInLog);
				}
				else{
					if(this.getRestrictMatchColumns() != -1){
						hmm = new ProfileHMM(this.getRestrictMatchColumns(),this.getAlphabet(),useNullModel,!transitionsEmissionsNotInLog);
					}
					else{
						int matchColumns = determineColumns(trainingSetClassi);
						hmm = new ProfileHMM(matchColumns,this.getAlphabet(),useNullModel, !transitionsEmissionsNotInLog);
					}
				}
				allHMMs.add(i,hmm);
				converged.add(i,false);
				logLikelihoodOfPHMM.add(i, Double.NEGATIVE_INFINITY);
				initiallogLikelihoodOfPHMM.add(i, Double.NEGATIVE_INFINITY);
				oldLogLikelihoodOfPHMM.add(i, Double.NEGATIVE_INFINITY);
				numIterationPerClass.add(i, -1);
			}

			String[] allTrainingSequencesForClass = new String[trainingSetClassi.numInstances()];
			for(int k =0;k<trainingSetClassi.numInstances();k++){
				String sequence = trainingSetClassi.instance(k).stringValue(trainingSetClassi.instance(k).attribute(sequenceIndex));
				if(this.getRestrictSequenceLength() != -1 && sequence.length()>this.getRestrictSequenceLength()){
					sequence = sequence.substring(0,this.getRestrictSequenceLength());
				}
				allTrainingSequencesForClass[k]=sequence;
			}
			allTrainingSequences.add(i,allTrainingSequencesForClass);
		}
	}



	public void next(int iteration) throws Exception {

		double logLikelihood;
		BaumWelchLearner bwl = null;

		this.iteration = iteration;

		if(!loadPositiveModel){
			trainOneClass(iteration, 0);
		}
		else{
			load();
		}
		if(sampleMethod != SAMPLE_ALL && sampleMethod != SAMPLE_UNIFORM_ONCE ){
			if(sampleMethod == SAMPLE_ARTIFICIAL){
				sampleArtifical();
			}
			else{
				if(sampleMethod == SAMPLE_ARTIFICIAL_NEGATIVE){
					sampleArtificalFromNegatives();
				}
				else{
					sampleNegatives();
				}
			}
		}
		trainOneClass(iteration, 1);
	}


	private void sampleArtifical() throws IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException {
		ProfileHMM positiveHMM = allHMMs.get(0);
		//System.out.println(positiveHMM.toString());
		String[] sampledTrainingSequencesForNegativeClass = new String[numPositiveInstances];
		String[] allTrainingSequencesForPositiveClass = allTrainingSequences.get(0);
		TreeMap sortedList = sort(positiveHMM, allTrainingSequencesForPositiveClass);
		SequenceGenerator seqGen = new SequenceGenerator(positiveHMM,iteration);

		double lowestPositiveScore = 0;
		if(artificialPercentage <= 0 || artificialPercentage >= 100){
			SortedNegative currentLow = (SortedNegative)sortedList.firstKey();
			lowestPositiveScore = currentLow.getScore();
		}
		else{
			int index = (int)Math.round((artificialPercentage/100)*numPositiveInstances);
			SortedNegative currentLow = null;
			for(int i = 0; i < index; i++){
				currentLow = (SortedNegative)sortedList.firstKey();
				sortedList.remove(currentLow);
			}
			lowestPositiveScore = currentLow.getScore();
		}
		//System.out.println("lowest score: "+lowestPositiveScore);
		sortedList = null;
		for(int i = 0; i < numPositiveInstances; i++){
			String negativeSeq = seqGen.generateLowerScoringSequence(lowestPositiveScore);
			sampledTrainingSequencesForNegativeClass[i] = negativeSeq;
		}
		allTrainingSequences.set(1, sampledTrainingSequencesForNegativeClass);

		if(allHMMs.get(1)==null){
			ProfileHMM hmm;
			if(this.getRestrictSequenceLength() != -1){
				hmm = new ProfileHMM(this.getRestrictSequenceLength(),this.getAlphabet(),useNullModel,!transitionsEmissionsNotInLog);
			}
			else{
				if(this.getRestrictMatchColumns() != -1){
					hmm = new ProfileHMM(this.getRestrictMatchColumns(),this.getAlphabet(),useNullModel,!transitionsEmissionsNotInLog);
				}
				else{
					int matchColumns = determineColumns(sampledTrainingSequencesForNegativeClass);
					hmm = new ProfileHMM(matchColumns,this.getAlphabet(),useNullModel, !transitionsEmissionsNotInLog);
				}
			}
			allHMMs.set(1,hmm);
		}
	}

	private void sampleArtificalFromNegatives() throws IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException {
		if(allHMMs.get(1)==null){
			ProfileHMM hmm;
			if(this.getRestrictSequenceLength() != -1){
				hmm = new ProfileHMM(this.getRestrictSequenceLength(),this.getAlphabet(),useNullModel,!transitionsEmissionsNotInLog);
			}
			else{
				if(this.getRestrictMatchColumns() != -1){
					hmm = new ProfileHMM(this.getRestrictMatchColumns(),this.getAlphabet(),useNullModel,!transitionsEmissionsNotInLog);
				}
				else{
					throw new InvalidStructureException("Doesn't work with this sampling method");
				}
			}
			allHMMs.set(1,hmm);
		}
		ProfileHMM negativeHMM = allHMMs.get(1);
		ProfileHMM positiveHMM = allHMMs.get(0);

		//test
		TreeMap testList = sort(positiveHMM, allTrainingSequences.get(0));
		double bound = 0 ;
		for(int u = 0; u < 2 ; u++){
			bound = ((SortedNegative)testList.firstKey()).getScore();
			testList.remove((SortedNegative)testList.firstKey());
		}
		System.out.println(bound);
		testList = null;

		String[] overSampledTrainingSequencesForNegativeClass = new String[2*numPositiveInstances];
		String[] sampledTrainingSequencesForNegativeClass = new String[numPositiveInstances];

		SequenceGenerator seqGen = new SequenceGenerator(negativeHMM,iteration);
		for(int i = 0; i < (2*numPositiveInstances); i ++){
			String negativeSeq = seqGen.generateLowerScoringSequence(bound);//String negativeSeq = seqGen.generateSequence();
			overSampledTrainingSequencesForNegativeClass[i] = negativeSeq;
		}

		//test
		String[] test = overSampledTrainingSequencesForNegativeClass;
		double[]length = new double[test.length];
		for(int i = 0; i < test.length; i++ ){
			length[i] = test[i].length();
		}
		Arrays.sort(length);
		for(int i = 0; i < length.length; i++ ){
			System.out.print(length[i]+" | ");
		}
		System.out.println("\n");


		if(artificialPercentage == 0){
			allTrainingSequences.set(1, overSampledTrainingSequencesForNegativeClass);
		}
		else{
			TreeMap sortedList = sort(positiveHMM, overSampledTrainingSequencesForNegativeClass);
			for(int i = 0; i < numPositiveInstances; i++){
				SortedNegative currentHigh;
				if(artificialPercentage > 0){
					//take highest
					currentHigh = (SortedNegative)sortedList.lastKey();
				}
				else{
					//take lowest
					currentHigh = (SortedNegative)sortedList.firstKey();
				}
				int index = currentHigh.getIndex();
				System.out.print(index+","+currentHigh.getScore()+" | ");

				//if(index == 45 || index == 96){
				double logProbNullModel = 0;
				NullModel nullmodel = new NullModel(true,(allHMMs.get(0)).getAlphabet());
				for(int u = 0; u < overSampledTrainingSequencesForNegativeClass[index].length(); u++){
					logProbNullModel = logsum(logProbNullModel,nullmodel.getNullModelEmissionProbability(null));
				}
				System.out.println("nullscore for index"+index+": "+logProbNullModel+";"+overSampledTrainingSequencesForNegativeClass[index].length());
				//}

				sortedList.remove(currentHigh);
				sampledTrainingSequencesForNegativeClass[i] = overSampledTrainingSequencesForNegativeClass[index];
			}
			System.out.println("");
			allTrainingSequences.set(1, sampledTrainingSequencesForNegativeClass);
		}
	}


	private void load() {
		allHMMs.set(0,loadPositiveModelForm.allHMMs.get(positiveClassIndex));
		converged.set(0,loadPositiveModelForm.converged.get(positiveClassIndex));
		logLikelihoodOfPHMM.set(0, loadPositiveModelForm.logLikelihoodOfPHMM.get(positiveClassIndex));
		initiallogLikelihoodOfPHMM.set(0, loadPositiveModelForm.initiallogLikelihoodOfPHMM.get(positiveClassIndex));
		oldLogLikelihoodOfPHMM.set(0, loadPositiveModelForm.oldLogLikelihoodOfPHMM.get(positiveClassIndex));
		numIterationPerClass.set(0, loadPositiveModelForm.numIterationPerClass.get(positiveClassIndex));
		loadPositiveModel = false;
	}


	private void trainOneClass(int iteration, int index)
	throws NumericStabilityException, IllegalSymbolException,
	InvalidStructureException, InvalidViterbiPathException {

		double logLikelihood;
		BaumWelchLearner bwl;
		if((allTrainingSequences.get(index)).length != 0 && converged.get(index)== false){

			//learn
			//System.out.println(allHMMs.get(index));
			bwl = new BaumWelchLearner(allTrainingSequences.get(index), logLikelihoodThreshold, allHMMs.get(index), learnInsertEmissions, isMemorySensitive());
			if(getBaumWelchOption() == 1){
				bwl.setAverageLikelihoodOverSequenceNumber(true);
			}
			if(getBaumWelchOption() == 2){
				bwl.setAverageLikelihoodOverResidueNumber(true);
			}
			ProfileHMM learnt;

			//System.out.println("start training HMM");
			learnt = bwl.learnFast();
			//System.out.println("finished training HMM");

			logLikelihood = bwl.getLogLikelihood();
			logLikelihoodOfPHMM.set(index, logLikelihood);
			if(iteration == 1){
				initiallogLikelihoodOfPHMM.set(index, bwl.getInitialLogLikelihood());
			}

			allHMMs.set(index, learnt);


			numIterationPerClass.set(index,iteration);
			if(Math.abs(oldLogLikelihoodOfPHMM.get(index) - logLikelihood) <= logLikelihoodThreshold){
				converged.set(index, true);
				numIterationPerClass.set(index,iteration);
			}
			oldLogLikelihoodOfPHMM.set(index,bwl.getLogLikelihood());
		}
	}

	private void sampleNegatives() throws IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException {
		ProfileHMM positiveHMM = allHMMs.get(0);
		int negativeClassIndex = 1;
		if(positiveClassIndex == 1){
			negativeClassIndex =0;
		}


		Instances negativeSet = new Instances(datasets.get(negativeClassIndex));
		String[] allTrainingSequencesForClass = new String[negativeSet.numInstances()];
		for(int k =0;k<negativeSet.numInstances();k++){
			String sequence = negativeSet.instance(k).stringValue(negativeSet.instance(k).attribute(sequenceIndex));
			if(this.getRestrictSequenceLength() != -1 && sequence.length()>this.getRestrictSequenceLength()){
				sequence = sequence.substring(0,this.getRestrictSequenceLength());
			}
			allTrainingSequencesForClass[k]=sequence;
		}


		String[] sampledTrainingSequencesForClass = new String[numPositiveInstances];

		//take completely at random
		if(sampleMethod == SAMPLE_UNIFORM){
			Random rand = new Random(iteration+numPositiveInstances);
			for(int i = 0; i < numPositiveInstances; i++){
				int index = rand.nextInt(negativeSet.numInstances());
				sampledTrainingSequencesForClass[i] = allTrainingSequencesForClass[index];
			}
			allTrainingSequences.set(1, sampledTrainingSequencesForClass);
		}
		else{
			//take completely at random twice as much as needed, score them with the positive PHMM and take the higher scoring half
			if(sampleMethod == SAMPLE_UNIFORM_PLUS){
				Random rand = new Random(iteration+numPositiveInstances);
				int limit = 2*numPositiveInstances;
				String[] overSampledTrainingSequencesForClass = new String[limit];
				if(negativeSet.numInstances() < limit){
					System.out.println("Sampling method doesn't make sense");
					limit = negativeSet.numInstances();
				}
				for(int i = 0; i < limit; i++){
					int index = rand.nextInt(negativeSet.numInstances());
					overSampledTrainingSequencesForClass[i] = allTrainingSequencesForClass[index];
				}
				if (artificialPercentage == 0){
					allTrainingSequences.set(1, overSampledTrainingSequencesForClass);
				}
				else{
					TreeMap sortedList = sort(positiveHMM, overSampledTrainingSequencesForClass);
					for(int i = 0; i < numPositiveInstances; i++){
						SortedNegative currentHigh;
						if(artificialPercentage > 0){
							//take highest
							currentHigh = (SortedNegative)sortedList.lastKey();
						}
						else{
							//take lowest
							currentHigh = (SortedNegative)sortedList.firstKey();
						}
						int index = currentHigh.getIndex();
						sortedList.remove(currentHigh);
						sampledTrainingSequencesForClass[i] = overSampledTrainingSequencesForClass[index];
					}
					allTrainingSequences.set(1, sampledTrainingSequencesForClass);
				}
			}

			else{
				TreeMap sortedList = sort(positiveHMM, allTrainingSequencesForClass);

				//take highest scoring
				if(sampleMethod == SAMPLE_HIGHEST){
					for(int i = 0; i < numPositiveInstances; i++){
						SortedNegative currentHigh = (SortedNegative)sortedList.lastKey();
						int index = currentHigh.getIndex();
						sortedList.remove(currentHigh);
						sampledTrainingSequencesForClass[i] = allTrainingSequencesForClass[index];
					}
				}
				//take lowest scoring
				if(sampleMethod == SAMPLE_LOWEST){
					for(int i = 0; i < numPositiveInstances; i++){
						SortedNegative currentHigh = (SortedNegative)sortedList.firstKey();
						int index = currentHigh.getIndex();
						sortedList.remove(currentHigh);
						sampledTrainingSequencesForClass[i] = allTrainingSequencesForClass[index];
					}
				}
				//take uniformly from sorted list
				if(sampleMethod == SAMPLE_STRATIFIED){
					int numNegativeInstances = allTrainingSequencesForClass.length;
					int groupSize = numNegativeInstances / numPositiveInstances;
					int remainder = numNegativeInstances - (groupSize * numPositiveInstances);
					Random rand = new Random(iteration+numPositiveInstances);
					for(int j = 0; j < numPositiveInstances; j++){
						int adjustedGroupSize = groupSize;
						if (remainder > 0){
							remainder --;
							adjustedGroupSize++;
						}
						int chosenIndex = rand.nextInt(adjustedGroupSize);
						for(int i = 0; i <adjustedGroupSize; i++){
							SortedNegative currentHigh = (SortedNegative)sortedList.firstKey();
							if(i == chosenIndex){
								int index = currentHigh.getIndex();
								sampledTrainingSequencesForClass[j] = allTrainingSequencesForClass[index];
							}
							sortedList.remove(currentHigh);
						}
					}
				}
				allTrainingSequences.set(1, sampledTrainingSequencesForClass);
				sortedList = null;
			}
		}
		if(allHMMs.get(1)==null){
			ProfileHMM hmm;
			if(this.getRestrictSequenceLength() != -1){
				hmm = new ProfileHMM(this.getRestrictSequenceLength(),this.getAlphabet(),useNullModel,!transitionsEmissionsNotInLog);
			}
			else{
				if(this.getRestrictMatchColumns() != -1){
					hmm = new ProfileHMM(this.getRestrictMatchColumns(),this.getAlphabet(),useNullModel,!transitionsEmissionsNotInLog);
				}
				else{
					int matchColumns = determineColumns(sampledTrainingSequencesForClass);
					hmm = new ProfileHMM(matchColumns,this.getAlphabet(),useNullModel, !transitionsEmissionsNotInLog);
				}
			}
			allHMMs.set(1,hmm);
		}
	}


	private TreeMap sort(ProfileHMM positiveHMM,
			String[] allTrainingSequencesForClass) throws IllegalSymbolException,
			InvalidStructureException, InvalidViterbiPathException {

		TreeMap sortedList = new TreeMap();
		for(int i = 0; i < allTrainingSequencesForClass.length; i ++){
			ForwardAlgorithm fwd = new ForwardAlgorithm(positiveHMM,allTrainingSequencesForClass[i]);
			fwd.calculateForward();
			double score = fwd.getScore();
			//System.out.println(score);
			SortedNegative sn = new SortedNegative(i,score);
			sortedList.put(sn,i);
		}
		return sortedList;
	}

	private class SortedNegative implements Comparable<SortedNegative>{

		double score;

		int index;

		public SortedNegative(int i, double d){
			this.index =i;
			this.score = d;
		}

		public int compareTo(SortedNegative sn){
			if(this.getScore() == Double.NEGATIVE_INFINITY){
				if(sn.getScore() == Double.NEGATIVE_INFINITY){
					return this.getIndex() - sn.getIndex();
				}
				else{
					return -1;
				}
			}
			if(sn.getScore() == Double.NEGATIVE_INFINITY){
				return 1;
			}
			double diff = this.getScore() - sn.getScore();
			if(diff > 0){
				return 1;
			}
			else{
				if (diff < 0){
					return -1;
				}
				else{
					return this.getIndex() - sn.getIndex();
				}
			}
		}

		public boolean equals(Object obj){
			SortedNegative sn = (SortedNegative)obj;
			if(sn.getScore() == this.getScore() && sn.getIndex() == this.getIndex()){
				return true;
			}
			return false;
		}

		public double getScore() {
			return score;
		}

		public void setScore(double score) {
			this.score = score;
		}

		public int getIndex() {
			return index;
		}

		public void setIndex(int index) {
			this.index = index;
		}

	}



	public Object clone() throws CloneNotSupportedException{
		return this.clone();
	}

	public boolean fullyConverged(){
		if(converged.isEmpty()){
			return false;
		}
		else{
			for(int i = 0; i < converged.size(); i++){
				if(converged.get(i) == false){
					return false;
				}
			}
			return true;
		}

	}


	public int getIteration() {
		return iteration;
	}


	public void resetAllTrainingSequences() {
		this.allTrainingSequences = null;
	}

	/**
	 * Returns the number of stored HMMs.
	 *
	 * @return the number of HMMs
	 */
	public int numProfilHMM() {
		return allHMMs.size();
	}

	/**
	 * Returns the specified HMM.
	 *
	 * @param i the index of the HMM to retrieve
	 * @return the requested HMM
	 * @see #numProfilHMM()
	 */
	public ProfileHMM getProfileHMM(int i) {
		return allHMMs.get(i);
	}

	/**
	 * Adds the specified HMM at the specified position.
	 *
	 * @param profileHMM the HMM to add
	 * @param i the index where to add the HMM
	 */
	public void setProfileHMM(ProfileHMM profileHMM, int i) {
		allHMMs.add(i, profileHMM);
	}


	public static long getSerialVersionUID() {
		return serialVersionUID;
	}

	public int getPositiveClassIndex() {
		return positiveClassIndex;
	}

	public void setPositiveClassIndex(int positiveClassIndex) {
		this.positiveClassIndex = positiveClassIndex;
	}

	public int getSampleMethod() {
		return sampleMethod;
	}

	public void setSampleMethod(int sampleMethod) {
		this.sampleMethod = sampleMethod;
	}


	public SelectedTag getSampleType() {
		return new SelectedTag(sampleMethod, TAGS_SAMPLE);
	}

	public void setSampleType(SelectedTag newType) {
		if (newType.getTags() == TAGS_SAMPLE) {
			sampleMethod = newType.getSelectedTag().getID();
		}
	}


	public boolean isLoadPositiveModel() {
		return loadPositiveModel;
	}


	public void setLoadPositiveModel(boolean loadPositiveModel) {
		this.loadPositiveModel = loadPositiveModel;
	}

	public IterativeProfileHMMClassifier getLoadPositiveModelForm() {
		return loadPositiveModelForm;
	}

	public void setLoadPositiveModelForm(
			IterativeProfileHMMClassifier loadPositiveModelForm) {
		this.loadPositiveModelForm = loadPositiveModelForm;
	}


	public double getArtificialPercentage() {
		return artificialPercentage;
	}


	public void setArtificialPercentage(double artificialPercentage) {
		this.artificialPercentage = artificialPercentage;
	}



}
