/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    ChangeableDistribution.java
 *    Copyright (C) 2010 Stefan Mutter
 *
 */
package weka.classifiers.sequence.core;

/**
<!-- globalinfo-start -->
* class that models a distribution that is not fixed but can be changed. After each change the probabilities are normalised to sum up to one again
<!-- globalinfo-end -->
* 
* @author Stefan Mutter (pHMM4weka@gmail.com)
* @author Royal Veterinary and Agricultural University, Copenhagen, Denmark (logplus method)
* @version $Revision: 4 $
*/
public class ChangeableDistribution extends Distribution {

  private static final long serialVersionUID = 5399043391530434461L;

  /**
 * Constructor
 * @param alpha
 * @param useLogSpace
 */
public ChangeableDistribution(Alphabet alpha, boolean useLogSpace){
    super(alpha,useLogSpace);
  }
  
  public void setProb(String s, double prob) throws NumericStabilityException, IllegalSymbolException{
    probs[alphabet.indexOfAlphabetSymbol(s)] = prob;
    adjust();
  }

  public void setProbWithIndex(int index, double prob) throws NumericStabilityException{
    probs[index] = prob;
    adjust();
  }
  
  

  /**
 * mormalises the probabilities after an adjustment
 * @throws NumericStabilityException
 */
protected void adjust() throws NumericStabilityException {
    if(!useLogSpace){
      double sum = 0;
      for (int i = 0; i < probs.length; i++) {
	sum += probs[i];
      }
      if (sum <= 0) {
	throw new NumericStabilityException("Can't normalize array. Sum is zero or smaller.");
      }
      for (int i = 0; i < probs.length; i++) {
	probs[i] /= sum;
      }
    }
    else{
      double sum = Double.NEGATIVE_INFINITY;
      for (int i = 0; i < probs.length; i++) {
	sum = logplus(sum,probs[i]);
      }
      if (sum == Double.NEGATIVE_INFINITY) {
	throw new NumericStabilityException("Can't normalize array. Sum is negative infinity.");
      }
      for (int i = 0; i < probs.length; i++) {
	probs[i] = probs[i] - sum;
      }
    }
  }
  
/**
 * compute log(p+q) from plog = log p and qlog = log q, using that
 * log (p + q) = log (p(1 + q/p)) = log p + log(1 + q/p) 
 *  = log p + log(1 + exp(log q - log p)) = plog + log(1 + exp(logq - logp))
 *  and that log(1 + exp(d)) < 1E-17 for d < -37.
 *  This method is taken from http://www.itu.dk/~sestoft/bsa/Match3.java
 *  For Licence agreement see licenceMatch3.txt
 * @param plog plog = log p
 * @param qlog qlog = log q
 * @return log(p+q)
 */
  protected static double logplus(double plog, double qlog) {
    double max, diff;
    if (plog > qlog) {
      if (qlog == Double.NEGATIVE_INFINITY)
        return plog;
      else {
        max = plog; diff = qlog - plog;
      } 
    } else {
      if (plog == Double.NEGATIVE_INFINITY)
        return qlog;
      else {
        max = qlog; diff = plog - qlog;
      }
    }
    // Now diff <= 0 so Math.exp(diff) will not overflow
    return max + (diff < -37 ? 0 : Math.log(1 + Math.exp(diff)));
  }
  
}
