/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    Distribution.java
 *    Copyright (C) 2010 Stefan Mutter
 *
 */
package weka.classifiers.sequence.core;

import java.io.Serializable;
import java.util.Random;

import weka.core.Utils;

/**
<!-- globalinfo-start -->
 * class to model a distribution. Offers general methods specific distribution may require.
<!-- globalinfo-end -->
 * 
 * @author Stefan Mutter (pHMM4weka@gmail.com)
 * @version $Revision: 4 $
 */
/**
 * @author mutter
 *
 */
/**
 * @author mutter
 *
 */
public class Distribution implements Serializable{

	private static final long serialVersionUID = 6791890020030731250L;

	protected Alphabet alphabet;

	/**
	 * array containing probability or their logs. One element per alphabet symbol
	 */
	protected double[] probs;

	/**
	 * flag to indicate that calculations are done in log space (recommended)
	 */
	protected boolean useLogSpace;

	/**
	 * Constructor that sets all probabilities to 0 or according log space value
	 * @param alpha
	 * @param useLogSpace
	 */
	public Distribution(Alphabet alpha, boolean useLogSpace){
		this.alphabet = alpha;
		this.probs = new double[alphabet.alphabetSize()];
		this.useLogSpace = useLogSpace;
		for(int i = 0;i < probs.length;i++){
			if(!useLogSpace){
				probs[i] = 0.0;
			}
			else{
				probs[i] = Double.NEGATIVE_INFINITY;
			}
		}
	}


	/**
	 * returns probability or its log for a specific symbol of the alphabet
	 * @param s string representing a symbol of the alphabet
	 * @return probability or its log value of s
	 * @throws IllegalSymbolException if symbol s does not exist in alphabet
	 */
	public double getProb(String s) throws IllegalSymbolException{
		return probs[alphabet.indexOfAlphabetSymbol(s)];
	}


	public String toString(){
		StringBuffer output = new StringBuffer();
		output.append(alphabet.toString()+"\n");
		for(int i = 0;i < probs.length; i++){
			output.append(probs[i]+" ");
		}
		output.append("\n");
		return output.toString();
	}

	/**
	 * @return true if calculation log space, false otherwise
	 */
	public boolean isUseLogSpace() {
		return useLogSpace;
	}

	/**
	 * returns a random symbol of the alphabet depending on its symbol dsitribution
	 * @param seed a seed for the random number generator that generates a probability
	 * @return a random symbol of the alphabet as a string according to the probability distribution
	 */
	public String getSymbol(int seed){
		Random rand = new Random(seed);
		double actualProb = rand.nextDouble();
		int index = 0;
		double sum = 0.0;
		if(!useLogSpace){
			for(; index < probs.length; index++){
				sum += probs[index];
				if(sum > actualProb){
					return alphabet.getSymbolAtIndex(index);
				}
			}
		}
		else{
			double[] realprobs = new double[probs.length];
			for(int i = 0; i < probs.length; i++){
				if(probs[i] == Double.NEGATIVE_INFINITY){
					realprobs[i] = 0;
				}
				else{
					realprobs[i] = Math.exp(probs[i]); 
				}
			}
			Utils.normalize(realprobs);
			for(; index < realprobs.length; index++){
				sum += realprobs[index];
				if(sum > actualProb){
					return alphabet.getSymbolAtIndex(index);
				}
			}
		}
		return alphabet.getSymbolAtIndex(index);
	}

}
