/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    DynamicProgAlgorithms.java
 *    Copyright (C) 2010 Stefan Mutter
 *
 */
package weka.classifiers.sequence.core;

/**
<!-- globalinfo-start -->
* provides the basic setup for the three dynamic programming algorithms used in training and testing PHMMs: forward, backward and Viterbi algorithm
<!-- globalinfo-end -->
* 
* @author Stefan Mutter (pHMM4weka@gmail.com)
* @version $Revision: 4 $
*/
/**
 * @author mutter
 *
 */
public class DynamicProgAlgorithms extends ProfileHMMAlgorithms {

  private static final long serialVersionUID = -3315055075359274012L;
  
  /**
 * the delete matrix
 */
  protected double[][] deleteMatrix;
  
  /**
 *the insert matrix 
 */
  protected double[][] insertMatrix;
  
  /**
   *the match matrix 
   */
  protected double[][] matchMatrix;
  
  /**
 * the final score of the dynamic programming algorithm
 */
  protected double score;
  
  /**
 * sequence under consideration
 */
  protected String sequence;
  
  
  /**
 * Constructor 
 * @param net the PHMM
 * @param sequence the sequence to be evaluated
 */
public DynamicProgAlgorithms(ProfileHMM net, String sequence) {
    super(net);
    
    //end s special end symbol to make dynamic programming easier
    EmissionState end = net.getEndState();
    SpecialAlphabet endAlphabet = (SpecialAlphabet)end.getAlphabet();
    this.sequence = sequence+endAlphabet.toString();

  }
  
  /**
 * returns the probability or log likelihood of sequence symbol at a given position for a emission state
 * @param actual the emission state
 * @param sequencePosition the sequence position
 * @return probability or log likelihood of sequence symbol at a given position for a emission state
 * @throws IllegalSymbolException
 */
private double getProbability(EmissionState actual, int sequencePosition) throws IllegalSymbolException{
    return actual.getEmissionProbability(""+sequence.charAt(sequencePosition));
  }
  
/**
 * returns the probability or log likelihood of sequence symbol at a given position for a emission state, 
 * if a null model is used it returns the probability given the null model or the log-odds score
 * @param actual the emission state
 * @param sequencePosition the sequence position
 * @return probability or log likelihood of sequence symbol at a given position for a emission state
 * @throws IllegalSymbolException
 */
  protected double calculateLogEmissionProbability(EmissionState actual, int sequencePosition) throws IllegalSymbolException {
    double prob = getProbability(actual, sequencePosition);
    if(!logForTransitionsEmissions){
      if(useNullModel){
        if(actual.equals(net.getEndState())){
          prob = 1.0;
        }
        else{
          prob = prob / ((net.getNullModel()).getNullModelEmissionProbability(""+sequence.charAt(sequencePosition)));
        }
      }
      return Math.log(prob);
    }
    else{
      if(useNullModel){
        if(actual.equals(net.getEndState())){
          prob = 0.0;
        }
        else{
          prob = prob - ((net.getNullModel()).getNullModelEmissionProbability(""+sequence.charAt(sequencePosition)));
        }
      }
      return prob;
    }
  }


public double getDeleteMatrixPos(int stateNumber, int sequencePosition) {
    return deleteMatrix[stateNumber][sequencePosition];
  }

  public double getInsertMatrixPos(int stateNumber, int sequencePosition) {
    return insertMatrix[stateNumber][sequencePosition];
  }

  public double getMatchMatrixPos(int stateNumber, int sequencePosition) {
    return matchMatrix[stateNumber][sequencePosition];
  }
  
  /**
 * @return the score of the dynamic programming calculation
 */
public double getScore(){
    return score;
  }

}