/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    ProfileHMMAlgorithms.java
 *    Copyright (C) 2010 Stefan Mutter
 *
 */
package weka.classifiers.sequence.core;

import java.io.Serializable;

/**
<!-- globalinfo-start -->
* class offering standard methods for PHMMs
<!-- globalinfo-end -->
* 
* @author Stefan Mutter (pHMM4weka@gmail.com)
* @author Peter Sestoft, Royal Veterinary and Agricultural University, Copenhagen, Denmark (logplus method)
* @version $Revision: 4 $
*/
public class ProfileHMMAlgorithms implements Serializable{

  private static final long serialVersionUID = -7404699439770901266L;
  protected ProfileHMM net;
  protected boolean useNullModel;
  protected boolean logForTransitionsEmissions;
  

  /**
 * compute log(p+q) from plog = log p and qlog = log q, using that
 * log (p + q) = log (p(1 + q/p)) = log p + log(1 + q/p) 
 *  = log p + log(1 + exp(log q - log p)) = plog + log(1 + exp(logq - logp))
 *  and that log(1 + exp(d)) < 1E-17 for d < -37.
 *  This method is taken from http://www.itu.dk/~sestoft/bsa/Match3.java
 *  For Licence agreement see licenceMatch3.txt
 * @param plog plog = log p
 * @param qlog qlog = log q
 * @return log(p+q)
 */
public static double logplus(double plog, double qlog) {
    double max, diff;
    if (plog > qlog) {
      if (qlog == Double.NEGATIVE_INFINITY)
        return plog;
      else {
        max = plog; diff = qlog - plog;
      } 
    } else {
      if (plog == Double.NEGATIVE_INFINITY)
        return qlog;
      else {
        max = qlog; diff = plog - qlog;
      }
    }
    // Now diff <= 0 so Math.exp(diff) will not overflow
    return max + (diff < -37 ? 0 : Math.log(1 + Math.exp(diff)));
  }

  
  /**
 * @param start start state of transition
 * @param end end state of transition
 * @return log of transition probability
 * @throws InvalidStructureException
 */
public double getIncomingTransitionProbTo(State start, State end) throws InvalidStructureException {
    if(logForTransitionsEmissions){
      return start.getIncomingTransitionProbTo(end);
    }
    else{
      return Math.log(start.getIncomingTransitionProbTo(end));
    }
  }

/**
 * @param start start state of transition
 * @param end end state of transition
 * @return log of transition probability
 * @throws InvalidStructureException
 */
  public double getIncomingTransitionProbFrom(State end, State start) throws InvalidStructureException {
    if(logForTransitionsEmissions){
      return end.getIncomingTransitionProbFrom(start);
    }
    else{
      return Math.log(end.getIncomingTransitionProbFrom(start));
    }
  }
  
  /**
 * @param transition
 * @return log of transition probability
 */
public double getTransitionProbability(Transition transition){
    if(logForTransitionsEmissions){
      return transition.getProbability();
    }
    else{
      return Math.log(transition.getProbability());
    }
  }
  
  /**
 * sets a transition probability. The value must be in log space!
 * @param transition
 * @param probabilityInLogForm log of transition probability
 */
public void setTransitionProbability(Transition transition, double probabilityInLogForm){
    if(logForTransitionsEmissions){
      transition.setProbability(probabilityInLogForm);
    }
    else{
      transition.setProbability(Math.exp(probabilityInLogForm));
    }
  }

  /**
 * computes log(a*b)
 * @param alog  alog = log a
 * @param blog  blog = log b
 * @return log(a*b) = alog + blog
 */
public static double logsum(double alog, double blog) {
    if(alog == Double.NEGATIVE_INFINITY || blog == Double.NEGATIVE_INFINITY){
      return Double.NEGATIVE_INFINITY;
    }
    else{
      return alog+blog;
    }
  }

  /**
 * Constructor
 * @param net the PHMM
 */
public ProfileHMMAlgorithms(ProfileHMM net) {
    super();
    this.net = net;
    if(net.getNullModel() != null){
      useNullModel = true;
    }
    else{
      useNullModel = false;
    }
    this.logForTransitionsEmissions = net.isUseLogSpace();
  }

}