/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    SequenceGenerator.java
 *    Copyright (C) 2010 Stefan Mutter
 *
 */
package weka.classifiers.sequence.core;

import java.io.Serializable;
import java.util.List;
import java.util.Random;

import weka.core.Utils;

/**
<!-- globalinfo-start -->
* generates an artificial sequence according to the PHMM. An upper bound on the score can be set.
<!-- globalinfo-end -->
* 
* @author Stefan Mutter (pHMM4weka@gmail.com)
* @version $Revision: 4 $
*/
public class SequenceGenerator implements Serializable {

  private static final long serialVersionUID = -2685704498277103647L;

  private ProfileHMM hmm;
  
  private Random seedGenerator;

  /**
 * Constructor
 * @param hmm
 * @param seed the random seed
 */
public SequenceGenerator(ProfileHMM hmm, int seed){
    this.hmm = hmm;
    this.seedGenerator = new Random(seed);
  }

  /**
 * generates a sequence
 * @return a sequence string
 */
public String generateSequence(){
    String sequence = "";
    State currentState = hmm.getMatchState(0);
    while(!currentState.equals(hmm.getEndState())){
      //System.out.print(currentState.getFullNameId()+" ");
      if(currentState instanceof EmissionState){
	Distribution dist = ((EmissionState)currentState).getDistribution();
	sequence += dist.getSymbol(seedGenerator.nextInt());
      }
      currentState = calculateNextState(currentState);
    }
    //System.out.print("\n");
    //System.out.println(sequence);
    return sequence;
  }

  /**
 * generates a sequence that has ascore lower than the specified bound
 * @param upperBoundOfScore the bound 
 * @return a sequence string with a score lower than upperBound
 * @throws IllegalSymbolException
 * @throws InvalidStructureException
 * @throws InvalidViterbiPathException
 */
public String generateLowerScoringSequence(double upperBoundOfScore) throws IllegalSymbolException, InvalidStructureException, InvalidViterbiPathException{
    double actualScore;
    String sequence;
    do{
      sequence = generateSequence();
      ForwardAlgorithm fwd = new ForwardAlgorithm(hmm, sequence);
      fwd.calculateForward();
      actualScore = fwd.getScore();
      System.out.println(actualScore);
    }while(actualScore >= upperBoundOfScore);
    return sequence;
  }

  private State calculateNextState(State currentState) {
    Random rand = new Random(seedGenerator.nextInt());
    double actualProb = rand.nextDouble();
    int index = 0;
    double sum = 0.0;
    List<Transition> transitions = currentState.getAllOutgoing();
    //System.out.print(currentState.getFullNameId()+": ");
    if(hmm.isUseLogSpace()){
      double[] realprobs = new double[transitions.size()];
      for(int i = 0; i < realprobs.length; i++){
	if((transitions.get(i)).getProbability() == Double.NEGATIVE_INFINITY){
	  realprobs[i] = 0;
	}
	else{
	  realprobs[i] = Math.exp((transitions.get(i)).getProbability());
	}
      }
      Utils.normalize(realprobs);
//      System.out.print(currentState.getFullNameId()+": ");
     //for(int i = 0; i < realprobs.length; i++){
	//System.out.print(realprobs[i]+" ");
      //}
      //System.out.print("leads to ");
      for(; index < realprobs.length; index++){
	sum += realprobs[index];
	if(sum > actualProb){
	  //System.out.println(index);
	  return (transitions.get(index)).getEnd();
	}
      }
    }
    else{
      for(; index < transitions.size(); index++){
	sum += (transitions.get(index)).getProbability();
	if(sum > actualProb){
	  return (transitions.get(index)).getEnd();
	}
      }
    }
    return (transitions.get(index)).getEnd();
  }

  public ProfileHMM getHmm() {
    return hmm;
  }

  public void setHmm(ProfileHMM hmm) {
    this.hmm = hmm;
  }

}
