/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    State.java
 *    Copyright (C) 2010 Stefan Mutter
 *
 */
package weka.classifiers.sequence.core;

import java.io.Serializable;
import java.util.List;
import java.util.Vector;

/**
<!-- globalinfo-start -->
* class defining a state in a (P)HMM.A state is uniquely identified by its name and id together
<!-- globalinfo-end -->
* 
* @author Stefan Mutter (pHMM4weka@gmail.com)
* @version $Revision: 4 $
*/
public class State implements Serializable{

  private static final long serialVersionUID = 6471794539900349365L;

  protected List<Transition> incoming;

  protected List<Transition> outgoing;
  
  protected String name;
  
  protected int id;
  
  public State(){
    
  }

  /**
 * constructs a new state with given name and id and empty lists for incoming and outgoing transitions
 * @param name
 * @param id
 */
public State(String name, int id) {
    super();
    this.name = name;
    this.id = id;
    this.incoming = new Vector<Transition>();
    this.outgoing = new Vector<Transition>();
  }
  
  public String getName(){
    return this.name;
  }
  
  public int getId(){
    return this.id;
  }
  
  public String getFullNameId(){
    return this.name+this.id;
  }

  public void addOutgoingTransition(Transition transition) throws InvalidStructureException {
    if(transition.getStart() == this){
      outgoing.add(transition);
    }
    else{
      throw new InvalidStructureException("transition start doesn't equal this state");
    }
    
  }

  public void addIncomingTransition(Transition transition) throws InvalidStructureException {
    if(transition.getEnd() == this){
      incoming.add(transition);
    }
    else{
      throw new InvalidStructureException("transition end doesn't equal this state");
    } 
  }
  
  public double getIncomingTransitionProbFrom(State from) throws InvalidStructureException {
    for(int i = 0; i < this.incoming.size(); i++){
      Transition actual = this.incoming.get(i);
      State toTest = actual.getStart();
      if(toTest.equals(from)){
	return actual.getProbability();
      }
    }
    throw new InvalidStructureException("There is no transition from the specified state into this one");
  }
  
  public double getIncomingTransitionProbTo(State to) throws InvalidStructureException {
    for(int i = 0; i < outgoing.size(); i++){
      Transition actual = outgoing.get(i);
      State toTest = actual.getEnd();
      if(toTest.equals(to)){
	return actual.getProbability();
      }
    }
    throw new InvalidStructureException("There is no transition from this state to the specified one");
  }
  
  /**
 * compares two states. They are equal if they share the same name and id.
 * @param toCompare
 * @return tru if states are equal
 */
public boolean equals(State toCompare){
    if(this.id == toCompare.id && this.name.equals(toCompare.name))
      return true;
    else
      return false;
  }
  
  public List<Transition> getAllOutgoing(){
    return this.outgoing;
  }
  
  public String toString(){
    StringBuffer output = new StringBuffer();
    output.append(this.getFullNameId()+": ");
    for(int i = 0;i < outgoing.size();i++){
      Transition actual = outgoing.get(i);
      output.append(actual.toString()+" ");
    }
    output.append("\n");
    return output.toString();
  }
  
}