/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    LogPathScores2Probabilities.java
 *    Copyright (C) 2010 Stefan Mutter
 *
 */
package weka.filters.supervised.attribute;

import weka.core.Capabilities;
import weka.core.Capabilities.Capability;
import weka.core.Instance;
import weka.core.Instances;
import weka.core.Utils;
import weka.filters.Filter;
import weka.filters.SupervisedFilter;

/**
<!-- globalinfo-start -->
* transfers log-odd scores in arff files created by binary PHMMs in mode -M 2 or -M 3 (not -M 2-3) (logarithmic representations) into probabilities (nomalised exponential representation).
* the original arff file contains the log-odd scores of both PHMMs and the class attribute. The resulting arff file contains two probabilities and the class attribute
* Class attribute must be the last attribute in the original file!
* <p/>
<!-- globalinfo-end -->
*
<!-- options-start -->
* no options
*
<!-- options-end -->
*
* @author Stefan Mutter (pHMM4weka@gmail.com)
* @version $Revision: 6 $
*/
public class LogPathScores2Probabilities extends Filter implements SupervisedFilter{


  private static final long serialVersionUID = 7048394782746762929L;



  public String getRevision() {
    return "1.0";
  }

  public String globalInfo() {
    return   "A batch filter that converts arff files created in Mode 2 or Mode 3 (not Mode 23) to probablities for binary PHMMs.";
  }

  public Capabilities getCapabilities() {
    Capabilities result = super.getCapabilities();
    result.enable(Capability.NUMERIC_CLASS);
    result.enable(Capability.NUMERIC_ATTRIBUTES);
    result.enableAllClasses();
    result.enable(Capability.NO_CLASS);  // filter doesn't need class to be set; treats last attribute as class
    return result;
  }

  public boolean setInputFormat(Instances instanceInfo) throws Exception {
    super.setInputFormat(instanceInfo);

    setOutputFormat(new Instances(instanceInfo, 0));

    return true;  // output format is immediately available
  }

  public boolean batchFinished() throws Exception {
    if (getInputFormat() == null)
      throw new NullPointerException("No input instance format defined");

    Instances inst = getInputFormat();//class must be last attribute!
    //System.out.println(inst);
    int numberLogAttributes = inst.numAttributes()-1;
    //System.out.println(hmmNumber);
    for (int i = 0; i < inst.numInstances(); i++) {
      Instance instance = inst.instance(i);
      double[] probs = new double [numberLogAttributes];
      for(int k = 0; k < numberLogAttributes; k++){
	probs [k] = instance.value(k);
      }
      probs = Utils.logs2probs(probs);
      Utils.normalize(probs);
      for(int k =0; k < numberLogAttributes; k++){
	instance.setValue(k,probs[k]);
      }

      push(instance);
    }

    flushInput();
    m_NewBatch = true;
    m_FirstBatchDone = true;
    return (numPendingOutput() != 0);
  }



  /**
   * Main method for testing this class.
   *
   * @param argv
   */
  public static void main(String [] argv) {
    runFilter(new LogPathScores2Probabilities(), argv);
  }

}
