/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    AbstractIterativeEvaluation.java
 *    Copyright (C) 2011 University of Waikato, Hamilton, NZ
 *
 */

package weka.utils;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Random;
import java.util.Vector;

import weka.classifiers.AbstractClassifier;
import weka.classifiers.Evaluation;
import weka.classifiers.evaluation.NominalPrediction;
import weka.classifiers.evaluation.output.prediction.AbstractOutput;
import weka.classifiers.evaluation.output.prediction.Null;
import weka.classifiers.meta.IterativeHMMClassifier;
import weka.classifiers.meta.IterativeHMMPropositionalizer;
import weka.core.Instances;
import weka.core.Utils;
import weka.core.converters.ConverterUtils;

/**
 * Simulates a 10 fold cross-validation run for IterativeHMMPropositionalizer.
 *
 * @author Stefan Mutter (pHMM4weka@gmail.com)
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 12 $
 */
public abstract class AbstractIterativeEvaluation<T extends IterativeHMMClassifier> {

	/**
	 * Container for storing data from each iteration step.
	 *
	 * @author FracPete (fracpete at waikato dot ac dot nz)
	 * @version $Revision: 12 $
	 */
	public static class EvaluationStep {

		/** the scores of the positive examples. */
		protected Vector pos;

		/** the scores of the negative examples. */
		protected Vector neg;

		/** the evaluation object. */
		protected Evaluation eval;

		/** the time in msec used for this step. */
		protected long time;

		public EvaluationStep(Evaluation eval, Vector pos, Vector neg, long time) {
			this.eval = eval;
			this.pos = pos;
			this.neg = neg;
			this.time = time;
		}

		/**
		 * Returns the evaluation object for this iteration.
		 *
		 * @return the evaluation
		 */
		public Evaluation getEvaluation() {
			return eval;
		}

		/**
		 * Returns the scores of the positive examples.
		 *
		 * @return the scores
		 */
		public Vector getPositiveScores() {
			return pos;
		}

		/**
		 * Returns the scores of the negative examples.
		 *
		 * @return the scores
		 */
		public Vector getNegativeScores() {
			return neg;
		}

		/**
		 * Returns the time used for this iteration.
		 *
		 * @return the time
		 */
		public long getTime() {
			return time;
		}
	}

	/**
	 * Used for iterating till convergence is reached.
	 *
   * @author FracPete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 12 $
	 */
	public static class EvaluationIterator<T extends IterativeHMMClassifier>
	  implements Iterator<EvaluationStep> {

		protected AbstractIterativeEvaluation<T> itEval;
		protected boolean converged;
		protected int stepSizeIteration;
		protected int seed;
		protected int folds;
		protected boolean saveOn;
		protected int stopAfterIteration;
		protected Random rand;
		protected long time;
		protected T actualClassifier;
		protected Instances randData;
		protected Vector pos;
		protected Vector neg;
		protected int iteration;
		protected List<IterativeHMMPropositionalizer> classifiersForAllFolds;
		protected List<Boolean> convergedInAllFolds;
	  protected AbstractOutput output;
	  protected StringBuffer outputBuffer;
	  protected boolean stopping;
	  protected boolean stopped;

		public EvaluationIterator(AbstractIterativeEvaluation<T> itEval) {
			this.itEval = itEval;
			actualClassifier = (T) itEval.getClassifier();
			stepSizeIteration = itEval.getIterationSteps();
			seed = actualClassifier.getSeed();
			folds = itEval.getFolds();
			saveOn = itEval.isSaveFolds();
			stopAfterIteration = itEval.getUpperBoundIterations();
			outputBuffer = new StringBuffer();
			output = itEval.getOutput();
			output.setBuffer(outputBuffer);
			output.setHeader(itEval.getTrainingSet());
			stopping = false;
			stopped = false;

			//  randomize data
			rand = new Random(seed);   // create seeded number generator
			randData = new Instances(itEval.getTrainingSet());   // create copy of original data
			System.out.println("Training and Prediction for: "+randData.relationName());
			randData.randomize(rand);         // randomize data with number generator
			if (randData.classAttribute().isNominal())
				randData.stratify(folds);

			actualClassifier.setIterationStepSize(stepSizeIteration);
			actualClassifier.setSeed(seed);

  		pos = new Vector();
  		neg = new Vector();
			converged = false;
			time = 0;

			try {
    		classifiersForAllFolds = new Vector<IterativeHMMPropositionalizer>();
    		convergedInAllFolds = new Vector<Boolean>();
    		for(int i = 0; i < folds; i++){
    			classifiersForAllFolds.add(i,(IterativeHMMPropositionalizer)AbstractClassifier.makeCopy(actualClassifier));
    			convergedInAllFolds.add(i, false);
    		}
			}
			catch (Exception e) {
				e.printStackTrace();
			}
		}

    /**
     * Returns <tt>true</tt> if the iteration has more elements. (In other
     * words, returns <tt>true</tt> if <tt>next</tt> would return an element
     * rather than throwing an exception.)
     *
     * @return <tt>true</tt> if the iterator has more elements.
     */
    public boolean hasNext() {
    	return !converged && !stopped;
    }

    /**
     * Returns the next element in the iteration.
     *
     * @return the next element in the iteration, null in case of an error
     * @throws NoSuchElementException iteration has no more elements.
     */
    public EvaluationStep next() {
    	if (converged)
    		throw new NoSuchElementException("Already converged!");

    	try {
    		long timeStart = System.currentTimeMillis();
    		converged = true;
    		Evaluation eval = new Evaluation(randData);
    		iteration = -1;
    		// perform cross-validation
    		for (int n = 0; n < folds; n++) {
    			if (stopping)
    				break;

    			Instances train = randData.trainCV(folds, n);
    			Instances test = randData.testCV(folds, n);

    			IterativeHMMPropositionalizer trainNow = classifiersForAllFolds.get(n);
    			iteration = trainNow.getIterationCount();
    			trainNow.setActualFold(n+1);
    			trainNow.setNumberOfFolds(folds);
    			System.out.println("Working on fold "+(n+1)+" of "+folds+" in iteration "+(iteration+1));

    			trainNow.buildClassifier(train);
    			if(trainNow.getReUse()){
    				test = trainNow.getPropositionalTestSet(test);
    			}
    			eval.evaluateModel(trainNow, test, output);
    			convergedInAllFolds.set(n, trainNow.hmmsConverged());
    			classifiersForAllFolds.set(n,trainNow);

    			System.out.println(trainNow);
    		}

    		if (stopping) {
    			stopped = true;
    			return null;
    		}

    		ArrayList pred = eval.predictions();
    		pos = new Vector();
    		neg = new Vector();
    		String clsInfo = Utils.toCommandLine(actualClassifier).replace(" ", "_");
    		pos.add(clsInfo);
    		neg.add(clsInfo);
    		for(int i = 0; i < pred.size(); i++){
    			NominalPrediction pred4Instance = (NominalPrediction)pred.get(i);
    			double prob = (pred4Instance.distribution())[0];
    			if( pred4Instance.actual() == 0.0){
    				pos.add(prob);
    			}
    			else{
    				neg.add(prob);
    			}
    		}

    		if(iteration == 0){
    			for(int i = 0; i < pos.size(); i++){
    				System.out.println(">score for positiv instance "+i+"\t"+pos.elementAt(i));
    			}
    			System.out.println("\n\n");
    			for(int i = 0; i < neg.size(); i++){
    				System.out.println(">score for negative instance "+i+"\t"+neg.elementAt(i));
    			}
    		}
    		System.out.println("Iterative Evaluation for Iteration: "+(iteration+1));
    		System.out.println(eval.toSummaryString());
    		System.out.println(eval.toClassDetailsString()+"\n\n");

    		for(int i = 0; i < folds; i++){
    			converged = converged && convergedInAllFolds.get(i);
    		}
    		if(stopAfterIteration != -1 && iteration >= (stopAfterIteration - 1)){
    			converged = true;
    			System.out.println("Maximum number of "+stopAfterIteration+" iteration(s) reached.");
    		}
    		long timeEnd = System.currentTimeMillis();
    		time += (timeEnd - timeStart);

    		return new EvaluationStep(eval, pos, neg, (timeEnd - timeStart));
    	}
    	catch (Exception e) {
    		e.printStackTrace();
    		return null;
    	}
    }

    /**
     * Does nothing.
     */
    public void remove() {
    }

    /**
     * Returns the overall time used for execution.
     *
     * @return the execution time
     */
    public long getTime() {
    	return time;
    }

    /**
     * Returns the scores of the positive examples.
     * First element is the classifier name.
     *
     * @return the scores
     */
    public Vector getPositiveScores() {
    	return pos;
    }

    /**
     * Returns the scores of the negative examples.
     * First element is the classifier name.
     *
     * @return the scores
     */
    public Vector getNegativeScores() {
    	return neg;
    }

    /**
     * ???
     */
    public int getIteration() {
    	return iteration;
    }

    /**
     * Stops the processing at the next earliest time.
     */
    public void stop() {
    	stopping = true;
    }

    /**
     * Returns whether the computation is about to stop.
     *
     * @return true if the computation is about to stop
     */
    public boolean isStopping() {
    	return stopping;
    }

    /**
     * Returns whether the computation has completely stopped.
     *
     * @return true if the computation has stopped
     */
    public boolean isStopped() {
    	return stopped;
    }
	}

	private T m_classifier;
	private int m_folds;
	private Instances trainingSet;
	private int m_iterationSteps;
	private boolean saveFolds;
	private int upperBoundIterations;
	public String classifierName;
  protected AbstractOutput output;

	public AbstractIterativeEvaluation(){
		saveFolds = false;
		m_iterationSteps = 1;
		m_folds = 10;
		upperBoundIterations = -1;
		output = new Null();
	}

	public void setOptions(String[] options) throws Exception {
		String		tmpStr;
		String[]	tmpOptions;

		tmpStr = Utils.getOption('F', options);
		if (tmpStr.length() != 0)
			m_folds = Integer.parseInt(tmpStr);
		else
			m_folds = 10;

		tmpStr = Utils.getOption('I', options);
		if (tmpStr.length() != 0)
			upperBoundIterations = Integer.parseInt(tmpStr);
		else
			upperBoundIterations = -1;
		if(upperBoundIterations <= 0)
			upperBoundIterations = -1;

		saveFolds = Utils.getFlag('M', options);

		tmpStr = Utils.getOption('S', options);
		if (tmpStr.length() != 0)
			m_iterationSteps = Integer.parseInt(tmpStr);
		else
			m_iterationSteps = 1;

		tmpStr = Utils.getOption('T', options);
		trainingSet = ConverterUtils.DataSource.read(tmpStr);
		trainingSet.setClassIndex(trainingSet.numAttributes() - 1);

		tmpStr = Utils.getOption("output", options);
		if (tmpStr.length() > 0) {
			tmpOptions    = Utils.splitOptions(tmpStr);
			tmpStr        = tmpOptions[0];
			tmpOptions[0] = "";
			output        = (AbstractOutput) Utils.forName(AbstractOutput.class, tmpStr, tmpOptions);
		}
		else {
			output = new Null();
		}

		classifierName = Utils.getOption('W', options);
		if (classifierName.length() > 0) {
			setClassifier((T) AbstractClassifier.forName(classifierName, null));
			setClassifier((T) AbstractClassifier.forName(classifierName, Utils.partitionOptions(options)));
		}
		else {
			throw new Exception("no classifier given");
		}
	}

	/**
	 * Set the base learner.
	 *
	 * @param newClassifier the classifier to use.
	 */
	public void setClassifier(T newClassifier) {
		m_classifier = newClassifier;
	}

	/**
	 * Get the classifier used as the base learner.
	 *
	 * @return the classifier used as the classifier
	 */
	public T getClassifier() {
		return m_classifier;
	}

	/**
	 * Returns an iterator object that allows to iterate till convergence.
	 *
	 * @return the iterator instance
	 */
	public EvaluationIterator iterator() {
		return new EvaluationIterator(this);
	}

	public int getFolds() {
		return m_folds;
	}

	public void setFolds(int folds) {
		this.m_folds = folds;
	}

	public Instances getTrainingSet() {
		return trainingSet;
	}

	public void setTrainingSet(Instances trainingSet) {
		this.trainingSet = trainingSet;
	}

	public void setIterationSteps(int value) {
		m_iterationSteps = value;
	}

	public int getIterationSteps() {
		return m_iterationSteps;
	}

	public boolean isSaveFolds() {
		return saveFolds;
	}

	public void setSaveFolds(boolean saveFolds) {
		this.saveFolds = saveFolds;
	}

	public int getUpperBoundIterations() {
		return upperBoundIterations;
	}

	public void setUpperBoundIterations(int upperBoundIterations) {
		this.upperBoundIterations = upperBoundIterations;
	}

	/**
	 * Sets the object for retrieving the prediction output.
	 *
	 * @param value the prediction output receiver
	 */
	public void setOutput(AbstractOutput value) {
		output = value;
	}

	/**
	 * Returns the object for storing the prediction output.
	 *
	 * @return the prediction output receiver
	 */
	public AbstractOutput getOutput() {
		return output;
	}

	public static void evaluate(AbstractIterativeEvaluation itEval, String[] args) throws Exception {
		itEval.setOptions(args);
		EvaluationIterator iter = itEval.iterator();
		while (iter.hasNext())
			iter.next();

		System.out.println("Time elapsed in milliseconds: "+(iter.getTime()));
		long hours, minutes, seconds;
		hours = (iter.getTime())/3600000;
		long remainder = (iter.getTime()) - (hours*3600000);
		minutes = remainder/60000;
		remainder = remainder - (minutes*60000);
		seconds = remainder/1000;
		System.out.println("Time elapsed: "+hours+"h "+minutes+"min "+seconds+"sec");

		if(iter.getIteration() != 0){
			for(int i = 0; i < iter.getPositiveScores().size(); i++){
				System.out.println(">>score for positiv instance "+i+"\t"+iter.getPositiveScores().elementAt(i));
			}
			System.out.println("\n\n");
			for(int i = 0; i < iter.getNegativeScores().size(); i++){
				System.out.println(">>score for negative instance "+i+"\t"+iter.getNegativeScores().elementAt(i));
			}
		}
	}
}
