/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.blob;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.ServerSocket;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.io.FileUtils;
import org.apache.flink.api.common.JobID;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.runtime.blob.BlobKey;
import org.apache.flink.runtime.blob.BlobServerConnection;
import org.apache.flink.runtime.blob.BlobService;
import org.apache.flink.runtime.blob.BlobUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlobServer
extends Thread
implements BlobService {
    private static final Logger LOG = LoggerFactory.getLogger(BlobServer.class);
    private final AtomicInteger tempFileCounter = new AtomicInteger(0);
    private final ServerSocket serverSocket;
    private final AtomicBoolean shutdownRequested = new AtomicBoolean();
    private final Thread shutdownHook;
    private final File storageDir;
    private final Set<BlobServerConnection> activeConnections = new HashSet<BlobServerConnection>();
    private final int maxConnections;

    public BlobServer(Configuration config) throws IOException {
        String storageDirectory = config.getString("blob.storage.directory", null);
        this.storageDir = BlobUtils.initStorageDirectory(storageDirectory);
        LOG.info("Created BLOB server storage directory {}", (Object)this.storageDir);
        int maxConnections = config.getInteger("blob.fetch.num-concurrent", 50);
        if (maxConnections >= 1) {
            this.maxConnections = maxConnections;
        } else {
            LOG.warn("Invalid value for maximum connections in BLOB server: {}. Using default value of {}", (Object)maxConnections, (Object)50);
            this.maxConnections = 50;
        }
        int backlog = config.getInteger("blob.fetch.backlog", 1000);
        if (backlog < 1) {
            LOG.warn("Invalid value for BLOB connection backlog: {}. Using default value of {}", (Object)backlog, (Object)1000);
            backlog = 1000;
        }
        this.shutdownHook = BlobUtils.addShutdownHook(this, LOG);
        try {
            this.serverSocket = new ServerSocket(0, backlog);
        }
        catch (IOException e) {
            throw new IOException("Could not create BlobServer with automatic port choice.", e);
        }
        this.setName("BLOB Server listener at " + this.getPort());
        this.setDaemon(true);
        this.start();
        if (LOG.isInfoEnabled()) {
            LOG.info("Started BLOB server at {}:{} - max concurrent requests: {} - max backlog: {}", new Object[]{this.serverSocket.getInetAddress().getHostAddress(), this.getPort(), maxConnections, backlog});
        }
    }

    public File getStorageLocation(BlobKey key) {
        return BlobUtils.getStorageLocation(this.storageDir, key);
    }

    public File getStorageLocation(JobID jobID, String key) {
        return BlobUtils.getStorageLocation(this.storageDir, jobID, key);
    }

    public void deleteJobDirectory(JobID jobID) throws IOException {
        BlobUtils.deleteJobDirectory(this.storageDir, jobID);
    }

    File createTemporaryFilename() {
        return new File(BlobUtils.getIncomingDirectory(this.storageDir), String.format("temp-%08d", this.tempFileCounter.getAndIncrement()));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public void run() {
        try {
            while (!this.shutdownRequested.get()) {
                Set<BlobServerConnection> set;
                BlobServerConnection conn = new BlobServerConnection(this.serverSocket.accept(), this);
                try {
                    set = this.activeConnections;
                    synchronized (set) {
                        while (this.activeConnections.size() >= this.maxConnections) {
                            this.activeConnections.wait(2000L);
                        }
                        this.activeConnections.add(conn);
                    }
                    conn.start();
                    conn = null;
                }
                finally {
                    if (conn == null) continue;
                    conn.close();
                    set = this.activeConnections;
                    synchronized (set) {
                        this.activeConnections.remove(conn);
                    }
                }
            }
            return;
        }
        catch (Throwable t) {
            if (this.shutdownRequested.get()) return;
            LOG.error("BLOB server stopped working. Shutting down", t);
            this.shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void shutdown() {
        if (this.shutdownRequested.compareAndSet(false, true)) {
            try {
                this.serverSocket.close();
            }
            catch (IOException ioe) {
                LOG.debug("Error while closing the server socket.", (Throwable)ioe);
            }
            this.interrupt();
            try {
                this.join();
            }
            catch (InterruptedException ie) {
                LOG.debug("Error while waiting for this thread to die.", (Throwable)ie);
            }
            Set<BlobServerConnection> ie = this.activeConnections;
            synchronized (ie) {
                if (!this.activeConnections.isEmpty()) {
                    for (BlobServerConnection conn : this.activeConnections) {
                        LOG.debug("Shutting down connection " + conn.getName());
                        conn.close();
                    }
                    this.activeConnections.clear();
                }
            }
            try {
                FileUtils.deleteDirectory((File)this.storageDir);
            }
            catch (IOException e) {
                LOG.error("BLOB server failed to properly clean up its storage directory.");
            }
            if (this.shutdownHook != null && this.shutdownHook != Thread.currentThread()) {
                try {
                    Runtime.getRuntime().removeShutdownHook(this.shutdownHook);
                }
                catch (IllegalStateException e) {
                }
                catch (Throwable t) {
                    LOG.warn("Exception while unregistering BLOB server's cleanup shutdown hook.");
                }
            }
        }
    }

    @Override
    public URL getURL(BlobKey requiredBlob) throws IOException {
        if (requiredBlob == null) {
            throw new IllegalArgumentException("Required BLOB cannot be null.");
        }
        File localFile = BlobUtils.getStorageLocation(this.storageDir, requiredBlob);
        if (!localFile.exists()) {
            throw new FileNotFoundException("File " + localFile.getCanonicalPath() + " does not exist.");
        }
        return localFile.toURI().toURL();
    }

    @Override
    public void delete(BlobKey key) throws IOException {
        File localFile = BlobUtils.getStorageLocation(this.storageDir, key);
        if (localFile.exists() && !localFile.delete()) {
            LOG.warn("Failed to delete locally BLOB " + key + " at " + localFile.getAbsolutePath());
        }
    }

    @Override
    public int getPort() {
        return this.serverSocket.getLocalPort();
    }

    public boolean isShutdown() {
        return this.shutdownRequested.get();
    }

    ServerSocket getServerSocket() {
        return this.serverSocket;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void unregisterConnection(BlobServerConnection conn) {
        Set<BlobServerConnection> set = this.activeConnections;
        synchronized (set) {
            this.activeConnections.remove(conn);
            this.activeConnections.notifyAll();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    List<BlobServerConnection> getCurrentActiveConnections() {
        Set<BlobServerConnection> set = this.activeConnections;
        synchronized (set) {
            return new ArrayList<BlobServerConnection>(this.activeConnections);
        }
    }
}

