/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.checkpoint;

import akka.actor.ActorRef;
import akka.actor.ActorSystem;
import akka.actor.PoisonPill;
import akka.actor.Props;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.flink.api.common.JobID;
import org.apache.flink.runtime.checkpoint.CheckpointCoordinatorDeActivator;
import org.apache.flink.runtime.checkpoint.PendingCheckpoint;
import org.apache.flink.runtime.checkpoint.StateForTask;
import org.apache.flink.runtime.checkpoint.SuccessfulCheckpoint;
import org.apache.flink.runtime.executiongraph.Execution;
import org.apache.flink.runtime.executiongraph.ExecutionAttemptID;
import org.apache.flink.runtime.executiongraph.ExecutionJobVertex;
import org.apache.flink.runtime.executiongraph.ExecutionVertex;
import org.apache.flink.runtime.jobgraph.JobVertexID;
import org.apache.flink.runtime.messages.checkpoint.AcknowledgeCheckpoint;
import org.apache.flink.runtime.messages.checkpoint.ConfirmCheckpoint;
import org.apache.flink.runtime.messages.checkpoint.TriggerCheckpoint;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CheckpointCoordinator {
    private static final Logger LOG = LoggerFactory.getLogger(CheckpointCoordinator.class);
    private static final int NUM_GHOST_CHECKPOINT_IDS = 16;
    private final Object lock = new Object();
    private final JobID job;
    private final ExecutionVertex[] tasksToTrigger;
    private final ExecutionVertex[] tasksToWaitFor;
    private final ExecutionVertex[] tasksToCommitTo;
    private final Map<Long, PendingCheckpoint> pendingCheckpoints;
    private final ArrayDeque<SuccessfulCheckpoint> completedCheckpoints;
    private final ArrayDeque<Long> recentPendingCheckpoints;
    private final AtomicLong checkpointIdCounter = new AtomicLong(1L);
    private final AtomicInteger numUnsuccessfulCheckpointsTriggers = new AtomicInteger();
    private final Timer timer;
    private final long checkpointTimeout;
    private final int numSuccessfulCheckpointsToRetain;
    private TimerTask periodicScheduler;
    private ActorRef jobStatusListener;
    private ClassLoader userClassLoader;
    private boolean shutdown;

    public CheckpointCoordinator(JobID job, int numSuccessfulCheckpointsToRetain, long checkpointTimeout, ExecutionVertex[] tasksToTrigger, ExecutionVertex[] tasksToWaitFor, ExecutionVertex[] tasksToCommitTo, ClassLoader userClassLoader) {
        if (job == null || tasksToTrigger == null || tasksToWaitFor == null || tasksToCommitTo == null) {
            throw new NullPointerException();
        }
        if (numSuccessfulCheckpointsToRetain < 1) {
            throw new IllegalArgumentException("Must retain at least one successful checkpoint");
        }
        if (checkpointTimeout < 1L) {
            throw new IllegalArgumentException("Checkpoint timeout must be larger than zero");
        }
        this.job = job;
        this.numSuccessfulCheckpointsToRetain = numSuccessfulCheckpointsToRetain;
        this.checkpointTimeout = checkpointTimeout;
        this.tasksToTrigger = tasksToTrigger;
        this.tasksToWaitFor = tasksToWaitFor;
        this.tasksToCommitTo = tasksToCommitTo;
        this.pendingCheckpoints = new LinkedHashMap<Long, PendingCheckpoint>();
        this.completedCheckpoints = new ArrayDeque(numSuccessfulCheckpointsToRetain + 1);
        this.recentPendingCheckpoints = new ArrayDeque(16);
        this.userClassLoader = userClassLoader;
        this.timer = new Timer("Checkpoint Timer", true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        Object object = this.lock;
        synchronized (object) {
            if (this.shutdown) {
                return;
            }
            this.shutdown = true;
            LOG.info("Stopping checkpoint coordinator for job " + this.job);
            this.timer.cancel();
            if (this.jobStatusListener != null) {
                this.jobStatusListener.tell((Object)PoisonPill.getInstance(), ActorRef.noSender());
                this.jobStatusListener = null;
            }
            if (this.periodicScheduler != null) {
                this.periodicScheduler.cancel();
                this.periodicScheduler = null;
            }
            for (PendingCheckpoint pending : this.pendingCheckpoints.values()) {
                pending.discard(this.userClassLoader, true);
            }
            this.pendingCheckpoints.clear();
            for (SuccessfulCheckpoint checkpoint : this.completedCheckpoints) {
                checkpoint.discard(this.userClassLoader);
            }
            this.completedCheckpoints.clear();
        }
    }

    public boolean isShutdown() {
        return this.shutdown;
    }

    public void triggerCheckpoint() {
        this.triggerCheckpoint(System.currentTimeMillis());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean triggerCheckpoint(long timestamp) {
        if (this.shutdown) {
            LOG.error("Cannot trigger checkpoint, checkpoint coordinator has been shutdown.");
            return false;
        }
        final long checkpointID = this.checkpointIdCounter.getAndIncrement();
        LOG.info("Triggering checkpoint " + checkpointID + " @ " + timestamp);
        try {
            ExecutionAttemptID[] triggerIDs = new ExecutionAttemptID[this.tasksToTrigger.length];
            for (int i = 0; i < this.tasksToTrigger.length; ++i) {
                ExecutionVertex[] ee = this.tasksToTrigger[i].getCurrentExecutionAttempt();
                if (ee == null) {
                    LOG.info("Checkpoint triggering task {} is not being executed at the moment. Aborting checkpoint.", (Object)this.tasksToTrigger[i].getSimpleName());
                    return false;
                }
                triggerIDs[i] = ee.getAttemptId();
            }
            HashMap<ExecutionAttemptID, ExecutionVertex> ackTasks = new HashMap<ExecutionAttemptID, ExecutionVertex>(this.tasksToWaitFor.length);
            for (ExecutionVertex ev : this.tasksToWaitFor) {
                Execution ee = ev.getCurrentExecutionAttempt();
                if (ee == null) {
                    LOG.info("Checkpoint acknowledging task {} is not being executed at the moment. Aborting checkpoint.", (Object)ev.getSimpleName());
                    return false;
                }
                ackTasks.put(ee.getAttemptId(), ev);
            }
            final PendingCheckpoint checkpoint = new PendingCheckpoint(this.job, checkpointID, timestamp, ackTasks);
            TimerTask canceller = new TimerTask(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void run() {
                    try {
                        Object object = CheckpointCoordinator.this.lock;
                        synchronized (object) {
                            if (!checkpoint.isDiscarded()) {
                                LOG.info("Checkpoint " + checkpointID + " expired before completing.");
                                checkpoint.discard(CheckpointCoordinator.this.userClassLoader, true);
                                CheckpointCoordinator.this.pendingCheckpoints.remove(checkpointID);
                                CheckpointCoordinator.this.rememberRecentCheckpointId(checkpointID);
                            }
                        }
                    }
                    catch (Throwable t) {
                        LOG.error("Exception while handling checkpoint timeout", t);
                    }
                }
            };
            Object object = this.lock;
            synchronized (object) {
                if (this.shutdown) {
                    throw new IllegalStateException("Checkpoint coordinator has been shutdown.");
                }
                this.pendingCheckpoints.put(checkpointID, checkpoint);
                this.timer.schedule(canceller, this.checkpointTimeout);
            }
            for (int i = 0; i < this.tasksToTrigger.length; ++i) {
                ExecutionAttemptID id = triggerIDs[i];
                TriggerCheckpoint message = new TriggerCheckpoint(this.job, id, checkpointID, timestamp);
                this.tasksToTrigger[i].sendMessageToCurrentExecution(message, id);
            }
            this.numUnsuccessfulCheckpointsTriggers.set(0);
            return true;
        }
        catch (Throwable t) {
            int numUnsuccessful = this.numUnsuccessfulCheckpointsTriggers.incrementAndGet();
            LOG.warn("Failed to trigger checkpoint (" + numUnsuccessful + " consecutive failed attempts so far)", t);
            Object object = this.lock;
            synchronized (object) {
                PendingCheckpoint checkpoint = this.pendingCheckpoints.remove(checkpointID);
                if (checkpoint != null && !checkpoint.isDiscarded()) {
                    checkpoint.discard(this.userClassLoader, true);
                }
            }
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void receiveAcknowledgeMessage(AcknowledgeCheckpoint message) {
        if (this.shutdown || message == null) {
            return;
        }
        if (!this.job.equals((Object)message.getJob())) {
            LOG.error("Received AcknowledgeCheckpoint message for wrong job: {}", (Object)message);
            return;
        }
        long checkpointId = message.getCheckpointId();
        SuccessfulCheckpoint completed = null;
        Object object = this.lock;
        synchronized (object) {
            if (this.shutdown) {
                return;
            }
            PendingCheckpoint checkpoint = this.pendingCheckpoints.get(checkpointId);
            if (checkpoint != null && !checkpoint.isDiscarded()) {
                if (checkpoint.acknowledgeTask(message.getTaskExecutionId(), message.getState())) {
                    if (checkpoint.isFullyAcknowledged()) {
                        LOG.info("Completed checkpoint " + checkpointId);
                        completed = checkpoint.toCompletedCheckpoint();
                        this.completedCheckpoints.addLast(completed);
                        if (this.completedCheckpoints.size() > this.numSuccessfulCheckpointsToRetain) {
                            this.completedCheckpoints.removeFirst().discard(this.userClassLoader);
                        }
                        this.pendingCheckpoints.remove(checkpointId);
                        this.rememberRecentCheckpointId(checkpointId);
                        this.dropSubsumedCheckpoints(completed.getTimestamp());
                    }
                } else {
                    LOG.error("Received duplicate or invalid acknowledge message for checkpoint " + checkpointId + " , task " + (Object)((Object)message.getTaskExecutionId()));
                }
            } else {
                if (checkpoint != null) {
                    throw new IllegalStateException("Received message for discarded but non-removed checkpoint " + checkpointId);
                }
                if (this.recentPendingCheckpoints.contains(checkpointId)) {
                    LOG.warn("Received late message for now expired checkpoint attempt " + checkpointId);
                } else {
                    LOG.info("Received message for non-existing checkpoint " + checkpointId);
                }
            }
        }
        if (completed != null) {
            long timestamp = completed.getTimestamp();
            for (ExecutionVertex ev : this.tasksToCommitTo) {
                Execution ee = ev.getCurrentExecutionAttempt();
                if (ee == null) continue;
                ExecutionAttemptID attemptId = ee.getAttemptId();
                ConfirmCheckpoint confirmMessage = new ConfirmCheckpoint(this.job, attemptId, checkpointId, timestamp);
                ev.sendMessageToCurrentExecution(confirmMessage, ee.getAttemptId());
            }
        }
    }

    private void rememberRecentCheckpointId(long id) {
        if (this.recentPendingCheckpoints.size() >= 16) {
            this.recentPendingCheckpoints.removeFirst();
        }
        this.recentPendingCheckpoints.addLast(id);
    }

    private void dropSubsumedCheckpoints(long timestamp) {
        Iterator<Map.Entry<Long, PendingCheckpoint>> entries = this.pendingCheckpoints.entrySet().iterator();
        while (entries.hasNext()) {
            PendingCheckpoint p = entries.next().getValue();
            if (p.getCheckpointTimestamp() >= timestamp) continue;
            this.rememberRecentCheckpointId(p.getCheckpointId());
            p.discard(this.userClassLoader, true);
            entries.remove();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void restoreLatestCheckpointedState(Map<JobVertexID, ExecutionJobVertex> tasks, boolean errorIfNoCheckpoint, boolean allOrNothingState) throws Exception {
        Object object = this.lock;
        synchronized (object) {
            if (this.shutdown) {
                throw new IllegalStateException("CheckpointCoordinator is shut down");
            }
            if (this.completedCheckpoints.isEmpty()) {
                if (errorIfNoCheckpoint) {
                    throw new IllegalStateException("No completed checkpoint available");
                }
                return;
            }
            SuccessfulCheckpoint latest = this.completedCheckpoints.getLast();
            if (allOrNothingState) {
                ExecutionJobVertex vertex;
                HashMap<ExecutionJobVertex, Integer> stateCounts = new HashMap<ExecutionJobVertex, Integer>();
                for (StateForTask stateForTask : latest.getStates()) {
                    vertex = tasks.get((Object)stateForTask.getOperatorId());
                    Execution exec = vertex.getTaskVertices()[stateForTask.getSubtask()].getCurrentExecutionAttempt();
                    exec.setInitialState(stateForTask.getState());
                    Integer count = (Integer)stateCounts.get(vertex);
                    if (count != null) {
                        stateCounts.put(vertex, count + 1);
                        continue;
                    }
                    stateCounts.put(vertex, 1);
                }
                for (Map.Entry entry : stateCounts.entrySet()) {
                    vertex = (ExecutionJobVertex)entry.getKey();
                    if (((Integer)entry.getValue()).intValue() == vertex.getParallelism()) continue;
                    throw new IllegalStateException("The checkpoint contained state only for a subset of tasks for vertex " + vertex);
                }
            } else {
                for (StateForTask state : latest.getStates()) {
                    ExecutionJobVertex executionJobVertex = tasks.get((Object)state.getOperatorId());
                    Execution exec = executionJobVertex.getTaskVertices()[state.getSubtask()].getCurrentExecutionAttempt();
                    exec.setInitialState(state.getState());
                }
            }
        }
    }

    public int getNumberOfPendingCheckpoints() {
        return this.pendingCheckpoints.size();
    }

    public int getNumberOfRetainedSuccessfulCheckpoints() {
        return this.completedCheckpoints.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Map<Long, PendingCheckpoint> getPendingCheckpoints() {
        Object object = this.lock;
        synchronized (object) {
            return new HashMap<Long, PendingCheckpoint>(this.pendingCheckpoints);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<SuccessfulCheckpoint> getSuccessfulCheckpoints() {
        Object object = this.lock;
        synchronized (object) {
            return new ArrayList<SuccessfulCheckpoint>(this.completedCheckpoints);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void startPeriodicCheckpointScheduler(long interval) {
        Object object = this.lock;
        synchronized (object) {
            if (this.shutdown) {
                throw new IllegalArgumentException("Checkpoint coordinator is shut down");
            }
            this.stopPeriodicCheckpointScheduler();
            this.periodicScheduler = new TimerTask(){

                @Override
                public void run() {
                    try {
                        CheckpointCoordinator.this.triggerCheckpoint();
                    }
                    catch (Exception e) {
                        LOG.error("Exception while triggering checkpoint", (Throwable)e);
                    }
                }
            };
            this.timer.scheduleAtFixedRate(this.periodicScheduler, interval, interval);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stopPeriodicCheckpointScheduler() {
        Object object = this.lock;
        synchronized (object) {
            if (this.periodicScheduler != null) {
                this.periodicScheduler.cancel();
                this.periodicScheduler = null;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ActorRef createJobStatusListener(ActorSystem actorSystem, long checkpointInterval) {
        Object object = this.lock;
        synchronized (object) {
            if (this.shutdown) {
                throw new IllegalArgumentException("Checkpoint coordinator is shut down");
            }
            if (this.jobStatusListener == null) {
                Props props = Props.create(CheckpointCoordinatorDeActivator.class, (Object[])new Object[]{this, checkpointInterval});
                this.jobStatusListener = actorSystem.actorOf(props);
            }
            return this.jobStatusListener;
        }
    }
}

