/*
 * Copyright (c) 2011-2013, Peter Abeles. All Rights Reserved.
 *
 * This file is part of BoofCV (http://boofcv.org).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package boofcv.alg.geo.f;

import org.ejml.data.DenseMatrix64F;

/**
 * Helper class for {@link EssentialNister5}.  Contains ugly unrolled linear systems.  See generate/python/nister5.py
 * for code which created the mess you see below.
 *
 * @author Peter Abeles
 */
public class HelperNister5 {

	// Null vectors which describe E = x*W + y*Y + z*Z + W
	double X00,X01,X02,X10,X11,X12,X20,X21,X22;
	double Y00,Y01,Y02,Y10,Y11,Y12,Y20,Y21,Y22;
	double Z00,Z01,Z02,Z10,Z11,Z12,Z20,Z21,Z22;
	double W00,W01,W02,W10,W11,W12,W20,W21,W22;

	// Vectors used to define polynomial
	double K00,K01,K02,K03,K04,K05,K06,K07,K08,K09,K10,K11,K12;
	double L00,L01,L02,L03,L04,L05,L06,L07,L08,L09,L10,L11,L12;
	double M00,M01,M02,M03,M04,M05,M06,M07,M08,M09,M10,M11,M12;

	public void setNullSpace( double x[], double y[], double z[] , double w[] ) {
		X00=x[0];X01=x[1];X02=x[2]; X10=x[3];X11=x[4];X12=x[5]; X20=x[6];X21=x[7];X22=x[8];
		Y00=y[0];Y01=y[1];Y02=y[2]; Y10=y[3];Y11=y[4];Y12=y[5]; Y20=y[6];Y21=y[7];Y22=y[8];
		Z00=z[0];Z01=z[1];Z02=z[2]; Z10=z[3];Z11=z[4];Z12=z[5]; Z20=z[6];Z21=z[7];Z22=z[8];
		W00=w[0];W01=w[1];W02=w[2]; W10=w[3];W11=w[4];W12=w[5]; W20=w[6];W21=w[7];W22=w[8];
	}

	/**
	 * Construct 10 by 10 coeffient matrix for the following coefficients
	 *
	 * 'x^3','y^3','x^2*y','x*y^2','x^2*z','x^2','y^2*z','y^2','x*y*z','x*y'
	 *
	 * @param A a 10 by 10 matrix
	 */
	public void setupA1(DenseMatrix64F A) {
		A.data[0] = X20*( X01*X12 - X02*X11 ) + X10*( -X01*X22 + X02*X21 ) + X00*( X11*X22 - X12*X21 );
		A.data[1] = Y02*( Y10*Y21 - Y11*Y20 ) + Y00*( Y11*Y22 - Y12*Y21 ) + Y01*( -Y10*Y22 + Y12*Y20 );
		A.data[2] = X22*( X00*Y11 - X01*Y10 - X10*Y01 + X11*Y00 ) + X20*( X01*Y12 - X02*Y11 - X11*Y02 + X12*Y01 ) + X21*( -X00*Y12 + X02*Y10 + X10*Y02 - X12*Y00 ) + X01*( -X10*Y22 + X12*Y20 ) + Y21*( -X00*X12 + X02*X10 ) + X11*( X00*Y22 - X02*Y20 );
		A.data[3] = Y02*( X10*Y21 - X11*Y20 - X20*Y11 + X21*Y10 ) + Y01*( -X10*Y22 + X12*Y20 + X20*Y12 - X22*Y10 ) + Y00*( X11*Y22 - X12*Y21 - X21*Y12 + X22*Y11 ) + Y10*( -X01*Y22 + X02*Y21 ) + Y20*( X01*Y12 - X02*Y11 ) + X00*( Y11*Y22 - Y12*Y21 );
		A.data[4] = X20*( X01*Z12 - X02*Z11 - X11*Z02 + X12*Z01 ) + X10*( -X01*Z22 + X02*Z21 + X21*Z02 - X22*Z01 ) + X00*( X11*Z22 - X12*Z21 - X21*Z12 + X22*Z11 ) + Z20*( X01*X12 - X02*X11 ) + Z00*( X11*X22 - X12*X21 ) + Z10*( -X01*X22 + X02*X21 );
		A.data[5] = X01*( -W10*X22 + W12*X20 + W20*X12 - W22*X10 ) + X02*( W10*X21 - W11*X20 - W20*X11 + W21*X10 ) + X00*( W11*X22 - W12*X21 - W21*X12 + W22*X11 ) + W00*( X11*X22 - X12*X21 ) + W01*( -X10*X22 + X12*X20 ) + W02*( X10*X21 - X11*X20 );
		A.data[6] = Y02*( Y10*Z21 - Y11*Z20 - Y20*Z11 + Y21*Z10 ) + Y00*( Y11*Z22 - Y12*Z21 - Y21*Z12 + Y22*Z11 ) + Y01*( -Y10*Z22 + Y12*Z20 + Y20*Z12 - Y22*Z10 ) + Z01*( -Y10*Y22 + Y12*Y20 ) + Z00*( Y11*Y22 - Y12*Y21 ) + Z02*( Y10*Y21 - Y11*Y20 );
		A.data[7] = Y02*( W10*Y21 - W11*Y20 - W20*Y11 + W21*Y10 ) + Y00*( W11*Y22 - W12*Y21 - W21*Y12 + W22*Y11 ) + Y01*( -W10*Y22 + W12*Y20 + W20*Y12 - W22*Y10 ) + W00*( Y11*Y22 - Y12*Y21 ) + W01*( -Y10*Y22 + Y12*Y20 ) + W02*( Y10*Y21 - Y11*Y20 );
		A.data[8] = Y02*( X10*Z21 - X11*Z20 - X20*Z11 + X21*Z10 ) + Y00*( X11*Z22 - X12*Z21 - X21*Z12 + X22*Z11 ) + Y01*( -X10*Z22 + X12*Z20 + X20*Z12 - X22*Z10 ) + Z01*( -X10*Y22 + X12*Y20 + X20*Y12 - X22*Y10 ) + Z00*( X11*Y22 - X12*Y21 - X21*Y12 + X22*Y11 ) + Z02*( X10*Y21 - X11*Y20 - X20*Y11 + X21*Y10 ) + X02*( Y10*Z21 - Y11*Z20 - Y20*Z11 + Y21*Z10 ) + X00*( Y11*Z22 - Y12*Z21 - Y21*Z12 + Y22*Z11 ) + X01*( -Y10*Z22 + Y12*Z20 + Y20*Z12 - Y22*Z10 );
		A.data[9] = Y02*( W10*X21 - W11*X20 - W20*X11 + W21*X10 ) + Y00*( W11*X22 - W12*X21 - W21*X12 + W22*X11 ) + Y22*( W00*X11 - W01*X10 - W10*X01 + W11*X00 ) + Y01*( -W10*X22 + W12*X20 + W20*X12 - W22*X10 ) + Y20*( W01*X12 - W02*X11 - W11*X02 + W12*X01 ) + Y11*( W00*X22 - W02*X20 - W20*X02 + W22*X00 ) + Y10*( -W01*X22 + W02*X21 + W21*X02 - W22*X01 ) + Y21*( -W00*X12 + W02*X10 + W10*X02 - W12*X00 ) + Y12*( -W00*X21 + W01*X20 + W20*X01 - W21*X00 );
		A.data[10] = X00*( 0.5*X00*X00 + 0.5*X01*X01 + 0.5*X02*X02 + 0.5*X10*X10 - 0.5*X11*X11 - 0.5*X12*X12 + 0.5*X20*X20 - 0.5*X21*X21 - 0.5*X22*X22 ) + X02*( X10*X12 + X20*X22 ) + X01*( X10*X11 + X20*X21 );
		A.data[11] = Y00*( 0.5*Y00*Y00 + 0.5*Y01*Y01 + 0.5*Y02*Y02 + 0.5*Y10*Y10 - 0.5*Y11*Y11 - 0.5*Y12*Y12 + 0.5*Y20*Y20 - 0.5*Y21*Y21 - 0.5*Y22*Y22 ) + Y02*( Y10*Y12 + Y20*Y22 ) + Y01*( Y10*Y11 + Y20*Y21 );
		A.data[12] = X00*( 1.5*X00*Y00 + X01*Y01 + X02*Y02 + X10*Y10 - X11*Y11 - X12*Y12 + X20*Y20 - X21*Y21 - X22*Y22 ) + Y00*( 0.5*X01*X01 + 0.5*X02*X02 + 0.5*X10*X10 - 0.5*X11*X11 - 0.5*X12*X12 + 0.5*X20*X20 - 0.5*X21*X21 - 0.5*X22*X22 ) + X20*( X01*Y21 + X02*Y22 + X21*Y01 + X22*Y02 ) + X10*( X01*Y11 + X02*Y12 + X11*Y01 + X12*Y02 ) + X02*( X12*Y10 + X22*Y20 ) + X01*( X11*Y10 + X21*Y20 );
		A.data[13] = Y00*( 1.5*X00*Y00 + X01*Y01 + X02*Y02 + X10*Y10 - X11*Y11 - X12*Y12 + X20*Y20 - X21*Y21 - X22*Y22 ) + 0.5*( X00*( Y01*Y01 + Y02*Y02 + Y10*Y10 - Y11*Y11 - Y12*Y12 + Y20*Y20 - Y21*Y21 - Y22*Y22 ) ) + Y02*( X10*Y12 + X12*Y10 + X20*Y22 + X22*Y20 ) + Y01*( X10*Y11 + X11*Y10 + X20*Y21 + X21*Y20 ) + X02*( Y10*Y12 + Y20*Y22 ) + X01*( Y10*Y11 + Y20*Y21 );
		A.data[14] = X00*( 1.5*X00*Z00 + X01*Z01 + X02*Z02 + X10*Z10 - X11*Z11 - X12*Z12 + X20*Z20 - X21*Z21 - X22*Z22 ) + Z00*( 0.5*X01*X01 + 0.5*X02*X02 + 0.5*X10*X10 - 0.5*X11*X11 - 0.5*X12*X12 + 0.5*X20*X20 - 0.5*X21*X21 - 0.5*X22*X22 ) + X02*( X10*Z12 + X12*Z10 + X20*Z22 + X22*Z20 ) + X01*( X10*Z11 + X11*Z10 + X20*Z21 + X21*Z20 ) + Z01*( X10*X11 + X20*X21 ) + Z02*( X10*X12 + X20*X22 );
		A.data[15] = X00*( 1.5*W00*X00 + W01*X01 + W02*X02 + W10*X10 - W11*X11 - W12*X12 + W20*X20 - W21*X21 - W22*X22 ) + 0.5*( W00*( X01*X01 + X02*X02 + X10*X10 - X11*X11 - X12*X12 + X20*X20 - X21*X21 - X22*X22 ) ) + X01*( W10*X11 + W11*X10 + W20*X21 + W21*X20 ) + X02*( W10*X12 + W12*X10 + W20*X22 + W22*X20 ) + W01*( X10*X11 + X20*X21 ) + W02*( X10*X12 + X20*X22 );
		A.data[16] = Y00*( 1.5*Y00*Z00 + Y01*Z01 + Y02*Z02 + Y10*Z10 - Y11*Z11 - Y12*Z12 + Y20*Z20 - Y21*Z21 - Y22*Z22 ) + Z00*( 0.5*Y01*Y01 + 0.5*Y02*Y02 + 0.5*Y10*Y10 - 0.5*Y11*Y11 - 0.5*Y12*Y12 + 0.5*Y20*Y20 - 0.5*Y21*Y21 - 0.5*Y22*Y22 ) + Y02*( Y10*Z12 + Y12*Z10 + Y20*Z22 + Y22*Z20 ) + Y01*( Y10*Z11 + Y11*Z10 + Y20*Z21 + Y21*Z20 ) + Z01*( Y10*Y11 + Y20*Y21 ) + Z02*( Y10*Y12 + Y20*Y22 );
		A.data[17] = Y00*( 1.5*W00*Y00 + W01*Y01 + W02*Y02 + W10*Y10 - W11*Y11 - W12*Y12 + W20*Y20 - W21*Y21 - W22*Y22 ) + 0.5*( W00*( Y01*Y01 + Y02*Y02 + Y10*Y10 - Y11*Y11 - Y12*Y12 + Y20*Y20 - Y21*Y21 - Y22*Y22 ) ) + Y02*( W10*Y12 + W12*Y10 + W20*Y22 + W22*Y20 ) + Y01*( W10*Y11 + W11*Y10 + W20*Y21 + W21*Y20 ) + W01*( Y10*Y11 + Y20*Y21 ) + W02*( Y10*Y12 + Y20*Y22 );
		A.data[18] = Y00*( 3.0*X00*Z00 + X01*Z01 + X02*Z02 + X10*Z10 - X11*Z11 - X12*Z12 + X20*Z20 - X21*Z21 - X22*Z22 ) + Z00*( X01*Y01 + X02*Y02 + X10*Y10 - X11*Y11 - X12*Y12 + X20*Y20 - X21*Y21 - X22*Y22 ) + X00*( Y01*Z01 + Y02*Z02 + Y10*Z10 - Y11*Z11 - Y12*Z12 + Y20*Z20 - Y21*Z21 - Y22*Z22 ) + Y02*( X10*Z12 + X12*Z10 + X20*Z22 + X22*Z20 ) + Y01*( X10*Z11 + X11*Z10 + X20*Z21 + X21*Z20 ) + Z01*( X10*Y11 + X11*Y10 + X20*Y21 + X21*Y20 ) + Z02*( X10*Y12 + X12*Y10 + X20*Y22 + X22*Y20 ) + X02*( Y10*Z12 + Y12*Z10 + Y20*Z22 + Y22*Z20 ) + X01*( Y10*Z11 + Y11*Z10 + Y20*Z21 + Y21*Z20 );
		A.data[19] = Y00*( 3.0*W00*X00 + W01*X01 + W02*X02 + W10*X10 - W11*X11 - W12*X12 + W20*X20 - W21*X21 - W22*X22 ) + X00*( W01*Y01 + W02*Y02 + W10*Y10 - W11*Y11 - W12*Y12 + W20*Y20 - W21*Y21 - W22*Y22 ) + W00*( X01*Y01 + X02*Y02 + X10*Y10 - X11*Y11 - X12*Y12 + X20*Y20 - X21*Y21 - X22*Y22 ) + Y02*( W10*X12 + W12*X10 + W20*X22 + W22*X20 ) + Y01*( W10*X11 + W11*X10 + W20*X21 + W21*X20 ) + Y20*( W01*X21 + W02*X22 + W21*X01 + W22*X02 ) + Y10*( W01*X11 + W02*X12 + W11*X01 + W12*X02 ) + W01*( X10*Y11 + X20*Y21 ) + W02*( X10*Y12 + X20*Y22 ) + W10*( X01*Y11 + X02*Y12 ) + W20*( X01*Y21 + X02*Y22 );
		A.data[20] = X01*( 0.5*X00*X00 + 0.5*X01*X01 + 0.5*X02*X02 - 0.5*X10*X10 + 0.5*X11*X11 - 0.5*X12*X12 - 0.5*X20*X20 + 0.5*X21*X21 - 0.5*X22*X22 ) + X02*( X11*X12 + X21*X22 ) + X00*( X10*X11 + X20*X21 );
		A.data[21] = Y01*( 0.5*Y00*Y00 + 0.5*Y01*Y01 + 0.5*Y02*Y02 - 0.5*Y10*Y10 + 0.5*Y11*Y11 - 0.5*Y12*Y12 - 0.5*Y20*Y20 + 0.5*Y21*Y21 - 0.5*Y22*Y22 ) + Y02*( Y11*Y12 + Y21*Y22 ) + Y00*( Y10*Y11 + Y20*Y21 );
		A.data[22] = X01*( X00*Y00 + 1.5*X01*Y01 + X02*Y02 - X10*Y10 + X11*Y11 - X12*Y12 - X20*Y20 + X21*Y21 - X22*Y22 ) + Y01*( 0.5*X00*X00 + 0.5*X02*X02 - 0.5*X10*X10 + 0.5*X11*X11 - 0.5*X12*X12 - 0.5*X20*X20 + 0.5*X21*X21 - 0.5*X22*X22 ) + X21*( X00*Y20 + X02*Y22 + X20*Y00 + X22*Y02 ) + X11*( X00*Y10 + X02*Y12 + X10*Y00 + X12*Y02 ) + X02*( X12*Y11 + X22*Y21 ) + X00*( X10*Y11 + X20*Y21 );
		A.data[23] = Y01*( X00*Y00 + 1.5*X01*Y01 + X02*Y02 - X10*Y10 + X11*Y11 - X12*Y12 - X20*Y20 + X21*Y21 - X22*Y22 ) + 0.5*( X01*( Y00*Y00 + Y02*Y02 - Y10*Y10 + Y11*Y11 - Y12*Y12 - Y20*Y20 + Y21*Y21 - Y22*Y22 ) ) + Y02*( X11*Y12 + X12*Y11 + X21*Y22 + X22*Y21 ) + Y00*( X10*Y11 + X11*Y10 + X20*Y21 + X21*Y20 ) + X02*( Y11*Y12 + Y21*Y22 ) + X00*( Y10*Y11 + Y20*Y21 );
		A.data[24] = X01*( X00*Z00 + 1.5*X01*Z01 + X02*Z02 - X10*Z10 + X11*Z11 - X12*Z12 - X20*Z20 + X21*Z21 - X22*Z22 ) + Z01*( 0.5*X00*X00 + 0.5*X02*X02 - 0.5*X10*X10 + 0.5*X11*X11 - 0.5*X12*X12 - 0.5*X20*X20 + 0.5*X21*X21 - 0.5*X22*X22 ) + X02*( X11*Z12 + X12*Z11 + X21*Z22 + X22*Z21 ) + X00*( X10*Z11 + X11*Z10 + X20*Z21 + X21*Z20 ) + Z00*( X10*X11 + X20*X21 ) + Z02*( X11*X12 + X21*X22 );
		A.data[25] = X01*( W00*X00 + 1.5*W01*X01 + W02*X02 - W10*X10 + W11*X11 - W12*X12 - W20*X20 + W21*X21 - W22*X22 ) + 0.5*( W01*( X00*X00 + X02*X02 - X10*X10 + X11*X11 - X12*X12 - X20*X20 + X21*X21 - X22*X22 ) ) + X02*( W11*X12 + W12*X11 + W21*X22 + W22*X21 ) + X00*( W10*X11 + W11*X10 + W20*X21 + W21*X20 ) + W00*( X10*X11 + X20*X21 ) + W02*( X11*X12 + X21*X22 );
		A.data[26] = Y01*( Y00*Z00 + 1.5*Y01*Z01 + Y02*Z02 - Y10*Z10 + Y11*Z11 - Y12*Z12 - Y20*Z20 + Y21*Z21 - Y22*Z22 ) + Z01*( 0.5*Y00*Y00 + 0.5*Y02*Y02 - 0.5*Y10*Y10 + 0.5*Y11*Y11 - 0.5*Y12*Y12 - 0.5*Y20*Y20 + 0.5*Y21*Y21 - 0.5*Y22*Y22 ) + Y02*( Y11*Z12 + Y12*Z11 + Y21*Z22 + Y22*Z21 ) + Y00*( Y10*Z11 + Y11*Z10 + Y20*Z21 + Y21*Z20 ) + Z00*( Y10*Y11 + Y20*Y21 ) + Z02*( Y11*Y12 + Y21*Y22 );
		A.data[27] = Y01*( W00*Y00 + 1.5*W01*Y01 + W02*Y02 - W10*Y10 + W11*Y11 - W12*Y12 - W20*Y20 + W21*Y21 - W22*Y22 ) + 0.5*( W01*( Y00*Y00 + Y02*Y02 - Y10*Y10 + Y11*Y11 - Y12*Y12 - Y20*Y20 + Y21*Y21 - Y22*Y22 ) ) + Y02*( W11*Y12 + W12*Y11 + W21*Y22 + W22*Y21 ) + Y00*( W10*Y11 + W11*Y10 + W20*Y21 + W21*Y20 ) + W00*( Y10*Y11 + Y20*Y21 ) + W02*( Y11*Y12 + Y21*Y22 );
		A.data[28] = Y01*( X00*Z00 + 3.0*X01*Z01 + X02*Z02 - X10*Z10 + X11*Z11 - X12*Z12 - X20*Z20 + X21*Z21 - X22*Z22 ) + Z01*( X00*Y00 + X02*Y02 - X10*Y10 + X11*Y11 - X12*Y12 - X20*Y20 + X21*Y21 - X22*Y22 ) + X01*( Y00*Z00 + Y02*Z02 - Y10*Z10 + Y11*Z11 - Y12*Z12 - Y20*Z20 + Y21*Z21 - Y22*Z22 ) + Y02*( X11*Z12 + X12*Z11 + X21*Z22 + X22*Z21 ) + Y00*( X10*Z11 + X11*Z10 + X20*Z21 + X21*Z20 ) + Z00*( X10*Y11 + X11*Y10 + X20*Y21 + X21*Y20 ) + Z02*( X11*Y12 + X12*Y11 + X21*Y22 + X22*Y21 ) + X02*( Y11*Z12 + Y12*Z11 + Y21*Z22 + Y22*Z21 ) + X00*( Y10*Z11 + Y11*Z10 + Y20*Z21 + Y21*Z20 );
		A.data[29] = Y01*( W00*X00 + 3.0*W01*X01 + W02*X02 - W10*X10 + W11*X11 - W12*X12 - W20*X20 + W21*X21 - W22*X22 ) + W01*( X00*Y00 + X02*Y02 - X10*Y10 + X11*Y11 - X12*Y12 - X20*Y20 + X21*Y21 - X22*Y22 ) + X01*( W00*Y00 + W02*Y02 - W10*Y10 + W11*Y11 - W12*Y12 - W20*Y20 + W21*Y21 - W22*Y22 ) + Y02*( W11*X12 + W12*X11 + W21*X22 + W22*X21 ) + Y00*( W10*X11 + W11*X10 + W20*X21 + W21*X20 ) + X02*( W11*Y12 + W12*Y11 + W21*Y22 + W22*Y21 ) + X00*( W10*Y11 + W11*Y10 + W20*Y21 + W21*Y20 ) + W00*( X10*Y11 + X11*Y10 + X20*Y21 + X21*Y20 ) + W02*( X11*Y12 + X12*Y11 + X21*Y22 + X22*Y21 );
		A.data[30] = X02*( 0.5*X00*X00 + 0.5*X01*X01 + 0.5*X02*X02 - 0.5*X10*X10 - 0.5*X11*X11 + 0.5*X12*X12 - 0.5*X20*X20 - 0.5*X21*X21 + 0.5*X22*X22 ) + X01*( X11*X12 + X21*X22 ) + X00*( X10*X12 + X20*X22 );
		A.data[31] = Y02*( 0.5*Y00*Y00 + 0.5*Y01*Y01 + 0.5*Y02*Y02 - 0.5*Y10*Y10 - 0.5*Y11*Y11 + 0.5*Y12*Y12 - 0.5*Y20*Y20 - 0.5*Y21*Y21 + 0.5*Y22*Y22 ) + Y00*( Y10*Y12 + Y20*Y22 ) + Y01*( Y11*Y12 + Y21*Y22 );
		A.data[32] = X02*( X00*Y00 + X01*Y01 + 1.5*X02*Y02 - X10*Y10 - X11*Y11 + X12*Y12 - X20*Y20 - X21*Y21 + X22*Y22 ) + Y02*( 0.5*X00*X00 + 0.5*X01*X01 - 0.5*X10*X10 - 0.5*X11*X11 + 0.5*X12*X12 - 0.5*X20*X20 - 0.5*X21*X21 + 0.5*X22*X22 ) + X22*( X00*Y20 + X01*Y21 + X20*Y00 + X21*Y01 ) + X12*( X00*Y10 + X01*Y11 + X10*Y00 + X11*Y01 ) + X01*( X11*Y12 + X21*Y22 ) + X00*( X10*Y12 + X20*Y22 );
		A.data[33] = Y02*( X00*Y00 + X01*Y01 + 1.5*X02*Y02 - X10*Y10 - X11*Y11 + X12*Y12 - X20*Y20 - X21*Y21 + X22*Y22 ) + 0.5*( X02*( Y00*Y00 + Y01*Y01 - Y10*Y10 - Y11*Y11 + Y12*Y12 - Y20*Y20 - Y21*Y21 + Y22*Y22 ) ) + Y00*( X10*Y12 + X12*Y10 + X20*Y22 + X22*Y20 ) + Y01*( X11*Y12 + X12*Y11 + X21*Y22 + X22*Y21 ) + X01*( Y11*Y12 + Y21*Y22 ) + X00*( Y10*Y12 + Y20*Y22 );
		A.data[34] = X02*( X00*Z00 + X01*Z01 + 1.5*X02*Z02 - X10*Z10 - X11*Z11 + X12*Z12 - X20*Z20 - X21*Z21 + X22*Z22 ) + Z02*( 0.5*X00*X00 + 0.5*X01*X01 - 0.5*X10*X10 - 0.5*X11*X11 + 0.5*X12*X12 - 0.5*X20*X20 - 0.5*X21*X21 + 0.5*X22*X22 ) + X01*( X11*Z12 + X12*Z11 + X21*Z22 + X22*Z21 ) + X00*( X10*Z12 + X12*Z10 + X20*Z22 + X22*Z20 ) + Z01*( X11*X12 + X21*X22 ) + Z00*( X10*X12 + X20*X22 );
		A.data[35] = X02*( W00*X00 + W01*X01 + 1.5*W02*X02 - W10*X10 - W11*X11 + W12*X12 - W20*X20 - W21*X21 + W22*X22 ) + 0.5*( W02*( X00*X00 + X01*X01 - X10*X10 - X11*X11 + X12*X12 - X20*X20 - X21*X21 + X22*X22 ) ) + X01*( W11*X12 + W12*X11 + W21*X22 + W22*X21 ) + X00*( W10*X12 + W12*X10 + W20*X22 + W22*X20 ) + W00*( X10*X12 + X20*X22 ) + W01*( X11*X12 + X21*X22 );
		A.data[36] = Y02*( Y00*Z00 + Y01*Z01 + 1.5*Y02*Z02 - Y10*Z10 - Y11*Z11 + Y12*Z12 - Y20*Z20 - Y21*Z21 + Y22*Z22 ) + 0.5*( Z02*( Y00*Y00 + Y01*Y01 - Y10*Y10 - Y11*Y11 + Y12*Y12 - Y20*Y20 - Y21*Y21 + Y22*Y22 ) ) + Y00*( Y10*Z12 + Y12*Z10 + Y20*Z22 + Y22*Z20 ) + Y01*( Y11*Z12 + Y12*Z11 + Y21*Z22 + Y22*Z21 ) + Z01*( Y11*Y12 + Y21*Y22 ) + Z00*( Y10*Y12 + Y20*Y22 );
		A.data[37] = Y02*( W00*Y00 + W01*Y01 + 1.5*W02*Y02 - W10*Y10 - W11*Y11 + W12*Y12 - W20*Y20 - W21*Y21 + W22*Y22 ) + 0.5*( W02*( Y00*Y00 + Y01*Y01 - Y10*Y10 - Y11*Y11 + Y12*Y12 - Y20*Y20 - Y21*Y21 + Y22*Y22 ) ) + Y00*( W10*Y12 + W12*Y10 + W20*Y22 + W22*Y20 ) + Y01*( W11*Y12 + W12*Y11 + W21*Y22 + W22*Y21 ) + W00*( Y10*Y12 + Y20*Y22 ) + W01*( Y11*Y12 + Y21*Y22 );
		A.data[38] = Y02*( X00*Z00 + X01*Z01 + 3.0*X02*Z02 - X10*Z10 - X11*Z11 + X12*Z12 - X20*Z20 - X21*Z21 + X22*Z22 ) + Z02*( X00*Y00 + X01*Y01 - X10*Y10 - X11*Y11 + X12*Y12 - X20*Y20 - X21*Y21 + X22*Y22 ) + X02*( Y00*Z00 + Y01*Z01 - Y10*Z10 - Y11*Z11 + Y12*Z12 - Y20*Z20 - Y21*Z21 + Y22*Z22 ) + Y00*( X10*Z12 + X12*Z10 + X20*Z22 + X22*Z20 ) + Y22*( X00*Z20 + X01*Z21 + X20*Z00 + X21*Z01 ) + Y01*( X11*Z12 + X12*Z11 + X21*Z22 + X22*Z21 ) + Y12*( X00*Z10 + X01*Z11 + X10*Z00 + X11*Z01 ) + X12*( Y10*Z00 + Y11*Z01 ) + Z22*( X00*Y20 + X01*Y21 ) + Z12*( X00*Y10 + X01*Y11 ) + X22*( Y20*Z00 + Y21*Z01 );
		A.data[39] = Y02*( W00*X00 + W01*X01 + 3.0*W02*X02 - W10*X10 - W11*X11 + W12*X12 - W20*X20 - W21*X21 + W22*X22 ) + X02*( W00*Y00 + W01*Y01 - W10*Y10 - W11*Y11 + W12*Y12 - W20*Y20 - W21*Y21 + W22*Y22 ) + W02*( X00*Y00 + X01*Y01 - X10*Y10 - X11*Y11 + X12*Y12 - X20*Y20 - X21*Y21 + X22*Y22 ) + Y00*( W10*X12 + W12*X10 + W20*X22 + W22*X20 ) + Y01*( W11*X12 + W12*X11 + W21*X22 + W22*X21 ) + Y22*( W00*X20 + W01*X21 + W20*X00 + W21*X01 ) + Y12*( W00*X10 + W01*X11 + W10*X00 + W11*X01 ) + W00*( X12*Y10 + X22*Y20 ) + W01*( X12*Y11 + X22*Y21 ) + W12*( X00*Y10 + X01*Y11 ) + W22*( X00*Y20 + X01*Y21 );
		A.data[40] = X10*( 0.5*X00*X00 - 0.5*X01*X01 - 0.5*X02*X02 + 0.5*X10*X10 + 0.5*X11*X11 + 0.5*X12*X12 + 0.5*X20*X20 - 0.5*X21*X21 - 0.5*X22*X22 ) + X12*( X00*X02 + X20*X22 ) + X11*( X00*X01 + X20*X21 );
		A.data[41] = Y10*( 0.5*Y00*Y00 - 0.5*Y01*Y01 - 0.5*Y02*Y02 + 0.5*Y10*Y10 + 0.5*Y11*Y11 + 0.5*Y12*Y12 + 0.5*Y20*Y20 - 0.5*Y21*Y21 - 0.5*Y22*Y22 ) + Y00*( Y01*Y11 + Y02*Y12 ) + Y20*( Y11*Y21 + Y12*Y22 );
		A.data[42] = X10*( X00*Y00 - X01*Y01 - X02*Y02 + 1.5*X10*Y10 + X11*Y11 + X12*Y12 + X20*Y20 - X21*Y21 - X22*Y22 ) + 0.5*( Y10*( X00*X00 - X01*X01 - X02*X02 + X11*X11 + X12*X12 + X20*X20 - X21*X21 - X22*X22 ) ) + X20*( X11*Y21 + X12*Y22 + X21*Y11 + X22*Y12 ) + X00*( X01*Y11 + X02*Y12 + X11*Y01 + X12*Y02 ) + Y00*( X01*X11 + X02*X12 ) + Y20*( X11*X21 + X12*X22 );
		A.data[43] = Y10*( X00*Y00 - X01*Y01 - X02*Y02 + 1.5*X10*Y10 + X11*Y11 + X12*Y12 + X20*Y20 - X21*Y21 - X22*Y22 ) + 0.5*( X10*( Y00*Y00 - Y01*Y01 - Y02*Y02 + Y11*Y11 + Y12*Y12 + Y20*Y20 - Y21*Y21 - Y22*Y22 ) ) + Y00*( X01*Y11 + X02*Y12 + X11*Y01 + X12*Y02 ) + Y20*( X11*Y21 + X12*Y22 + X21*Y11 + X22*Y12 ) + X00*( Y01*Y11 + Y02*Y12 ) + X20*( Y11*Y21 + Y12*Y22 );
		A.data[44] = X10*( X00*Z00 - X01*Z01 - X02*Z02 + 1.5*X10*Z10 + X11*Z11 + X12*Z12 + X20*Z20 - X21*Z21 - X22*Z22 ) + 0.5*( Z10*( X00*X00 - X01*X01 - X02*X02 + X11*X11 + X12*X12 + X20*X20 - X21*X21 - X22*X22 ) ) + X20*( X11*Z21 + X12*Z22 + X21*Z11 + X22*Z12 ) + X00*( X01*Z11 + X02*Z12 + X11*Z01 + X12*Z02 ) + Z20*( X11*X21 + X12*X22 ) + Z00*( X01*X11 + X02*X12 );
		A.data[45] = X10*( W00*X00 - W01*X01 - W02*X02 + 1.5*W10*X10 + W11*X11 + W12*X12 + W20*X20 - W21*X21 - W22*X22 ) + 0.5*( W10*( X00*X00 - X01*X01 - X02*X02 + X11*X11 + X12*X12 + X20*X20 - X21*X21 - X22*X22 ) ) + X20*( W11*X21 + W12*X22 + W21*X11 + W22*X12 ) + X00*( W01*X11 + W02*X12 + W11*X01 + W12*X02 ) + W00*( X01*X11 + X02*X12 ) + W20*( X11*X21 + X12*X22 );
		A.data[46] = Y10*( Y00*Z00 - Y01*Z01 - Y02*Z02 + 1.5*Y10*Z10 + Y11*Z11 + Y12*Z12 + Y20*Z20 - Y21*Z21 - Y22*Z22 ) + 0.5*( Z10*( Y00*Y00 - Y01*Y01 - Y02*Y02 + Y11*Y11 + Y12*Y12 + Y20*Y20 - Y21*Y21 - Y22*Y22 ) ) + Y00*( Y01*Z11 + Y02*Z12 + Y11*Z01 + Y12*Z02 ) + Y20*( Y11*Z21 + Y12*Z22 + Y21*Z11 + Y22*Z12 ) + Z20*( Y11*Y21 + Y12*Y22 ) + Z00*( Y01*Y11 + Y02*Y12 );
		A.data[47] = Y10*( W00*Y00 - W01*Y01 - W02*Y02 + 1.5*W10*Y10 + W11*Y11 + W12*Y12 + W20*Y20 - W21*Y21 - W22*Y22 ) + 0.5*( W10*( Y00*Y00 - Y01*Y01 - Y02*Y02 + Y11*Y11 + Y12*Y12 + Y20*Y20 - Y21*Y21 - Y22*Y22 ) ) + Y00*( W01*Y11 + W02*Y12 + W11*Y01 + W12*Y02 ) + Y20*( W11*Y21 + W12*Y22 + W21*Y11 + W22*Y12 ) + W00*( Y01*Y11 + Y02*Y12 ) + W20*( Y11*Y21 + Y12*Y22 );
		A.data[48] = Z10*( X00*Y00 - X01*Y01 - X02*Y02 + 3.0*X10*Y10 + X11*Y11 + X12*Y12 + X20*Y20 - X21*Y21 - X22*Y22 ) + X10*( Y00*Z00 - Y01*Z01 - Y02*Z02 + Y11*Z11 + Y12*Z12 + Y20*Z20 - Y21*Z21 - Y22*Z22 ) + Y10*( X00*Z00 - X01*Z01 - X02*Z02 + X11*Z11 + X12*Z12 + X20*Z20 - X21*Z21 - X22*Z22 ) + Y00*( X01*Z11 + X02*Z12 + X11*Z01 + X12*Z02 ) + Z20*( X11*Y21 + X12*Y22 + X21*Y11 + X22*Y12 ) + Z00*( X01*Y11 + X02*Y12 + X11*Y01 + X12*Y02 ) + Y20*( X11*Z21 + X12*Z22 + X21*Z11 + X22*Z12 ) + X00*( Y01*Z11 + Y02*Z12 + Y11*Z01 + Y12*Z02 ) + X20*( Y11*Z21 + Y12*Z22 + Y21*Z11 + Y22*Z12 );
		A.data[49] = X10*( W00*Y00 - W01*Y01 - W02*Y02 + 3.0*W10*Y10 + W11*Y11 + W12*Y12 + W20*Y20 - W21*Y21 - W22*Y22 ) + Y10*( W00*X00 - W01*X01 - W02*X02 + W11*X11 + W12*X12 + W20*X20 - W21*X21 - W22*X22 ) + W10*( X00*Y00 - X01*Y01 - X02*Y02 + X11*Y11 + X12*Y12 + X20*Y20 - X21*Y21 - X22*Y22 ) + Y00*( W01*X11 + W02*X12 + W11*X01 + W12*X02 ) + W20*( X11*Y21 + X12*Y22 + X21*Y11 + X22*Y12 ) + Y20*( W11*X21 + W12*X22 + W21*X11 + W22*X12 ) + W00*( X01*Y11 + X02*Y12 + X11*Y01 + X12*Y02 ) + X00*( W01*Y11 + W02*Y12 + W11*Y01 + W12*Y02 ) + X20*( W11*Y21 + W12*Y22 + W21*Y11 + W22*Y12 );
		A.data[50] = X11*( -0.5*X00*X00 + 0.5*X01*X01 - 0.5*X02*X02 + 0.5*X10*X10 + 0.5*X11*X11 + 0.5*X12*X12 - 0.5*X20*X20 + 0.5*X21*X21 - 0.5*X22*X22 ) + X10*( X00*X01 + X20*X21 ) + X12*( X01*X02 + X21*X22 );
		A.data[51] = Y11*( -0.5*Y00*Y00 + 0.5*Y01*Y01 - 0.5*Y02*Y02 + 0.5*Y10*Y10 + 0.5*Y11*Y11 + 0.5*Y12*Y12 - 0.5*Y20*Y20 + 0.5*Y21*Y21 - 0.5*Y22*Y22 ) + Y01*( Y00*Y10 + Y02*Y12 ) + Y21*( Y10*Y20 + Y12*Y22 );
		A.data[52] = X11*( -X00*Y00 + X01*Y01 - X02*Y02 + X10*Y10 + 1.5*X11*Y11 + X12*Y12 - X20*Y20 + X21*Y21 - X22*Y22 ) + 0.5*( Y11*( -X00*X00 + X01*X01 - X02*X02 + X10*X10 + X12*X12 - X20*X20 + X21*X21 - X22*X22 ) ) + X12*( X01*Y02 + X02*Y01 + X21*Y22 + X22*Y21 ) + X10*( X00*Y01 + X01*Y00 + X20*Y21 + X21*Y20 ) + Y10*( X00*X01 + X20*X21 ) + Y12*( X01*X02 + X21*X22 );
		A.data[53] = Y11*( -X00*Y00 + X01*Y01 - X02*Y02 + X10*Y10 + 1.5*X11*Y11 + X12*Y12 - X20*Y20 + X21*Y21 - X22*Y22 ) + X11*( -0.5*Y00*Y00 + 0.5*Y01*Y01 - 0.5*Y02*Y02 + 0.5*Y10*Y10 + 0.5*Y12*Y12 - 0.5*Y20*Y20 + 0.5*Y21*Y21 - 0.5*Y22*Y22 ) + Y01*( X00*Y10 + X02*Y12 + X10*Y00 + X12*Y02 ) + Y21*( X10*Y20 + X12*Y22 + X20*Y10 + X22*Y12 ) + X01*( Y00*Y10 + Y02*Y12 ) + X21*( Y10*Y20 + Y12*Y22 );
		A.data[54] = X11*( -X00*Z00 + X01*Z01 - X02*Z02 + X10*Z10 + 1.5*X11*Z11 + X12*Z12 - X20*Z20 + X21*Z21 - X22*Z22 ) + 0.5*( Z11*( -X00*X00 + X01*X01 - X02*X02 + X10*X10 + X12*X12 - X20*X20 + X21*X21 - X22*X22 ) ) + X10*( X00*Z01 + X01*Z00 + X20*Z21 + X21*Z20 ) + X12*( X01*Z02 + X02*Z01 + X21*Z22 + X22*Z21 ) + Z12*( X01*X02 + X21*X22 ) + Z10*( X00*X01 + X20*X21 );
		A.data[55] = X11*( -W00*X00 + W01*X01 - W02*X02 + W10*X10 + 1.5*W11*X11 + W12*X12 - W20*X20 + W21*X21 - W22*X22 ) + 0.5*( W11*( -X00*X00 + X01*X01 - X02*X02 + X10*X10 + X12*X12 - X20*X20 + X21*X21 - X22*X22 ) ) + X10*( W00*X01 + W01*X00 + W20*X21 + W21*X20 ) + X12*( W01*X02 + W02*X01 + W21*X22 + W22*X21 ) + W12*( X01*X02 + X21*X22 ) + W10*( X00*X01 + X20*X21 );
		A.data[56] = Y11*( -Y00*Z00 + Y01*Z01 - Y02*Z02 + Y10*Z10 + 1.5*Y11*Z11 + Y12*Z12 - Y20*Z20 + Y21*Z21 - Y22*Z22 ) + 0.5*( Z11*( -Y00*Y00 + Y01*Y01 - Y02*Y02 + Y10*Y10 + Y12*Y12 - Y20*Y20 + Y21*Y21 - Y22*Y22 ) ) + Y01*( Y00*Z10 + Y02*Z12 + Y10*Z00 + Y12*Z02 ) + Y21*( Y10*Z20 + Y12*Z22 + Y20*Z10 + Y22*Z12 ) + Z21*( Y10*Y20 + Y12*Y22 ) + Z01*( Y00*Y10 + Y02*Y12 );
		A.data[57] = Y11*( -W00*Y00 + W01*Y01 - W02*Y02 + W10*Y10 + 1.5*W11*Y11 + W12*Y12 - W20*Y20 + W21*Y21 - W22*Y22 ) + 0.5*( W11*( -Y00*Y00 + Y01*Y01 - Y02*Y02 + Y10*Y10 + Y12*Y12 - Y20*Y20 + Y21*Y21 - Y22*Y22 ) ) + Y01*( W00*Y10 + W02*Y12 + W10*Y00 + W12*Y02 ) + Y21*( W10*Y20 + W12*Y22 + W20*Y10 + W22*Y12 ) + W01*( Y00*Y10 + Y02*Y12 ) + W21*( Y10*Y20 + Y12*Y22 );
		A.data[58] = Z11*( -X00*Y00 + X01*Y01 - X02*Y02 + X10*Y10 + 3.0*X11*Y11 + X12*Y12 - X20*Y20 + X21*Y21 - X22*Y22 ) + X11*( -Y00*Z00 + Y01*Z01 - Y02*Z02 + Y10*Z10 + Y12*Z12 - Y20*Z20 + Y21*Z21 - Y22*Z22 ) + Y11*( -X00*Z00 + X01*Z01 - X02*Z02 + X10*Z10 + X12*Z12 - X20*Z20 + X21*Z21 - X22*Z22 ) + Y01*( X00*Z10 + X02*Z12 + X10*Z00 + X12*Z02 ) + Z21*( X10*Y20 + X12*Y22 + X20*Y10 + X22*Y12 ) + Z01*( X00*Y10 + X02*Y12 + X10*Y00 + X12*Y02 ) + Y21*( X10*Z20 + X12*Z22 + X20*Z10 + X22*Z12 ) + X01*( Y00*Z10 + Y02*Z12 + Y10*Z00 + Y12*Z02 ) + X21*( Y10*Z20 + Y12*Z22 + Y20*Z10 + Y22*Z12 );
		A.data[59] = X11*( -W00*Y00 + W01*Y01 - W02*Y02 + W10*Y10 + 3.0*W11*Y11 + W12*Y12 - W20*Y20 + W21*Y21 - W22*Y22 ) + Y11*( -W00*X00 + W01*X01 - W02*X02 + W10*X10 + W12*X12 - W20*X20 + W21*X21 - W22*X22 ) + W11*( -X00*Y00 + X01*Y01 - X02*Y02 + X10*Y10 + X12*Y12 - X20*Y20 + X21*Y21 - X22*Y22 ) + Y01*( W00*X10 + W02*X12 + W10*X00 + W12*X02 ) + W21*( X10*Y20 + X12*Y22 + X20*Y10 + X22*Y12 ) + Y21*( W10*X20 + W12*X22 + W20*X10 + W22*X12 ) + W01*( X00*Y10 + X02*Y12 + X10*Y00 + X12*Y02 ) + X01*( W00*Y10 + W02*Y12 + W10*Y00 + W12*Y02 ) + X21*( W10*Y20 + W12*Y22 + W20*Y10 + W22*Y12 );
		A.data[60] = X12*( -0.5*X00*X00 - 0.5*X01*X01 + 0.5*X02*X02 + 0.5*X10*X10 + 0.5*X11*X11 + 0.5*X12*X12 - 0.5*X20*X20 - 0.5*X21*X21 + 0.5*X22*X22 ) + X11*( X01*X02 + X21*X22 ) + X10*( X00*X02 + X20*X22 );
		A.data[61] = Y12*( -0.5*Y00*Y00 - 0.5*Y01*Y01 + 0.5*Y02*Y02 + 0.5*Y10*Y10 + 0.5*Y11*Y11 + 0.5*Y12*Y12 - 0.5*Y20*Y20 - 0.5*Y21*Y21 + 0.5*Y22*Y22 ) + Y02*( Y00*Y10 + Y01*Y11 ) + Y22*( Y10*Y20 + Y11*Y21 );
		A.data[62] = X12*( -X00*Y00 - X01*Y01 + X02*Y02 + X10*Y10 + X11*Y11 + 1.5*X12*Y12 - X20*Y20 - X21*Y21 + X22*Y22 ) + 0.5*( Y12*( -X00*X00 - X01*X01 + X02*X02 + X10*X10 + X11*X11 - X20*X20 - X21*X21 + X22*X22 ) ) + X11*( X01*Y02 + X02*Y01 + X21*Y22 + X22*Y21 ) + X10*( X00*Y02 + X02*Y00 + X20*Y22 + X22*Y20 ) + X02*( X00*Y10 + X01*Y11 ) + X22*( X20*Y10 + X21*Y11 );
		A.data[63] = Y12*( -X00*Y00 - X01*Y01 + X02*Y02 + X10*Y10 + X11*Y11 + 1.5*X12*Y12 - X20*Y20 - X21*Y21 + X22*Y22 ) + 0.5*( X12*( -Y00*Y00 - Y01*Y01 + Y02*Y02 + Y10*Y10 + Y11*Y11 - Y20*Y20 - Y21*Y21 + Y22*Y22 ) ) + Y02*( X00*Y10 + X01*Y11 + X10*Y00 + X11*Y01 ) + Y22*( X10*Y20 + X11*Y21 + X20*Y10 + X21*Y11 ) + X02*( Y00*Y10 + Y01*Y11 ) + X22*( Y10*Y20 + Y11*Y21 );
		A.data[64] = X12*( -X00*Z00 - X01*Z01 + X02*Z02 + X10*Z10 + X11*Z11 + 1.5*X12*Z12 - X20*Z20 - X21*Z21 + X22*Z22 ) + 0.5*( Z12*( -X00*X00 - X01*X01 + X02*X02 + X10*X10 + X11*X11 - X20*X20 - X21*X21 + X22*X22 ) ) + X11*( X01*Z02 + X02*Z01 + X21*Z22 + X22*Z21 ) + X10*( X00*Z02 + X02*Z00 + X20*Z22 + X22*Z20 ) + Z10*( X00*X02 + X20*X22 ) + Z11*( X01*X02 + X21*X22 );
		A.data[65] = X12*( -W00*X00 - W01*X01 + W02*X02 + W10*X10 + W11*X11 + 1.5*W12*X12 - W20*X20 - W21*X21 + W22*X22 ) + 0.5*( W12*( -X00*X00 - X01*X01 + X02*X02 + X10*X10 + X11*X11 - X20*X20 - X21*X21 + X22*X22 ) ) + X11*( W01*X02 + W02*X01 + W21*X22 + W22*X21 ) + X10*( W00*X02 + W02*X00 + W20*X22 + W22*X20 ) + W11*( X01*X02 + X21*X22 ) + W10*( X00*X02 + X20*X22 );
		A.data[66] = Y12*( -Y00*Z00 - Y01*Z01 + Y02*Z02 + Y10*Z10 + Y11*Z11 + 1.5*Y12*Z12 - Y20*Z20 - Y21*Z21 + Y22*Z22 ) + 0.5*( Z12*( -Y00*Y00 - Y01*Y01 + Y02*Y02 + Y10*Y10 + Y11*Y11 - Y20*Y20 - Y21*Y21 + Y22*Y22 ) ) + Y02*( Y00*Z10 + Y01*Z11 + Y10*Z00 + Y11*Z01 ) + Y22*( Y10*Z20 + Y11*Z21 + Y20*Z10 + Y21*Z11 ) + Z22*( Y10*Y20 + Y11*Y21 ) + Z02*( Y00*Y10 + Y01*Y11 );
		A.data[67] = Y12*( -W00*Y00 - W01*Y01 + W02*Y02 + W10*Y10 + W11*Y11 + 1.5*W12*Y12 - W20*Y20 - W21*Y21 + W22*Y22 ) + 0.5*( W12*( -Y00*Y00 - Y01*Y01 + Y02*Y02 + Y10*Y10 + Y11*Y11 - Y20*Y20 - Y21*Y21 + Y22*Y22 ) ) + Y02*( W00*Y10 + W01*Y11 + W10*Y00 + W11*Y01 ) + Y22*( W10*Y20 + W11*Y21 + W20*Y10 + W21*Y11 ) + W02*( Y00*Y10 + Y01*Y11 ) + W22*( Y10*Y20 + Y11*Y21 );
		A.data[68] = Z12*( -X00*Y00 - X01*Y01 + X02*Y02 + X10*Y10 + X11*Y11 + 3.0*X12*Y12 - X20*Y20 - X21*Y21 + X22*Y22 ) + X12*( -Y00*Z00 - Y01*Z01 + Y02*Z02 + Y10*Z10 + Y11*Z11 - Y20*Z20 - Y21*Z21 + Y22*Z22 ) + Y12*( -X00*Z00 - X01*Z01 + X02*Z02 + X10*Z10 + X11*Z11 - X20*Z20 - X21*Z21 + X22*Z22 ) + Y02*( X00*Z10 + X01*Z11 + X10*Z00 + X11*Z01 ) + Z22*( X10*Y20 + X11*Y21 + X20*Y10 + X21*Y11 ) + Z02*( X00*Y10 + X01*Y11 + X10*Y00 + X11*Y01 ) + X02*( Y00*Z10 + Y01*Z11 + Y10*Z00 + Y11*Z01 ) + Y22*( X10*Z20 + X11*Z21 + X20*Z10 + X21*Z11 ) + X22*( Y10*Z20 + Y11*Z21 + Y20*Z10 + Y21*Z11 );
		A.data[69] = X12*( -W00*Y00 - W01*Y01 + W02*Y02 + W10*Y10 + W11*Y11 + 3.0*W12*Y12 - W20*Y20 - W21*Y21 + W22*Y22 ) + Y12*( -W00*X00 - W01*X01 + W02*X02 + W10*X10 + W11*X11 - W20*X20 - W21*X21 + W22*X22 ) + W12*( -X00*Y00 - X01*Y01 + X02*Y02 + X10*Y10 + X11*Y11 - X20*Y20 - X21*Y21 + X22*Y22 ) + Y02*( W00*X10 + W01*X11 + W10*X00 + W11*X01 ) + W22*( X10*Y20 + X11*Y21 + X20*Y10 + X21*Y11 ) + Y22*( W10*X20 + W11*X21 + W20*X10 + W21*X11 ) + W02*( X00*Y10 + X01*Y11 + X10*Y00 + X11*Y01 ) + X02*( W00*Y10 + W01*Y11 + W10*Y00 + W11*Y01 ) + X22*( W10*Y20 + W11*Y21 + W20*Y10 + W21*Y11 );
		A.data[70] = X20*( 0.5*X00*X00 - 0.5*X01*X01 - 0.5*X02*X02 + 0.5*X10*X10 - 0.5*X11*X11 - 0.5*X12*X12 + 0.5*X20*X20 + 0.5*X21*X21 + 0.5*X22*X22 ) + X10*( X11*X21 + X12*X22 ) + X00*( X01*X21 + X02*X22 );
		A.data[71] = Y20*( 0.5*Y00*Y00 - 0.5*Y01*Y01 - 0.5*Y02*Y02 + 0.5*Y10*Y10 - 0.5*Y11*Y11 - 0.5*Y12*Y12 + 0.5*Y20*Y20 + 0.5*Y21*Y21 + 0.5*Y22*Y22 ) + Y00*( Y01*Y21 + Y02*Y22 ) + Y10*( Y11*Y21 + Y12*Y22 );
		A.data[72] = X20*( X00*Y00 - X01*Y01 - X02*Y02 + X10*Y10 - X11*Y11 - X12*Y12 + 1.5*X20*Y20 + X21*Y21 + X22*Y22 ) + 0.5*( Y20*( X00*X00 - X01*X01 - X02*X02 + X10*X10 - X11*X11 - X12*X12 + X21*X21 + X22*X22 ) ) + X10*( X11*Y21 + X12*Y22 + X21*Y11 + X22*Y12 ) + X00*( X01*Y21 + X02*Y22 + X21*Y01 + X22*Y02 ) + Y00*( X01*X21 + X02*X22 ) + Y10*( X11*X21 + X12*X22 );
		A.data[73] = Y20*( X00*Y00 - X01*Y01 - X02*Y02 + X10*Y10 - X11*Y11 - X12*Y12 + 1.5*X20*Y20 + X21*Y21 + X22*Y22 ) + 0.5*( X20*( Y00*Y00 - Y01*Y01 - Y02*Y02 + Y10*Y10 - Y11*Y11 - Y12*Y12 + Y21*Y21 + Y22*Y22 ) ) + Y00*( X01*Y21 + X02*Y22 + X21*Y01 + X22*Y02 ) + Y10*( X11*Y21 + X12*Y22 + X21*Y11 + X22*Y12 ) + X10*( Y11*Y21 + Y12*Y22 ) + X00*( Y01*Y21 + Y02*Y22 );
		A.data[74] = X20*( X00*Z00 - X01*Z01 - X02*Z02 + X10*Z10 - X11*Z11 - X12*Z12 + 1.5*X20*Z20 + X21*Z21 + X22*Z22 ) + Z20*( 0.5*X00*X00 - 0.5*X01*X01 - 0.5*X02*X02 + 0.5*X10*X10 - 0.5*X11*X11 - 0.5*X12*X12 + 0.5*X21*X21 + 0.5*X22*X22 ) + X10*( X11*Z21 + X12*Z22 + X21*Z11 + X22*Z12 ) + X00*( X01*Z21 + X02*Z22 + X21*Z01 + X22*Z02 ) + Z00*( X01*X21 + X02*X22 ) + Z10*( X11*X21 + X12*X22 );
		A.data[75] = X20*( W00*X00 - W01*X01 - W02*X02 + W10*X10 - W11*X11 - W12*X12 + 1.5*W20*X20 + W21*X21 + W22*X22 ) + W20*( 0.5*X00*X00 - 0.5*X01*X01 - 0.5*X02*X02 + 0.5*X10*X10 - 0.5*X11*X11 - 0.5*X12*X12 + 0.5*X21*X21 + 0.5*X22*X22 ) + X10*( W11*X21 + W12*X22 + W21*X11 + W22*X12 ) + X00*( W01*X21 + W02*X22 + W21*X01 + W22*X02 ) + W00*( X01*X21 + X02*X22 ) + W10*( X11*X21 + X12*X22 );
		A.data[76] = Y20*( Y00*Z00 - Y01*Z01 - Y02*Z02 + Y10*Z10 - Y11*Z11 - Y12*Z12 + 1.5*Y20*Z20 + Y21*Z21 + Y22*Z22 ) + Z20*( 0.5*Y00*Y00 - 0.5*Y01*Y01 - 0.5*Y02*Y02 + 0.5*Y10*Y10 - 0.5*Y11*Y11 - 0.5*Y12*Y12 + 0.5*Y21*Y21 + 0.5*Y22*Y22 ) + Y00*( Y01*Z21 + Y02*Z22 + Y21*Z01 + Y22*Z02 ) + Y10*( Y11*Z21 + Y12*Z22 + Y21*Z11 + Y22*Z12 ) + Z00*( Y01*Y21 + Y02*Y22 ) + Z10*( Y11*Y21 + Y12*Y22 );
		A.data[77] = Y20*( W00*Y00 - W01*Y01 - W02*Y02 + W10*Y10 - W11*Y11 - W12*Y12 + 1.5*W20*Y20 + W21*Y21 + W22*Y22 ) + W20*( 0.5*Y00*Y00 - 0.5*Y01*Y01 - 0.5*Y02*Y02 + 0.5*Y10*Y10 - 0.5*Y11*Y11 - 0.5*Y12*Y12 + 0.5*Y21*Y21 + 0.5*Y22*Y22 ) + Y00*( W01*Y21 + W02*Y22 + W21*Y01 + W22*Y02 ) + Y10*( W11*Y21 + W12*Y22 + W21*Y11 + W22*Y12 ) + W00*( Y01*Y21 + Y02*Y22 ) + W10*( Y11*Y21 + Y12*Y22 );
		A.data[78] = Z20*( X00*Y00 - X01*Y01 - X02*Y02 + X10*Y10 - X11*Y11 - X12*Y12 + 3.0*X20*Y20 + X21*Y21 + X22*Y22 ) + Y20*( X00*Z00 - X01*Z01 - X02*Z02 + X10*Z10 - X11*Z11 - X12*Z12 + X21*Z21 + X22*Z22 ) + X20*( Y00*Z00 - Y01*Z01 - Y02*Z02 + Y10*Z10 - Y11*Z11 - Y12*Z12 + Y21*Z21 + Y22*Z22 ) + Y00*( X01*Z21 + X02*Z22 + X21*Z01 + X22*Z02 ) + Z00*( X01*Y21 + X02*Y22 + X21*Y01 + X22*Y02 ) + Z10*( X11*Y21 + X12*Y22 + X21*Y11 + X22*Y12 ) + X10*( Y11*Z21 + Y12*Z22 + Y21*Z11 + Y22*Z12 ) + X00*( Y01*Z21 + Y02*Z22 + Y21*Z01 + Y22*Z02 ) + Y10*( X11*Z21 + X12*Z22 + X21*Z11 + X22*Z12 );
		A.data[79] = W20*( X00*Y00 - X01*Y01 - X02*Y02 + X10*Y10 - X11*Y11 - X12*Y12 + 3.0*X20*Y20 + X21*Y21 + X22*Y22 ) + Y20*( W00*X00 - W01*X01 - W02*X02 + W10*X10 - W11*X11 - W12*X12 + W21*X21 + W22*X22 ) + X20*( W00*Y00 - W01*Y01 - W02*Y02 + W10*Y10 - W11*Y11 - W12*Y12 + W21*Y21 + W22*Y22 ) + Y00*( W01*X21 + W02*X22 + W21*X01 + W22*X02 ) + X00*( W01*Y21 + W02*Y22 + W21*Y01 + W22*Y02 ) + Y10*( W11*X21 + W12*X22 + W21*X11 + W22*X12 ) + W00*( X01*Y21 + X02*Y22 + X21*Y01 + X22*Y02 ) + W10*( X11*Y21 + X12*Y22 + X21*Y11 + X22*Y12 ) + X10*( W11*Y21 + W12*Y22 + W21*Y11 + W22*Y12 );
		A.data[80] = X21*( -0.5*X00*X00 + 0.5*X01*X01 - 0.5*X02*X02 - 0.5*X10*X10 + 0.5*X11*X11 - 0.5*X12*X12 + 0.5*X20*X20 + 0.5*X21*X21 + 0.5*X22*X22 ) + X11*( X10*X20 + X12*X22 ) + X01*( X00*X20 + X02*X22 );
		A.data[81] = Y21*( -0.5*Y00*Y00 + 0.5*Y01*Y01 - 0.5*Y02*Y02 - 0.5*Y10*Y10 + 0.5*Y11*Y11 - 0.5*Y12*Y12 + 0.5*Y20*Y20 + 0.5*Y21*Y21 + 0.5*Y22*Y22 ) + Y01*( Y00*Y20 + Y02*Y22 ) + Y11*( Y10*Y20 + Y12*Y22 );
		A.data[82] = X21*( -X00*Y00 + X01*Y01 - X02*Y02 - X10*Y10 + X11*Y11 - X12*Y12 + X20*Y20 + 1.5*X21*Y21 + X22*Y22 ) + 0.5*( Y21*( -X00*X00 + X01*X01 - X02*X02 - X10*X10 + X11*X11 - X12*X12 + X20*X20 + X22*X22 ) ) + X11*( X10*Y20 + X12*Y22 + X20*Y10 + X22*Y12 ) + X01*( X00*Y20 + X02*Y22 + X20*Y00 + X22*Y02 ) + Y01*( X00*X20 + X02*X22 ) + Y11*( X10*X20 + X12*X22 );
		A.data[83] = Y21*( -X00*Y00 + X01*Y01 - X02*Y02 - X10*Y10 + X11*Y11 - X12*Y12 + X20*Y20 + 1.5*X21*Y21 + X22*Y22 ) + 0.5*( X21*( -Y00*Y00 + Y01*Y01 - Y02*Y02 - Y10*Y10 + Y11*Y11 - Y12*Y12 + Y20*Y20 + Y22*Y22 ) ) + Y01*( X00*Y20 + X02*Y22 + X20*Y00 + X22*Y02 ) + Y11*( X10*Y20 + X12*Y22 + X20*Y10 + X22*Y12 ) + Y20*( X01*Y00 + X11*Y10 ) + Y22*( X01*Y02 + X11*Y12 );
		A.data[84] = X21*( -X00*Z00 + X01*Z01 - X02*Z02 - X10*Z10 + X11*Z11 - X12*Z12 + X20*Z20 + 1.5*X21*Z21 + X22*Z22 ) + Z21*( -0.5*X00*X00 + 0.5*X01*X01 - 0.5*X02*X02 - 0.5*X10*X10 + 0.5*X11*X11 - 0.5*X12*X12 + 0.5*X20*X20 + 0.5*X22*X22 ) + X11*( X10*Z20 + X12*Z22 + X20*Z10 + X22*Z12 ) + X01*( X00*Z20 + X02*Z22 + X20*Z00 + X22*Z02 ) + Z01*( X00*X20 + X02*X22 ) + Z11*( X10*X20 + X12*X22 );
		A.data[85] = X21*( -W00*X00 + W01*X01 - W02*X02 - W10*X10 + W11*X11 - W12*X12 + W20*X20 + 1.5*W21*X21 + W22*X22 ) + 0.5*( W21*( -X00*X00 + X01*X01 - X02*X02 - X10*X10 + X11*X11 - X12*X12 + X20*X20 + X22*X22 ) ) + X11*( W10*X20 + W12*X22 + W20*X10 + W22*X12 ) + X01*( W00*X20 + W02*X22 + W20*X00 + W22*X02 ) + W01*( X00*X20 + X02*X22 ) + W11*( X10*X20 + X12*X22 );
		A.data[86] = Y21*( -Y00*Z00 + Y01*Z01 - Y02*Z02 - Y10*Z10 + Y11*Z11 - Y12*Z12 + Y20*Z20 + 1.5*Y21*Z21 + Y22*Z22 ) + Z21*( -0.5*Y00*Y00 + 0.5*Y01*Y01 - 0.5*Y02*Y02 - 0.5*Y10*Y10 + 0.5*Y11*Y11 - 0.5*Y12*Y12 + 0.5*Y20*Y20 + 0.5*Y22*Y22 ) + Y01*( Y00*Z20 + Y02*Z22 + Y20*Z00 + Y22*Z02 ) + Y11*( Y10*Z20 + Y12*Z22 + Y20*Z10 + Y22*Z12 ) + Z01*( Y00*Y20 + Y02*Y22 ) + Z11*( Y10*Y20 + Y12*Y22 );
		A.data[87] = Y21*( -W00*Y00 + W01*Y01 - W02*Y02 - W10*Y10 + W11*Y11 - W12*Y12 + W20*Y20 + 1.5*W21*Y21 + W22*Y22 ) + 0.5*( W21*( -Y00*Y00 + Y01*Y01 - Y02*Y02 - Y10*Y10 + Y11*Y11 - Y12*Y12 + Y20*Y20 + Y22*Y22 ) ) + Y01*( W00*Y20 + W02*Y22 + W20*Y00 + W22*Y02 ) + Y11*( W10*Y20 + W12*Y22 + W20*Y10 + W22*Y12 ) + W01*( Y00*Y20 + Y02*Y22 ) + W11*( Y10*Y20 + Y12*Y22 );
		A.data[88] = Z21*( -X00*Y00 + X01*Y01 - X02*Y02 - X10*Y10 + X11*Y11 - X12*Y12 + X20*Y20 + 3.0*X21*Y21 + X22*Y22 ) + Y21*( -X00*Z00 + X01*Z01 - X02*Z02 - X10*Z10 + X11*Z11 - X12*Z12 + X20*Z20 + X22*Z22 ) + X21*( -Y00*Z00 + Y01*Z01 - Y02*Z02 - Y10*Z10 + Y11*Z11 - Y12*Z12 + Y20*Z20 + Y22*Z22 ) + Y01*( X00*Z20 + X02*Z22 + X20*Z00 + X22*Z02 ) + Z01*( X00*Y20 + X02*Y22 + X20*Y00 + X22*Y02 ) + Z11*( X10*Y20 + X12*Y22 + X20*Y10 + X22*Y12 ) + X11*( Y10*Z20 + Y12*Z22 + Y20*Z10 + Y22*Z12 ) + X01*( Y00*Z20 + Y02*Z22 + Y20*Z00 + Y22*Z02 ) + Y11*( X10*Z20 + X12*Z22 + X20*Z10 + X22*Z12 );
		A.data[89] = W21*( -X00*Y00 + X01*Y01 - X02*Y02 - X10*Y10 + X11*Y11 - X12*Y12 + X20*Y20 + 3.0*X21*Y21 + X22*Y22 ) + Y21*( -W00*X00 + W01*X01 - W02*X02 - W10*X10 + W11*X11 - W12*X12 + W20*X20 + W22*X22 ) + X21*( -W00*Y00 + W01*Y01 - W02*Y02 - W10*Y10 + W11*Y11 - W12*Y12 + W20*Y20 + W22*Y22 ) + Y01*( W00*X20 + W02*X22 + W20*X00 + W22*X02 ) + Y20*( W00*X01 + W01*X00 + W10*X11 + W11*X10 ) + Y22*( W01*X02 + W02*X01 + W11*X12 + W12*X11 ) + Y11*( W10*X20 + W12*X22 + W20*X10 + W22*X12 ) + Y02*( W01*X22 + W22*X01 ) + Y00*( W01*X20 + W20*X01 ) + W11*( X20*Y10 + X22*Y12 ) + X11*( W20*Y10 + W22*Y12 );
		A.data[90] = X22*( -0.5*X00*X00 - 0.5*X01*X01 + 0.5*X02*X02 - 0.5*X10*X10 - 0.5*X11*X11 + 0.5*X12*X12 + 0.5*X20*X20 + 0.5*X21*X21 + 0.5*X22*X22 ) + X12*( X10*X20 + X11*X21 ) + X02*( X00*X20 + X01*X21 );
		A.data[91] = Y22*( -0.5*Y00*Y00 - 0.5*Y01*Y01 + 0.5*Y02*Y02 - 0.5*Y10*Y10 - 0.5*Y11*Y11 + 0.5*Y12*Y12 + 0.5*Y20*Y20 + 0.5*Y21*Y21 + 0.5*Y22*Y22 ) + Y02*( Y00*Y20 + Y01*Y21 ) + Y12*( Y10*Y20 + Y11*Y21 );
		A.data[92] = X22*( -X00*Y00 - X01*Y01 + X02*Y02 - X10*Y10 - X11*Y11 + X12*Y12 + X20*Y20 + X21*Y21 + 1.5*X22*Y22 ) + 0.5*( Y22*( -X00*X00 - X01*X01 + X02*X02 - X10*X10 - X11*X11 + X12*X12 + X20*X20 + X21*X21 ) ) + X02*( X00*Y20 + X01*Y21 + X20*Y00 + X21*Y01 ) + X12*( X10*Y20 + X11*Y21 + X20*Y10 + X21*Y11 ) + Y02*( X00*X20 + X01*X21 ) + Y12*( X10*X20 + X11*X21 );
		A.data[93] = Y22*( -X00*Y00 - X01*Y01 + X02*Y02 - X10*Y10 - X11*Y11 + X12*Y12 + X20*Y20 + X21*Y21 + 1.5*X22*Y22 ) + 0.5*( X22*( -Y00*Y00 - Y01*Y01 + Y02*Y02 - Y10*Y10 - Y11*Y11 + Y12*Y12 + Y20*Y20 + Y21*Y21 ) ) + Y02*( X00*Y20 + X01*Y21 + X20*Y00 + X21*Y01 ) + Y12*( X10*Y20 + X11*Y21 + X20*Y10 + X21*Y11 ) + Y21*( X02*Y01 + X12*Y11 ) + Y20*( X02*Y00 + X12*Y10 );
		A.data[94] = X22*( -X00*Z00 - X01*Z01 + X02*Z02 - X10*Z10 - X11*Z11 + X12*Z12 + X20*Z20 + X21*Z21 + 1.5*X22*Z22 ) + Z22*( -0.5*X00*X00 - 0.5*X01*X01 + 0.5*X02*X02 - 0.5*X10*X10 - 0.5*X11*X11 + 0.5*X12*X12 + 0.5*X20*X20 + 0.5*X21*X21 ) + X12*( X10*Z20 + X11*Z21 + X20*Z10 + X21*Z11 ) + X02*( X00*Z20 + X01*Z21 + X20*Z00 + X21*Z01 ) + Z02*( X00*X20 + X01*X21 ) + Z12*( X10*X20 + X11*X21 );
		A.data[95] = X22*( -W00*X00 - W01*X01 + W02*X02 - W10*X10 - W11*X11 + W12*X12 + W20*X20 + W21*X21 + 1.5*W22*X22 ) + 0.5*( W22*( -X00*X00 - X01*X01 + X02*X02 - X10*X10 - X11*X11 + X12*X12 + X20*X20 + X21*X21 ) ) + X12*( W10*X20 + W11*X21 + W20*X10 + W21*X11 ) + X02*( W00*X20 + W01*X21 + W20*X00 + W21*X01 ) + W12*( X10*X20 + X11*X21 ) + W02*( X00*X20 + X01*X21 );
		A.data[96] = Y22*( -Y00*Z00 - Y01*Z01 + Y02*Z02 - Y10*Z10 - Y11*Z11 + Y12*Z12 + Y20*Z20 + Y21*Z21 + 1.5*Y22*Z22 ) + Z22*( -0.5*Y00*Y00 - 0.5*Y01*Y01 + 0.5*Y02*Y02 - 0.5*Y10*Y10 - 0.5*Y11*Y11 + 0.5*Y12*Y12 + 0.5*Y20*Y20 + 0.5*Y21*Y21 ) + Y02*( Y00*Z20 + Y01*Z21 + Y20*Z00 + Y21*Z01 ) + Y12*( Y10*Z20 + Y11*Z21 + Y20*Z10 + Y21*Z11 ) + Z02*( Y00*Y20 + Y01*Y21 ) + Z12*( Y10*Y20 + Y11*Y21 );
		A.data[97] = Y22*( -W00*Y00 - W01*Y01 + W02*Y02 - W10*Y10 - W11*Y11 + W12*Y12 + W20*Y20 + W21*Y21 + 1.5*W22*Y22 ) + 0.5*( W22*( -Y00*Y00 - Y01*Y01 + Y02*Y02 - Y10*Y10 - Y11*Y11 + Y12*Y12 + Y20*Y20 + Y21*Y21 ) ) + Y02*( W00*Y20 + W01*Y21 + W20*Y00 + W21*Y01 ) + Y12*( W10*Y20 + W11*Y21 + W20*Y10 + W21*Y11 ) + W12*( Y10*Y20 + Y11*Y21 ) + W02*( Y00*Y20 + Y01*Y21 );
		A.data[98] = Z22*( -X00*Y00 - X01*Y01 + X02*Y02 - X10*Y10 - X11*Y11 + X12*Y12 + X20*Y20 + X21*Y21 + 3.0*X22*Y22 ) + Y22*( -X00*Z00 - X01*Z01 + X02*Z02 - X10*Z10 - X11*Z11 + X12*Z12 + X20*Z20 + X21*Z21 ) + X22*( -Y00*Z00 - Y01*Z01 + Y02*Z02 - Y10*Z10 - Y11*Z11 + Y12*Z12 + Y20*Z20 + Y21*Z21 ) + Y02*( X00*Z20 + X01*Z21 + X20*Z00 + X21*Z01 ) + Z02*( X00*Y20 + X01*Y21 + X20*Y00 + X21*Y01 ) + Z12*( X10*Y20 + X11*Y21 + X20*Y10 + X21*Y11 ) + X02*( Y00*Z20 + Y01*Z21 + Y20*Z00 + Y21*Z01 ) + X12*( Y10*Z20 + Y11*Z21 + Y20*Z10 + Y21*Z11 ) + Y12*( X10*Z20 + X11*Z21 + X20*Z10 + X21*Z11 );
		A.data[99] = W22*( -X00*Y00 - X01*Y01 + X02*Y02 - X10*Y10 - X11*Y11 + X12*Y12 + X20*Y20 + X21*Y21 + 3.0*X22*Y22 ) + Y22*( -W00*X00 - W01*X01 + W02*X02 - W10*X10 - W11*X11 + W12*X12 + W20*X20 + W21*X21 ) + X22*( -W00*Y00 - W01*Y01 + W02*Y02 - W10*Y10 - W11*Y11 + W12*Y12 + W20*Y20 + W21*Y21 ) + Y02*( W00*X20 + W01*X21 + W20*X00 + W21*X01 ) + X02*( W00*Y20 + W01*Y21 + W20*Y00 + W21*Y01 ) + X12*( W10*Y20 + W11*Y21 + W20*Y10 + W21*Y11 ) + W12*( X10*Y20 + X11*Y21 + X20*Y10 + X21*Y11 ) + W02*( X00*Y20 + X01*Y21 + X20*Y00 + X21*Y01 ) + Y12*( W10*X20 + W11*X21 + W20*X10 + W21*X11 );
	}

	/**
	 * Construct 10 by 10 coefficient matrix for the following coefficients
	 *
	 * 'x*z^2','x*z','x','y*z^2','y*z','y','z^3','z^2','z','1'
	 *
	 * @param B a 10 by 10 matrix
	 */
	public void setupA2(DenseMatrix64F B) {
		B.data[0] = Z22*( X00*Z11 - X01*Z10 - X10*Z01 + X11*Z00 ) + Z21*( -X00*Z12 + X02*Z10 + X10*Z02 - X12*Z00 ) + Z20*( X01*Z12 - X02*Z11 - X11*Z02 + X12*Z01 ) + Z01*( X20*Z12 - X22*Z10 ) + Z00*( -X21*Z12 + X22*Z11 ) + Z02*( -X20*Z11 + X21*Z10 );
		B.data[1] = Z22*( W00*X11 - W01*X10 - W10*X01 + W11*X00 ) + Z21*( -W00*X12 + W02*X10 + W10*X02 - W12*X00 ) + Z20*( W01*X12 - W02*X11 - W11*X02 + W12*X01 ) + Z01*( -W10*X22 + W12*X20 + W20*X12 - W22*X10 ) + Z00*( W11*X22 - W12*X21 - W21*X12 + W22*X11 ) + Z02*( W10*X21 - W11*X20 - W20*X11 + W21*X10 ) + Z12*( -W00*X21 + W01*X20 + W20*X01 - W21*X00 ) + Z11*( W00*X22 - W02*X20 - W20*X02 + W22*X00 ) + Z10*( -W01*X22 + W02*X21 + W21*X02 - W22*X01 );
		B.data[2] = W00*( W11*X22 - W12*X21 - W21*X12 + W22*X11 ) + W01*( -W10*X22 + W12*X20 + W20*X12 - W22*X10 ) + W02*( W10*X21 - W11*X20 - W20*X11 + W21*X10 ) + W12*( W20*X01 - W21*X00 ) + W11*( -W20*X02 + W22*X00 ) + W10*( W21*X02 - W22*X01 );
		B.data[3] = Z22*( Y00*Z11 - Y01*Z10 - Y10*Z01 + Y11*Z00 ) + Z21*( -Y00*Z12 + Y02*Z10 + Y10*Z02 - Y12*Z00 ) + Z20*( Y01*Z12 - Y02*Z11 - Y11*Z02 + Y12*Z01 ) + Z01*( Y20*Z12 - Y22*Z10 ) + Z00*( -Y21*Z12 + Y22*Z11 ) + Z02*( -Y20*Z11 + Y21*Z10 );
		B.data[4] = Y02*( W10*Z21 - W11*Z20 - W20*Z11 + W21*Z10 ) + Y00*( W11*Z22 - W12*Z21 - W21*Z12 + W22*Z11 ) + Y01*( -W10*Z22 + W12*Z20 + W20*Z12 - W22*Z10 ) + Z01*( -W10*Y22 + W12*Y20 + W20*Y12 - W22*Y10 ) + Z00*( W11*Y22 - W12*Y21 - W21*Y12 + W22*Y11 ) + Z02*( W10*Y21 - W11*Y20 - W20*Y11 + W21*Y10 ) + W00*( Y11*Z22 - Y12*Z21 - Y21*Z12 + Y22*Z11 ) + W01*( -Y10*Z22 + Y12*Z20 + Y20*Z12 - Y22*Z10 ) + W02*( Y10*Z21 - Y11*Z20 - Y20*Z11 + Y21*Z10 );
		B.data[5] = W00*( W11*Y22 - W12*Y21 - W21*Y12 + W22*Y11 ) + W01*( -W10*Y22 + W12*Y20 + W20*Y12 - W22*Y10 ) + W02*( W10*Y21 - W11*Y20 - W20*Y11 + W21*Y10 ) + Y02*( W10*W21 - W11*W20 ) + Y00*( W11*W22 - W12*W21 ) + Y01*( -W10*W22 + W12*W20 );
		B.data[6] = Z22*( Z00*Z11 - Z01*Z10 ) + Z21*( -Z00*Z12 + Z02*Z10 ) + Z20*( Z01*Z12 - Z02*Z11 );
		B.data[7] = Z22*( W00*Z11 - W01*Z10 - W10*Z01 + W11*Z00 ) + Z21*( -W00*Z12 + W02*Z10 + W10*Z02 - W12*Z00 ) + Z20*( W01*Z12 - W02*Z11 - W11*Z02 + W12*Z01 ) + Z01*( W20*Z12 - W22*Z10 ) + Z00*( -W21*Z12 + W22*Z11 ) + Z02*( -W20*Z11 + W21*Z10 );
		B.data[8] = W02*( W10*Z21 - W11*Z20 - W20*Z11 + W21*Z10 ) + W00*( W11*Z22 - W12*Z21 - W21*Z12 + W22*Z11 ) + W01*( -W10*Z22 + W12*Z20 + W20*Z12 - W22*Z10 ) + Z01*( -W10*W22 + W12*W20 ) + Z00*( W11*W22 - W12*W21 ) + Z02*( W10*W21 - W11*W20 );
		B.data[9] = W00*( W11*W22 - W12*W21 ) + W01*( -W10*W22 + W12*W20 ) + W02*( W10*W21 - W11*W20 );
		B.data[10] = Z00*( 1.5*X00*Z00 + X01*Z01 + X02*Z02 + X10*Z10 - X11*Z11 - X12*Z12 + X20*Z20 - X21*Z21 - X22*Z22 ) + 0.5*( X00*( Z01*Z01 + Z02*Z02 + Z10*Z10 - Z11*Z11 - Z12*Z12 + Z20*Z20 - Z21*Z21 - Z22*Z22 ) ) + Z20*( X01*Z21 + X02*Z22 + X21*Z01 + X22*Z02 ) + Z10*( X01*Z11 + X02*Z12 + X11*Z01 + X12*Z02 ) + Z01*( X10*Z11 + X20*Z21 ) + Z02*( X10*Z12 + X20*Z22 );
		B.data[11] = Z00*( 3.0*W00*X00 + W01*X01 + W02*X02 + W10*X10 - W11*X11 - W12*X12 + W20*X20 - W21*X21 - W22*X22 ) + X00*( W01*Z01 + W02*Z02 + W10*Z10 - W11*Z11 - W12*Z12 + W20*Z20 - W21*Z21 - W22*Z22 ) + W00*( X01*Z01 + X02*Z02 + X10*Z10 - X11*Z11 - X12*Z12 + X20*Z20 - X21*Z21 - X22*Z22 ) + Z20*( W01*X21 + W02*X22 + W21*X01 + W22*X02 ) + Z01*( W10*X11 + W11*X10 + W20*X21 + W21*X20 ) + Z02*( W10*X12 + W12*X10 + W20*X22 + W22*X20 ) + Z10*( W01*X11 + W02*X12 + W11*X01 + W12*X02 ) + Z22*( W02*X20 + W20*X02 ) + Z21*( W01*X20 + W20*X01 ) + Z12*( W02*X10 + W10*X02 ) + Z11*( W01*X10 + W10*X01 );
		B.data[12] = W00*( 1.5*W00*X00 + W01*X01 + W02*X02 + W10*X10 - W11*X11 - W12*X12 + W20*X20 - W21*X21 - W22*X22 ) + 0.5*( X00*( W01*W01 + W02*W02 + W10*W10 - W11*W11 - W12*W12 + W20*W20 - W21*W21 - W22*W22 ) ) + W01*( W10*X11 + W11*X10 + W20*X21 + W21*X20 ) + W02*( W10*X12 + W12*X10 + W20*X22 + W22*X20 ) + W10*( W11*X01 + W12*X02 ) + W20*( W21*X01 + W22*X02 );
		B.data[13] = Z00*( 1.5*Y00*Z00 + Y01*Z01 + Y02*Z02 + Y10*Z10 - Y11*Z11 - Y12*Z12 + Y20*Z20 - Y21*Z21 - Y22*Z22 ) + Y00*( 0.5*Z01*Z01 + 0.5*Z02*Z02 + 0.5*Z10*Z10 - 0.5*Z11*Z11 - 0.5*Z12*Z12 + 0.5*Z20*Z20 - 0.5*Z21*Z21 - 0.5*Z22*Z22 ) + Z20*( Y01*Z21 + Y02*Z22 + Y21*Z01 + Y22*Z02 ) + Z10*( Y01*Z11 + Y02*Z12 + Y11*Z01 + Y12*Z02 ) + Z01*( Y10*Z11 + Y20*Z21 ) + Z02*( Y10*Z12 + Y20*Z22 );
		B.data[14] = Y00*( 3.0*W00*Z00 + W01*Z01 + W02*Z02 + W10*Z10 - W11*Z11 - W12*Z12 + W20*Z20 - W21*Z21 - W22*Z22 ) + Z00*( W01*Y01 + W02*Y02 + W10*Y10 - W11*Y11 - W12*Y12 + W20*Y20 - W21*Y21 - W22*Y22 ) + W00*( Y01*Z01 + Y02*Z02 + Y10*Z10 - Y11*Z11 - Y12*Z12 + Y20*Z20 - Y21*Z21 - Y22*Z22 ) + Y02*( W10*Z12 + W12*Z10 + W20*Z22 + W22*Z20 ) + Y01*( W10*Z11 + W11*Z10 + W20*Z21 + W21*Z20 ) + Z01*( W10*Y11 + W11*Y10 + W20*Y21 + W21*Y20 ) + Z02*( W10*Y12 + W12*Y10 + W20*Y22 + W22*Y20 ) + W01*( Y10*Z11 + Y11*Z10 + Y20*Z21 + Y21*Z20 ) + W02*( Y10*Z12 + Y12*Z10 + Y20*Z22 + Y22*Z20 );
		B.data[15] = W00*( 1.5*W00*Y00 + W01*Y01 + W02*Y02 + W10*Y10 - W11*Y11 - W12*Y12 + W20*Y20 - W21*Y21 - W22*Y22 ) + Y00*( 0.5*W01*W01 + 0.5*W02*W02 + 0.5*W10*W10 - 0.5*W11*W11 - 0.5*W12*W12 + 0.5*W20*W20 - 0.5*W21*W21 - 0.5*W22*W22 ) + W01*( W10*Y11 + W11*Y10 + W20*Y21 + W21*Y20 ) + W02*( W10*Y12 + W12*Y10 + W20*Y22 + W22*Y20 ) + Y02*( W10*W12 + W20*W22 ) + Y01*( W10*W11 + W20*W21 );
		B.data[16] = Z00*( 0.5*Z00*Z00 + 0.5*Z01*Z01 + 0.5*Z02*Z02 + 0.5*Z10*Z10 - 0.5*Z11*Z11 - 0.5*Z12*Z12 + 0.5*Z20*Z20 - 0.5*Z21*Z21 - 0.5*Z22*Z22 ) + Z20*( Z01*Z21 + Z02*Z22 ) + Z10*( Z01*Z11 + Z02*Z12 );
		B.data[17] = Z00*( 1.5*W00*Z00 + W01*Z01 + W02*Z02 + W10*Z10 - W11*Z11 - W12*Z12 + W20*Z20 - W21*Z21 - W22*Z22 ) + 0.5*( W00*( Z01*Z01 + Z02*Z02 + Z10*Z10 - Z11*Z11 - Z12*Z12 + Z20*Z20 - Z21*Z21 - Z22*Z22 ) ) + Z20*( W01*Z21 + W02*Z22 + W21*Z01 + W22*Z02 ) + Z10*( W01*Z11 + W02*Z12 + W11*Z01 + W12*Z02 ) + Z01*( W10*Z11 + W20*Z21 ) + Z02*( W10*Z12 + W20*Z22 );
		B.data[18] = W00*( 1.5*W00*Z00 + W01*Z01 + W02*Z02 + W10*Z10 - W11*Z11 - W12*Z12 + W20*Z20 - W21*Z21 - W22*Z22 ) + Z00*( 0.5*W01*W01 + 0.5*W02*W02 + 0.5*W10*W10 - 0.5*W11*W11 - 0.5*W12*W12 + 0.5*W20*W20 - 0.5*W21*W21 - 0.5*W22*W22 ) + W02*( W10*Z12 + W12*Z10 + W20*Z22 + W22*Z20 ) + W01*( W10*Z11 + W11*Z10 + W20*Z21 + W21*Z20 ) + Z01*( W10*W11 + W20*W21 ) + Z02*( W10*W12 + W20*W22 );
		B.data[19] = W00*( 0.5*W00*W00 + 0.5*W01*W01 + 0.5*W02*W02 + 0.5*W10*W10 - 0.5*W11*W11 - 0.5*W12*W12 + 0.5*W20*W20 - 0.5*W21*W21 - 0.5*W22*W22 ) + W01*( W10*W11 + W20*W21 ) + W02*( W10*W12 + W20*W22 );
		B.data[20] = Z01*( X00*Z00 + 1.5*X01*Z01 + X02*Z02 - X10*Z10 + X11*Z11 - X12*Z12 - X20*Z20 + X21*Z21 - X22*Z22 ) + 0.5*( X01*( Z00*Z00 + Z02*Z02 - Z10*Z10 + Z11*Z11 - Z12*Z12 - Z20*Z20 + Z21*Z21 - Z22*Z22 ) ) + Z21*( X00*Z20 + X02*Z22 + X20*Z00 + X22*Z02 ) + Z11*( X00*Z10 + X02*Z12 + X10*Z00 + X12*Z02 ) + Z00*( X11*Z10 + X21*Z20 ) + Z02*( X11*Z12 + X21*Z22 );
		B.data[21] = Z01*( W00*X00 + 3.0*W01*X01 + W02*X02 - W10*X10 + W11*X11 - W12*X12 - W20*X20 + W21*X21 - W22*X22 ) + X01*( W00*Z00 + W02*Z02 - W10*Z10 + W11*Z11 - W12*Z12 - W20*Z20 + W21*Z21 - W22*Z22 ) + W01*( X00*Z00 + X02*Z02 - X10*Z10 + X11*Z11 - X12*Z12 - X20*Z20 + X21*Z21 - X22*Z22 ) + Z21*( W00*X20 + W02*X22 + W20*X00 + W22*X02 ) + Z00*( W10*X11 + W11*X10 + W20*X21 + W21*X20 ) + Z02*( W11*X12 + W12*X11 + W21*X22 + W22*X21 ) + Z11*( W00*X10 + W02*X12 + W10*X00 + W12*X02 ) + Z22*( W02*X21 + W21*X02 ) + Z20*( W00*X21 + W21*X00 ) + Z10*( W00*X11 + W11*X00 ) + Z12*( W02*X11 + W11*X02 );
		B.data[22] = W01*( W00*X00 + 1.5*W01*X01 + W02*X02 - W10*X10 + W11*X11 - W12*X12 - W20*X20 + W21*X21 - W22*X22 ) + 0.5*( X01*( W00*W00 + W02*W02 - W10*W10 + W11*W11 - W12*W12 - W20*W20 + W21*W21 - W22*W22 ) ) + W00*( W10*X11 + W11*X10 + W20*X21 + W21*X20 ) + W02*( W11*X12 + W12*X11 + W21*X22 + W22*X21 ) + W11*( W10*X00 + W12*X02 ) + W21*( W20*X00 + W22*X02 );
		B.data[23] = Z01*( Y00*Z00 + 1.5*Y01*Z01 + Y02*Z02 - Y10*Z10 + Y11*Z11 - Y12*Z12 - Y20*Z20 + Y21*Z21 - Y22*Z22 ) + Y01*( 0.5*Z00*Z00 + 0.5*Z02*Z02 - 0.5*Z10*Z10 + 0.5*Z11*Z11 - 0.5*Z12*Z12 - 0.5*Z20*Z20 + 0.5*Z21*Z21 - 0.5*Z22*Z22 ) + Z21*( Y00*Z20 + Y02*Z22 + Y20*Z00 + Y22*Z02 ) + Z11*( Y00*Z10 + Y02*Z12 + Y10*Z00 + Y12*Z02 ) + Z00*( Y11*Z10 + Y21*Z20 ) + Z02*( Y11*Z12 + Y21*Z22 );
		B.data[24] = Y01*( W00*Z00 + 3.0*W01*Z01 + W02*Z02 - W10*Z10 + W11*Z11 - W12*Z12 - W20*Z20 + W21*Z21 - W22*Z22 ) + Z01*( W00*Y00 + W02*Y02 - W10*Y10 + W11*Y11 - W12*Y12 - W20*Y20 + W21*Y21 - W22*Y22 ) + W01*( Y00*Z00 + Y02*Z02 - Y10*Z10 + Y11*Z11 - Y12*Z12 - Y20*Z20 + Y21*Z21 - Y22*Z22 ) + Y02*( W11*Z12 + W12*Z11 + W21*Z22 + W22*Z21 ) + Y00*( W10*Z11 + W11*Z10 + W20*Z21 + W21*Z20 ) + Z00*( W10*Y11 + W11*Y10 + W20*Y21 + W21*Y20 ) + Z02*( W11*Y12 + W12*Y11 + W21*Y22 + W22*Y21 ) + W00*( Y10*Z11 + Y11*Z10 + Y20*Z21 + Y21*Z20 ) + W02*( Y11*Z12 + Y12*Z11 + Y21*Z22 + Y22*Z21 );
		B.data[25] = W01*( W00*Y00 + 1.5*W01*Y01 + W02*Y02 - W10*Y10 + W11*Y11 - W12*Y12 - W20*Y20 + W21*Y21 - W22*Y22 ) + Y01*( 0.5*W00*W00 + 0.5*W02*W02 - 0.5*W10*W10 + 0.5*W11*W11 - 0.5*W12*W12 - 0.5*W20*W20 + 0.5*W21*W21 - 0.5*W22*W22 ) + W00*( W10*Y11 + W11*Y10 + W20*Y21 + W21*Y20 ) + W02*( W11*Y12 + W12*Y11 + W21*Y22 + W22*Y21 ) + Y02*( W11*W12 + W21*W22 ) + Y00*( W10*W11 + W20*W21 );
		B.data[26] = Z01*( 0.5*Z00*Z00 + 0.5*Z01*Z01 + 0.5*Z02*Z02 - 0.5*Z10*Z10 + 0.5*Z11*Z11 - 0.5*Z12*Z12 - 0.5*Z20*Z20 + 0.5*Z21*Z21 - 0.5*Z22*Z22 ) + Z21*( Z00*Z20 + Z02*Z22 ) + Z11*( Z00*Z10 + Z02*Z12 );
		B.data[27] = Z01*( W00*Z00 + 1.5*W01*Z01 + W02*Z02 - W10*Z10 + W11*Z11 - W12*Z12 - W20*Z20 + W21*Z21 - W22*Z22 ) + 0.5*( W01*( Z00*Z00 + Z02*Z02 - Z10*Z10 + Z11*Z11 - Z12*Z12 - Z20*Z20 + Z21*Z21 - Z22*Z22 ) ) + Z21*( W00*Z20 + W02*Z22 + W20*Z00 + W22*Z02 ) + Z11*( W00*Z10 + W02*Z12 + W10*Z00 + W12*Z02 ) + Z00*( W11*Z10 + W21*Z20 ) + Z02*( W11*Z12 + W21*Z22 );
		B.data[28] = W01*( W00*Z00 + 1.5*W01*Z01 + W02*Z02 - W10*Z10 + W11*Z11 - W12*Z12 - W20*Z20 + W21*Z21 - W22*Z22 ) + Z01*( 0.5*W00*W00 + 0.5*W02*W02 - 0.5*W10*W10 + 0.5*W11*W11 - 0.5*W12*W12 - 0.5*W20*W20 + 0.5*W21*W21 - 0.5*W22*W22 ) + W00*( W10*Z11 + W11*Z10 + W20*Z21 + W21*Z20 ) + W02*( W11*Z12 + W12*Z11 + W21*Z22 + W22*Z21 ) + Z00*( W10*W11 + W20*W21 ) + Z02*( W11*W12 + W21*W22 );
		B.data[29] = W01*( 0.5*W00*W00 + 0.5*W01*W01 + 0.5*W02*W02 - 0.5*W10*W10 + 0.5*W11*W11 - 0.5*W12*W12 - 0.5*W20*W20 + 0.5*W21*W21 - 0.5*W22*W22 ) + W00*( W10*W11 + W20*W21 ) + W02*( W11*W12 + W21*W22 );
		B.data[30] = Z02*( X00*Z00 + X01*Z01 + 1.5*X02*Z02 - X10*Z10 - X11*Z11 + X12*Z12 - X20*Z20 - X21*Z21 + X22*Z22 ) + 0.5*( X02*( Z00*Z00 + Z01*Z01 - Z10*Z10 - Z11*Z11 + Z12*Z12 - Z20*Z20 - Z21*Z21 + Z22*Z22 ) ) + Z22*( X00*Z20 + X01*Z21 + X20*Z00 + X21*Z01 ) + Z12*( X00*Z10 + X01*Z11 + X10*Z00 + X11*Z01 ) + Z01*( X12*Z11 + X22*Z21 ) + Z00*( X12*Z10 + X22*Z20 );
		B.data[31] = Z02*( W00*X00 + W01*X01 + 3.0*W02*X02 - W10*X10 - W11*X11 + W12*X12 - W20*X20 - W21*X21 + W22*X22 ) + X02*( W00*Z00 + W01*Z01 - W10*Z10 - W11*Z11 + W12*Z12 - W20*Z20 - W21*Z21 + W22*Z22 ) + W02*( X00*Z00 + X01*Z01 - X10*Z10 - X11*Z11 + X12*Z12 - X20*Z20 - X21*Z21 + X22*Z22 ) + Z22*( W00*X20 + W01*X21 + W20*X00 + W21*X01 ) + Z01*( W11*X12 + W12*X11 + W21*X22 + W22*X21 ) + Z12*( W00*X10 + W01*X11 + W10*X00 + W11*X01 ) + Z00*( W10*X12 + W12*X10 + W20*X22 + W22*X20 ) + Z21*( W01*X22 + W22*X01 ) + Z20*( W00*X22 + W22*X00 ) + W12*( X00*Z10 + X01*Z11 ) + X12*( W00*Z10 + W01*Z11 );
		B.data[32] = W02*( W00*X00 + W01*X01 + 1.5*W02*X02 - W10*X10 - W11*X11 + W12*X12 - W20*X20 - W21*X21 + W22*X22 ) + 0.5*( X02*( W00*W00 + W01*W01 - W10*W10 - W11*W11 + W12*W12 - W20*W20 - W21*W21 + W22*W22 ) ) + W00*( W10*X12 + W12*X10 + W20*X22 + W22*X20 ) + W01*( W11*X12 + W12*X11 + W21*X22 + W22*X21 ) + W12*( W10*X00 + W11*X01 ) + W22*( W20*X00 + W21*X01 );
		B.data[33] = Z02*( Y00*Z00 + Y01*Z01 + 1.5*Y02*Z02 - Y10*Z10 - Y11*Z11 + Y12*Z12 - Y20*Z20 - Y21*Z21 + Y22*Z22 ) + Y02*( 0.5*Z00*Z00 + 0.5*Z01*Z01 - 0.5*Z10*Z10 - 0.5*Z11*Z11 + 0.5*Z12*Z12 - 0.5*Z20*Z20 - 0.5*Z21*Z21 + 0.5*Z22*Z22 ) + Z22*( Y00*Z20 + Y01*Z21 + Y20*Z00 + Y21*Z01 ) + Z12*( Y00*Z10 + Y01*Z11 + Y10*Z00 + Y11*Z01 ) + Z01*( Y12*Z11 + Y22*Z21 ) + Z00*( Y12*Z10 + Y22*Z20 );
		B.data[34] = Y02*( W00*Z00 + W01*Z01 + 3.0*W02*Z02 - W10*Z10 - W11*Z11 + W12*Z12 - W20*Z20 - W21*Z21 + W22*Z22 ) + Z02*( W00*Y00 + W01*Y01 - W10*Y10 - W11*Y11 + W12*Y12 - W20*Y20 - W21*Y21 + W22*Y22 ) + W02*( Y00*Z00 + Y01*Z01 - Y10*Z10 - Y11*Z11 + Y12*Z12 - Y20*Z20 - Y21*Z21 + Y22*Z22 ) + Y00*( W10*Z12 + W12*Z10 + W20*Z22 + W22*Z20 ) + Y01*( W11*Z12 + W12*Z11 + W21*Z22 + W22*Z21 ) + Z01*( W11*Y12 + W12*Y11 + W21*Y22 + W22*Y21 ) + W00*( Y10*Z12 + Y12*Z10 + Y20*Z22 + Y22*Z20 ) + Z00*( W10*Y12 + W12*Y10 + W20*Y22 + W22*Y20 ) + W01*( Y11*Z12 + Y12*Z11 + Y21*Z22 + Y22*Z21 );
		B.data[35] = W02*( W00*Y00 + W01*Y01 + 1.5*W02*Y02 - W10*Y10 - W11*Y11 + W12*Y12 - W20*Y20 - W21*Y21 + W22*Y22 ) + Y02*( 0.5*W00*W00 + 0.5*W01*W01 - 0.5*W10*W10 - 0.5*W11*W11 + 0.5*W12*W12 - 0.5*W20*W20 - 0.5*W21*W21 + 0.5*W22*W22 ) + W00*( W10*Y12 + W12*Y10 + W20*Y22 + W22*Y20 ) + W01*( W11*Y12 + W12*Y11 + W21*Y22 + W22*Y21 ) + Y00*( W10*W12 + W20*W22 ) + Y01*( W11*W12 + W21*W22 );
		B.data[36] = Z02*( 0.5*Z00*Z00 + 0.5*Z01*Z01 + 0.5*Z02*Z02 - 0.5*Z10*Z10 - 0.5*Z11*Z11 + 0.5*Z12*Z12 - 0.5*Z20*Z20 - 0.5*Z21*Z21 + 0.5*Z22*Z22 ) + Z22*( Z00*Z20 + Z01*Z21 ) + Z12*( Z00*Z10 + Z01*Z11 );
		B.data[37] = Z02*( W00*Z00 + W01*Z01 + 1.5*W02*Z02 - W10*Z10 - W11*Z11 + W12*Z12 - W20*Z20 - W21*Z21 + W22*Z22 ) + W02*( 0.5*Z00*Z00 + 0.5*Z01*Z01 - 0.5*Z10*Z10 - 0.5*Z11*Z11 + 0.5*Z12*Z12 - 0.5*Z20*Z20 - 0.5*Z21*Z21 + 0.5*Z22*Z22 ) + Z22*( W00*Z20 + W01*Z21 + W20*Z00 + W21*Z01 ) + Z12*( W00*Z10 + W01*Z11 + W10*Z00 + W11*Z01 ) + Z01*( W12*Z11 + W22*Z21 ) + Z00*( W12*Z10 + W22*Z20 );
		B.data[38] = W02*( W00*Z00 + W01*Z01 + 1.5*W02*Z02 - W10*Z10 - W11*Z11 + W12*Z12 - W20*Z20 - W21*Z21 + W22*Z22 ) + Z02*( 0.5*W00*W00 + 0.5*W01*W01 - 0.5*W10*W10 - 0.5*W11*W11 + 0.5*W12*W12 - 0.5*W20*W20 - 0.5*W21*W21 + 0.5*W22*W22 ) + W00*( W10*Z12 + W12*Z10 + W20*Z22 + W22*Z20 ) + W01*( W11*Z12 + W12*Z11 + W21*Z22 + W22*Z21 ) + Z01*( W11*W12 + W21*W22 ) + Z00*( W10*W12 + W20*W22 );
		B.data[39] = W02*( 0.5*W00*W00 + 0.5*W01*W01 + 0.5*W02*W02 - 0.5*W10*W10 - 0.5*W11*W11 + 0.5*W12*W12 - 0.5*W20*W20 - 0.5*W21*W21 + 0.5*W22*W22 ) + W00*( W10*W12 + W20*W22 ) + W01*( W11*W12 + W21*W22 );
		B.data[40] = Z10*( X00*Z00 - X01*Z01 - X02*Z02 + 1.5*X10*Z10 + X11*Z11 + X12*Z12 + X20*Z20 - X21*Z21 - X22*Z22 ) + 0.5*( X10*( Z00*Z00 - Z01*Z01 - Z02*Z02 + Z11*Z11 + Z12*Z12 + Z20*Z20 - Z21*Z21 - Z22*Z22 ) ) + Z20*( X11*Z21 + X12*Z22 + X21*Z11 + X22*Z12 ) + Z00*( X01*Z11 + X02*Z12 + X11*Z01 + X12*Z02 ) + Z12*( X00*Z02 + X20*Z22 ) + Z11*( X00*Z01 + X20*Z21 );
		B.data[41] = Z10*( W00*X00 - W01*X01 - W02*X02 + 3.0*W10*X10 + W11*X11 + W12*X12 + W20*X20 - W21*X21 - W22*X22 ) + X10*( W00*Z00 - W01*Z01 - W02*Z02 + W11*Z11 + W12*Z12 + W20*Z20 - W21*Z21 - W22*Z22 ) + W10*( X00*Z00 - X01*Z01 - X02*Z02 + X11*Z11 + X12*Z12 + X20*Z20 - X21*Z21 - X22*Z22 ) + Z20*( W11*X21 + W12*X22 + W21*X11 + W22*X12 ) + Z00*( W01*X11 + W02*X12 + W11*X01 + W12*X02 ) + W20*( X11*Z21 + X12*Z22 + X21*Z11 + X22*Z12 ) + W00*( X01*Z11 + X02*Z12 + X11*Z01 + X12*Z02 ) + X00*( W01*Z11 + W02*Z12 + W11*Z01 + W12*Z02 ) + X20*( W11*Z21 + W12*Z22 + W21*Z11 + W22*Z12 );
		B.data[42] = W10*( W00*X00 - W01*X01 - W02*X02 + 1.5*W10*X10 + W11*X11 + W12*X12 + W20*X20 - W21*X21 - W22*X22 ) + 0.5*( X10*( W00*W00 - W01*W01 - W02*W02 + W11*W11 + W12*W12 + W20*W20 - W21*W21 - W22*W22 ) ) + W00*( W01*X11 + W02*X12 + W11*X01 + W12*X02 ) + W20*( W11*X21 + W12*X22 + W21*X11 + W22*X12 ) + W12*( W02*X00 + W22*X20 ) + W11*( W01*X00 + W21*X20 );
		B.data[43] = Z10*( Y00*Z00 - Y01*Z01 - Y02*Z02 + 1.5*Y10*Z10 + Y11*Z11 + Y12*Z12 + Y20*Z20 - Y21*Z21 - Y22*Z22 ) + 0.5*( Y10*( Z00*Z00 - Z01*Z01 - Z02*Z02 + Z11*Z11 + Z12*Z12 + Z20*Z20 - Z21*Z21 - Z22*Z22 ) ) + Z20*( Y11*Z21 + Y12*Z22 + Y21*Z11 + Y22*Z12 ) + Z00*( Y01*Z11 + Y02*Z12 + Y11*Z01 + Y12*Z02 ) + Y00*( Z01*Z11 + Z02*Z12 ) + Y20*( Z11*Z21 + Z12*Z22 );
		B.data[44] = Z10*( W00*Y00 - W01*Y01 - W02*Y02 + 3.0*W10*Y10 + W11*Y11 + W12*Y12 + W20*Y20 - W21*Y21 - W22*Y22 ) + Y10*( W00*Z00 - W01*Z01 - W02*Z02 + W11*Z11 + W12*Z12 + W20*Z20 - W21*Z21 - W22*Z22 ) + W10*( Y00*Z00 - Y01*Z01 - Y02*Z02 + Y11*Z11 + Y12*Z12 + Y20*Z20 - Y21*Z21 - Y22*Z22 ) + Y00*( W01*Z11 + W02*Z12 + W11*Z01 + W12*Z02 ) + Z20*( W11*Y21 + W12*Y22 + W21*Y11 + W22*Y12 ) + Z00*( W01*Y11 + W02*Y12 + W11*Y01 + W12*Y02 ) + W00*( Y01*Z11 + Y02*Z12 + Y11*Z01 + Y12*Z02 ) + W20*( Y11*Z21 + Y12*Z22 + Y21*Z11 + Y22*Z12 ) + Y20*( W11*Z21 + W12*Z22 + W21*Z11 + W22*Z12 );
		B.data[45] = W10*( W00*Y00 - W01*Y01 - W02*Y02 + 1.5*W10*Y10 + W11*Y11 + W12*Y12 + W20*Y20 - W21*Y21 - W22*Y22 ) + 0.5*( Y10*( W00*W00 - W01*W01 - W02*W02 + W11*W11 + W12*W12 + W20*W20 - W21*W21 - W22*W22 ) ) + W00*( W01*Y11 + W02*Y12 + W11*Y01 + W12*Y02 ) + W20*( W11*Y21 + W12*Y22 + W21*Y11 + W22*Y12 ) + Y00*( W01*W11 + W02*W12 ) + Y20*( W11*W21 + W12*W22 );
		B.data[46] = Z10*( 0.5*Z00*Z00 - 0.5*Z01*Z01 - 0.5*Z02*Z02 + 0.5*Z10*Z10 + 0.5*Z11*Z11 + 0.5*Z12*Z12 + 0.5*Z20*Z20 - 0.5*Z21*Z21 - 0.5*Z22*Z22 ) + Z20*( Z11*Z21 + Z12*Z22 ) + Z00*( Z01*Z11 + Z02*Z12 );
		B.data[47] = Z10*( W00*Z00 - W01*Z01 - W02*Z02 + 1.5*W10*Z10 + W11*Z11 + W12*Z12 + W20*Z20 - W21*Z21 - W22*Z22 ) + 0.5*( W10*( Z00*Z00 - Z01*Z01 - Z02*Z02 + Z11*Z11 + Z12*Z12 + Z20*Z20 - Z21*Z21 - Z22*Z22 ) ) + Z20*( W11*Z21 + W12*Z22 + W21*Z11 + W22*Z12 ) + Z00*( W01*Z11 + W02*Z12 + W11*Z01 + W12*Z02 ) + W00*( Z01*Z11 + Z02*Z12 ) + W20*( Z11*Z21 + Z12*Z22 );
		B.data[48] = W10*( W00*Z00 - W01*Z01 - W02*Z02 + 1.5*W10*Z10 + W11*Z11 + W12*Z12 + W20*Z20 - W21*Z21 - W22*Z22 ) + 0.5*( Z10*( W00*W00 - W01*W01 - W02*W02 + W11*W11 + W12*W12 + W20*W20 - W21*W21 - W22*W22 ) ) + W00*( W01*Z11 + W02*Z12 + W11*Z01 + W12*Z02 ) + W20*( W11*Z21 + W12*Z22 + W21*Z11 + W22*Z12 ) + Z20*( W11*W21 + W12*W22 ) + Z00*( W01*W11 + W02*W12 );
		B.data[49] = W10*( 0.5*W00*W00 - 0.5*W01*W01 - 0.5*W02*W02 + 0.5*W10*W10 + 0.5*W11*W11 + 0.5*W12*W12 + 0.5*W20*W20 - 0.5*W21*W21 - 0.5*W22*W22 ) + W00*( W01*W11 + W02*W12 ) + W20*( W11*W21 + W12*W22 );
		B.data[50] = Z11*( -X00*Z00 + X01*Z01 - X02*Z02 + X10*Z10 + 1.5*X11*Z11 + X12*Z12 - X20*Z20 + X21*Z21 - X22*Z22 ) + 0.5*( X11*( -Z00*Z00 + Z01*Z01 - Z02*Z02 + Z10*Z10 + Z12*Z12 - Z20*Z20 + Z21*Z21 - Z22*Z22 ) ) + Z21*( X10*Z20 + X12*Z22 + X20*Z10 + X22*Z12 ) + Z01*( X00*Z10 + X02*Z12 + X10*Z00 + X12*Z02 ) + Z12*( X01*Z02 + X21*Z22 ) + Z10*( X01*Z00 + X21*Z20 );
		B.data[51] = Z11*( -W00*X00 + W01*X01 - W02*X02 + W10*X10 + 3.0*W11*X11 + W12*X12 - W20*X20 + W21*X21 - W22*X22 ) + X11*( -W00*Z00 + W01*Z01 - W02*Z02 + W10*Z10 + W12*Z12 - W20*Z20 + W21*Z21 - W22*Z22 ) + W11*( -X00*Z00 + X01*Z01 - X02*Z02 + X10*Z10 + X12*Z12 - X20*Z20 + X21*Z21 - X22*Z22 ) + Z21*( W10*X20 + W12*X22 + W20*X10 + W22*X12 ) + Z01*( W00*X10 + W02*X12 + W10*X00 + W12*X02 ) + W21*( X10*Z20 + X12*Z22 + X20*Z10 + X22*Z12 ) + W01*( X00*Z10 + X02*Z12 + X10*Z00 + X12*Z02 ) + X01*( W00*Z10 + W02*Z12 + W10*Z00 + W12*Z02 ) + X21*( W10*Z20 + W12*Z22 + W20*Z10 + W22*Z12 );
		B.data[52] = W11*( -W00*X00 + W01*X01 - W02*X02 + W10*X10 + 1.5*W11*X11 + W12*X12 - W20*X20 + W21*X21 - W22*X22 ) + 0.5*( X11*( -W00*W00 + W01*W01 - W02*W02 + W10*W10 + W12*W12 - W20*W20 + W21*W21 - W22*W22 ) ) + W01*( W00*X10 + W02*X12 + W10*X00 + W12*X02 ) + W21*( W10*X20 + W12*X22 + W20*X10 + W22*X12 ) + W12*( W02*X01 + W22*X21 ) + W10*( W00*X01 + W20*X21 );
		B.data[53] = Z11*( -Y00*Z00 + Y01*Z01 - Y02*Z02 + Y10*Z10 + 1.5*Y11*Z11 + Y12*Z12 - Y20*Z20 + Y21*Z21 - Y22*Z22 ) + 0.5*( Y11*( -Z00*Z00 + Z01*Z01 - Z02*Z02 + Z10*Z10 + Z12*Z12 - Z20*Z20 + Z21*Z21 - Z22*Z22 ) ) + Z21*( Y10*Z20 + Y12*Z22 + Y20*Z10 + Y22*Z12 ) + Z01*( Y00*Z10 + Y02*Z12 + Y10*Z00 + Y12*Z02 ) + Y01*( Z00*Z10 + Z02*Z12 ) + Y21*( Z10*Z20 + Z12*Z22 );
		B.data[54] = Z11*( -W00*Y00 + W01*Y01 - W02*Y02 + W10*Y10 + 3.0*W11*Y11 + W12*Y12 - W20*Y20 + W21*Y21 - W22*Y22 ) + Y11*( -W00*Z00 + W01*Z01 - W02*Z02 + W10*Z10 + W12*Z12 - W20*Z20 + W21*Z21 - W22*Z22 ) + W11*( -Y00*Z00 + Y01*Z01 - Y02*Z02 + Y10*Z10 + Y12*Z12 - Y20*Z20 + Y21*Z21 - Y22*Z22 ) + Y01*( W00*Z10 + W02*Z12 + W10*Z00 + W12*Z02 ) + Z21*( W10*Y20 + W12*Y22 + W20*Y10 + W22*Y12 ) + Z01*( W00*Y10 + W02*Y12 + W10*Y00 + W12*Y02 ) + W21*( Y10*Z20 + Y12*Z22 + Y20*Z10 + Y22*Z12 ) + W01*( Y00*Z10 + Y02*Z12 + Y10*Z00 + Y12*Z02 ) + Y21*( W10*Z20 + W12*Z22 + W20*Z10 + W22*Z12 );
		B.data[55] = W11*( -W00*Y00 + W01*Y01 - W02*Y02 + W10*Y10 + 1.5*W11*Y11 + W12*Y12 - W20*Y20 + W21*Y21 - W22*Y22 ) + 0.5*( Y11*( -W00*W00 + W01*W01 - W02*W02 + W10*W10 + W12*W12 - W20*W20 + W21*W21 - W22*W22 ) ) + W01*( W00*Y10 + W02*Y12 + W10*Y00 + W12*Y02 ) + W21*( W10*Y20 + W12*Y22 + W20*Y10 + W22*Y12 ) + Y01*( W00*W10 + W02*W12 ) + Y21*( W10*W20 + W12*W22 );
		B.data[56] = Z11*( -0.5*Z00*Z00 + 0.5*Z01*Z01 - 0.5*Z02*Z02 + 0.5*Z10*Z10 + 0.5*Z11*Z11 + 0.5*Z12*Z12 - 0.5*Z20*Z20 + 0.5*Z21*Z21 - 0.5*Z22*Z22 ) + Z21*( Z10*Z20 + Z12*Z22 ) + Z01*( Z00*Z10 + Z02*Z12 );
		B.data[57] = Z11*( -W00*Z00 + W01*Z01 - W02*Z02 + W10*Z10 + 1.5*W11*Z11 + W12*Z12 - W20*Z20 + W21*Z21 - W22*Z22 ) + 0.5*( W11*( -Z00*Z00 + Z01*Z01 - Z02*Z02 + Z10*Z10 + Z12*Z12 - Z20*Z20 + Z21*Z21 - Z22*Z22 ) ) + Z12*( W01*Z02 + W02*Z01 + W21*Z22 + W22*Z21 ) + Z10*( W00*Z01 + W01*Z00 + W20*Z21 + W21*Z20 ) + Z21*( W10*Z20 + W12*Z22 ) + Z01*( W10*Z00 + W12*Z02 );
		B.data[58] = W11*( -W00*Z00 + W01*Z01 - W02*Z02 + W10*Z10 + 1.5*W11*Z11 + W12*Z12 - W20*Z20 + W21*Z21 - W22*Z22 ) + 0.5*( Z11*( -W00*W00 + W01*W01 - W02*W02 + W10*W10 + W12*W12 - W20*W20 + W21*W21 - W22*W22 ) ) + W21*( W10*Z20 + W12*Z22 + W20*Z10 + W22*Z12 ) + W01*( W00*Z10 + W02*Z12 + W10*Z00 + W12*Z02 ) + Z21*( W10*W20 + W12*W22 ) + Z01*( W00*W10 + W02*W12 );
		B.data[59] = W11*( -0.5*W00*W00 + 0.5*W01*W01 - 0.5*W02*W02 + 0.5*W10*W10 + 0.5*W11*W11 + 0.5*W12*W12 - 0.5*W20*W20 + 0.5*W21*W21 - 0.5*W22*W22 ) + W12*( W01*W02 + W21*W22 ) + W10*( W00*W01 + W20*W21 );
		B.data[60] = Z12*( -X00*Z00 - X01*Z01 + X02*Z02 + X10*Z10 + X11*Z11 + 1.5*X12*Z12 - X20*Z20 - X21*Z21 + X22*Z22 ) + 0.5*( X12*( -Z00*Z00 - Z01*Z01 + Z02*Z02 + Z10*Z10 + Z11*Z11 - Z20*Z20 - Z21*Z21 + Z22*Z22 ) ) + Z22*( X10*Z20 + X11*Z21 + X20*Z10 + X21*Z11 ) + Z02*( X00*Z10 + X01*Z11 + X10*Z00 + X11*Z01 ) + Z10*( X02*Z00 + X22*Z20 ) + Z11*( X02*Z01 + X22*Z21 );
		B.data[61] = Z12*( -W00*X00 - W01*X01 + W02*X02 + W10*X10 + W11*X11 + 3.0*W12*X12 - W20*X20 - W21*X21 + W22*X22 ) + X12*( -W00*Z00 - W01*Z01 + W02*Z02 + W10*Z10 + W11*Z11 - W20*Z20 - W21*Z21 + W22*Z22 ) + W12*( -X00*Z00 - X01*Z01 + X02*Z02 + X10*Z10 + X11*Z11 - X20*Z20 - X21*Z21 + X22*Z22 ) + Z22*( W10*X20 + W11*X21 + W20*X10 + W21*X11 ) + Z02*( W00*X10 + W01*X11 + W10*X00 + W11*X01 ) + W22*( X10*Z20 + X11*Z21 + X20*Z10 + X21*Z11 ) + W02*( X00*Z10 + X01*Z11 + X10*Z00 + X11*Z01 ) + X02*( W00*Z10 + W01*Z11 + W10*Z00 + W11*Z01 ) + X22*( W10*Z20 + W11*Z21 + W20*Z10 + W21*Z11 );
		B.data[62] = W12*( -W00*X00 - W01*X01 + W02*X02 + W10*X10 + W11*X11 + 1.5*W12*X12 - W20*X20 - W21*X21 + W22*X22 ) + 0.5*( X12*( -W00*W00 - W01*W01 + W02*W02 + W10*W10 + W11*W11 - W20*W20 - W21*W21 + W22*W22 ) ) + W02*( W00*X10 + W01*X11 + W10*X00 + W11*X01 ) + W22*( W10*X20 + W11*X21 + W20*X10 + W21*X11 ) + W11*( W01*X02 + W21*X22 ) + W10*( W00*X02 + W20*X22 );
		B.data[63] = Z12*( -Y00*Z00 - Y01*Z01 + Y02*Z02 + Y10*Z10 + Y11*Z11 + 1.5*Y12*Z12 - Y20*Z20 - Y21*Z21 + Y22*Z22 ) + 0.5*( Y12*( -Z00*Z00 - Z01*Z01 + Z02*Z02 + Z10*Z10 + Z11*Z11 - Z20*Z20 - Z21*Z21 + Z22*Z22 ) ) + Z22*( Y10*Z20 + Y11*Z21 + Y20*Z10 + Y21*Z11 ) + Z02*( Y00*Z10 + Y01*Z11 + Y10*Z00 + Y11*Z01 ) + Y02*( Z00*Z10 + Z01*Z11 ) + Y22*( Z10*Z20 + Z11*Z21 );
		B.data[64] = Z12*( -W00*Y00 - W01*Y01 + W02*Y02 + W10*Y10 + W11*Y11 + 3.0*W12*Y12 - W20*Y20 - W21*Y21 + W22*Y22 ) + Y12*( -W00*Z00 - W01*Z01 + W02*Z02 + W10*Z10 + W11*Z11 - W20*Z20 - W21*Z21 + W22*Z22 ) + W12*( -Y00*Z00 - Y01*Z01 + Y02*Z02 + Y10*Z10 + Y11*Z11 - Y20*Z20 - Y21*Z21 + Y22*Z22 ) + Y02*( W00*Z10 + W01*Z11 + W10*Z00 + W11*Z01 ) + Z22*( W10*Y20 + W11*Y21 + W20*Y10 + W21*Y11 ) + Z02*( W00*Y10 + W01*Y11 + W10*Y00 + W11*Y01 ) + W22*( Y10*Z20 + Y11*Z21 + Y20*Z10 + Y21*Z11 ) + W02*( Y00*Z10 + Y01*Z11 + Y10*Z00 + Y11*Z01 ) + Y22*( W10*Z20 + W11*Z21 + W20*Z10 + W21*Z11 );
		B.data[65] = W12*( -W00*Y00 - W01*Y01 + W02*Y02 + W10*Y10 + W11*Y11 + 1.5*W12*Y12 - W20*Y20 - W21*Y21 + W22*Y22 ) + 0.5*( Y12*( -W00*W00 - W01*W01 + W02*W02 + W10*W10 + W11*W11 - W20*W20 - W21*W21 + W22*W22 ) ) + W02*( W00*Y10 + W01*Y11 + W10*Y00 + W11*Y01 ) + W22*( W10*Y20 + W11*Y21 + W20*Y10 + W21*Y11 ) + Y02*( W00*W10 + W01*W11 ) + Y22*( W10*W20 + W11*W21 );
		B.data[66] = Z12*( -0.5*Z00*Z00 - 0.5*Z01*Z01 + 0.5*Z02*Z02 + 0.5*Z10*Z10 + 0.5*Z11*Z11 + 0.5*Z12*Z12 - 0.5*Z20*Z20 - 0.5*Z21*Z21 + 0.5*Z22*Z22 ) + Z22*( Z10*Z20 + Z11*Z21 ) + Z02*( Z00*Z10 + Z01*Z11 );
		B.data[67] = Z12*( -W00*Z00 - W01*Z01 + W02*Z02 + W10*Z10 + W11*Z11 + 1.5*W12*Z12 - W20*Z20 - W21*Z21 + W22*Z22 ) + 0.5*( W12*( -Z00*Z00 - Z01*Z01 + Z02*Z02 + Z10*Z10 + Z11*Z11 - Z20*Z20 - Z21*Z21 + Z22*Z22 ) ) + Z22*( W10*Z20 + W11*Z21 + W20*Z10 + W21*Z11 ) + Z02*( W00*Z10 + W01*Z11 + W10*Z00 + W11*Z01 ) + Z10*( W02*Z00 + W22*Z20 ) + Z11*( W02*Z01 + W22*Z21 );
		B.data[68] = W12*( -W00*Z00 - W01*Z01 + W02*Z02 + W10*Z10 + W11*Z11 + 1.5*W12*Z12 - W20*Z20 - W21*Z21 + W22*Z22 ) + Z12*( -0.5*W00*W00 - 0.5*W01*W01 + 0.5*W02*W02 + 0.5*W10*W10 + 0.5*W11*W11 - 0.5*W20*W20 - 0.5*W21*W21 + 0.5*W22*W22 ) + W10*( W00*Z02 + W02*Z00 + W20*Z22 + W22*Z20 ) + W11*( W01*Z02 + W02*Z01 + W21*Z22 + W22*Z21 ) + Z10*( W00*W02 + W20*W22 ) + Z11*( W01*W02 + W21*W22 );
		B.data[69] = W12*( -0.5*W00*W00 - 0.5*W01*W01 + 0.5*W02*W02 + 0.5*W10*W10 + 0.5*W11*W11 + 0.5*W12*W12 - 0.5*W20*W20 - 0.5*W21*W21 + 0.5*W22*W22 ) + W11*( W01*W02 + W21*W22 ) + W10*( W00*W02 + W20*W22 );
		B.data[70] = Z20*( X00*Z00 - X01*Z01 - X02*Z02 + X10*Z10 - X11*Z11 - X12*Z12 + 1.5*X20*Z20 + X21*Z21 + X22*Z22 ) + 0.5*( X20*( Z00*Z00 - Z01*Z01 - Z02*Z02 + Z10*Z10 - Z11*Z11 - Z12*Z12 + Z21*Z21 + Z22*Z22 ) ) + Z22*( X00*Z02 + X02*Z00 + X10*Z12 + X12*Z10 ) + Z21*( X00*Z01 + X01*Z00 + X10*Z11 + X11*Z10 ) + Z00*( X21*Z01 + X22*Z02 ) + Z10*( X21*Z11 + X22*Z12 );
		B.data[71] = Z20*( W00*X00 - W01*X01 - W02*X02 + W10*X10 - W11*X11 - W12*X12 + 3.0*W20*X20 + W21*X21 + W22*X22 ) + W20*( X00*Z00 - X01*Z01 - X02*Z02 + X10*Z10 - X11*Z11 - X12*Z12 + X21*Z21 + X22*Z22 ) + X20*( W00*Z00 - W01*Z01 - W02*Z02 + W10*Z10 - W11*Z11 - W12*Z12 + W21*Z21 + W22*Z22 ) + Z22*( W00*X02 + W02*X00 + W10*X12 + W12*X10 ) + Z21*( W00*X01 + W01*X00 + W10*X11 + W11*X10 ) + Z00*( W01*X21 + W02*X22 + W21*X01 + W22*X02 ) + Z10*( W11*X21 + W12*X22 + W21*X11 + W22*X12 ) + X00*( W21*Z01 + W22*Z02 ) + Z12*( W10*X22 + W22*X10 ) + W00*( X21*Z01 + X22*Z02 ) + Z11*( W10*X21 + W21*X10 );
		B.data[72] = W20*( W00*X00 - W01*X01 - W02*X02 + W10*X10 - W11*X11 - W12*X12 + 1.5*W20*X20 + W21*X21 + W22*X22 ) + 0.5*( X20*( W00*W00 - W01*W01 - W02*W02 + W10*W10 - W11*W11 - W12*W12 + W21*W21 + W22*W22 ) ) + W00*( W01*X21 + W02*X22 + W21*X01 + W22*X02 ) + W10*( W11*X21 + W12*X22 + W21*X11 + W22*X12 ) + W22*( W02*X00 + W12*X10 ) + W21*( W01*X00 + W11*X10 );
		B.data[73] = Z20*( Y00*Z00 - Y01*Z01 - Y02*Z02 + Y10*Z10 - Y11*Z11 - Y12*Z12 + 1.5*Y20*Z20 + Y21*Z21 + Y22*Z22 ) + Y20*( 0.5*Z00*Z00 - 0.5*Z01*Z01 - 0.5*Z02*Z02 + 0.5*Z10*Z10 - 0.5*Z11*Z11 - 0.5*Z12*Z12 + 0.5*Z21*Z21 + 0.5*Z22*Z22 ) + Z22*( Y00*Z02 + Y02*Z00 + Y10*Z12 + Y12*Z10 ) + Z21*( Y00*Z01 + Y01*Z00 + Y10*Z11 + Y11*Z10 ) + Z00*( Y21*Z01 + Y22*Z02 ) + Z10*( Y21*Z11 + Y22*Z12 );
		B.data[74] = Z20*( W00*Y00 - W01*Y01 - W02*Y02 + W10*Y10 - W11*Y11 - W12*Y12 + 3.0*W20*Y20 + W21*Y21 + W22*Y22 ) + W20*( Y00*Z00 - Y01*Z01 - Y02*Z02 + Y10*Z10 - Y11*Z11 - Y12*Z12 + Y21*Z21 + Y22*Z22 ) + Y20*( W00*Z00 - W01*Z01 - W02*Z02 + W10*Z10 - W11*Z11 - W12*Z12 + W21*Z21 + W22*Z22 ) + Y00*( W01*Z21 + W02*Z22 + W21*Z01 + W22*Z02 ) + Z00*( W01*Y21 + W02*Y22 + W21*Y01 + W22*Y02 ) + W00*( Y01*Z21 + Y02*Z22 + Y21*Z01 + Y22*Z02 ) + W10*( Y11*Z21 + Y12*Z22 + Y21*Z11 + Y22*Z12 ) + Z10*( W11*Y21 + W12*Y22 + W21*Y11 + W22*Y12 ) + Y10*( W11*Z21 + W12*Z22 + W21*Z11 + W22*Z12 );
		B.data[75] = W20*( W00*Y00 - W01*Y01 - W02*Y02 + W10*Y10 - W11*Y11 - W12*Y12 + 1.5*W20*Y20 + W21*Y21 + W22*Y22 ) + 0.5*( Y20*( W00*W00 - W01*W01 - W02*W02 + W10*W10 - W11*W11 - W12*W12 + W21*W21 + W22*W22 ) ) + W00*( W01*Y21 + W02*Y22 + W21*Y01 + W22*Y02 ) + W10*( W11*Y21 + W12*Y22 + W21*Y11 + W22*Y12 ) + Y00*( W01*W21 + W02*W22 ) + Y10*( W11*W21 + W12*W22 );
		B.data[76] = Z20*( 0.5*Z00*Z00 - 0.5*Z01*Z01 - 0.5*Z02*Z02 + 0.5*Z10*Z10 - 0.5*Z11*Z11 - 0.5*Z12*Z12 + 0.5*Z20*Z20 + 0.5*Z21*Z21 + 0.5*Z22*Z22 ) + Z22*( Z00*Z02 + Z10*Z12 ) + Z21*( Z00*Z01 + Z10*Z11 );
		B.data[77] = Z20*( W00*Z00 - W01*Z01 - W02*Z02 + W10*Z10 - W11*Z11 - W12*Z12 + 1.5*W20*Z20 + W21*Z21 + W22*Z22 ) + W20*( 0.5*Z00*Z00 - 0.5*Z01*Z01 - 0.5*Z02*Z02 + 0.5*Z10*Z10 - 0.5*Z11*Z11 - 0.5*Z12*Z12 + 0.5*Z21*Z21 + 0.5*Z22*Z22 ) + Z22*( W00*Z02 + W02*Z00 + W10*Z12 + W12*Z10 ) + Z21*( W00*Z01 + W01*Z00 + W10*Z11 + W11*Z10 ) + Z00*( W21*Z01 + W22*Z02 ) + Z10*( W21*Z11 + W22*Z12 );
		B.data[78] = W20*( W00*Z00 - W01*Z01 - W02*Z02 + W10*Z10 - W11*Z11 - W12*Z12 + 1.5*W20*Z20 + W21*Z21 + W22*Z22 ) + Z20*( 0.5*W00*W00 - 0.5*W01*W01 - 0.5*W02*W02 + 0.5*W10*W10 - 0.5*W11*W11 - 0.5*W12*W12 + 0.5*W21*W21 + 0.5*W22*W22 ) + W00*( W01*Z21 + W02*Z22 + W21*Z01 + W22*Z02 ) + W10*( W11*Z21 + W12*Z22 + W21*Z11 + W22*Z12 ) + Z10*( W11*W21 + W12*W22 ) + Z00*( W01*W21 + W02*W22 );
		B.data[79] = W20*( 0.5*W00*W00 - 0.5*W01*W01 - 0.5*W02*W02 + 0.5*W10*W10 - 0.5*W11*W11 - 0.5*W12*W12 + 0.5*W20*W20 + 0.5*W21*W21 + 0.5*W22*W22 ) + W00*( W01*W21 + W02*W22 ) + W10*( W11*W21 + W12*W22 );
		B.data[80] = Z21*( -X00*Z00 + X01*Z01 - X02*Z02 - X10*Z10 + X11*Z11 - X12*Z12 + X20*Z20 + 1.5*X21*Z21 + X22*Z22 ) + 0.5*( X21*( -Z00*Z00 + Z01*Z01 - Z02*Z02 - Z10*Z10 + Z11*Z11 - Z12*Z12 + Z20*Z20 + Z22*Z22 ) ) + Z22*( X01*Z02 + X02*Z01 + X11*Z12 + X12*Z11 ) + Z20*( X00*Z01 + X01*Z00 + X10*Z11 + X11*Z10 ) + Z01*( X20*Z00 + X22*Z02 ) + Z11*( X20*Z10 + X22*Z12 );
		B.data[81] = Z21*( -W00*X00 + W01*X01 - W02*X02 - W10*X10 + W11*X11 - W12*X12 + W20*X20 + 3.0*W21*X21 + W22*X22 ) + W21*( -X00*Z00 + X01*Z01 - X02*Z02 - X10*Z10 + X11*Z11 - X12*Z12 + X20*Z20 + X22*Z22 ) + X21*( -W00*Z00 + W01*Z01 - W02*Z02 - W10*Z10 + W11*Z11 - W12*Z12 + W20*Z20 + W22*Z22 ) + Z22*( W01*X02 + W02*X01 + W11*X12 + W12*X11 ) + Z20*( W00*X01 + W01*X00 + W10*X11 + W11*X10 ) + Z01*( W00*X20 + W02*X22 + W20*X00 + W22*X02 ) + Z11*( W10*X20 + W12*X22 + W20*X10 + W22*X12 ) + Z00*( W01*X20 + W20*X01 ) + Z02*( W01*X22 + W22*X01 ) + Z12*( W11*X22 + W22*X11 ) + Z10*( W11*X20 + W20*X11 );
		B.data[82] = W21*( -W00*X00 + W01*X01 - W02*X02 - W10*X10 + W11*X11 - W12*X12 + W20*X20 + 1.5*W21*X21 + W22*X22 ) + 0.5*( X21*( -W00*W00 + W01*W01 - W02*W02 - W10*W10 + W11*W11 - W12*W12 + W20*W20 + W22*W22 ) ) + W01*( W00*X20 + W02*X22 + W20*X00 + W22*X02 ) + W11*( W10*X20 + W12*X22 + W20*X10 + W22*X12 ) + W22*( W02*X01 + W12*X11 ) + W20*( W00*X01 + W10*X11 );
		B.data[83] = Z21*( -Y00*Z00 + Y01*Z01 - Y02*Z02 - Y10*Z10 + Y11*Z11 - Y12*Z12 + Y20*Z20 + 1.5*Y21*Z21 + Y22*Z22 ) + Y21*( -0.5*Z00*Z00 + 0.5*Z01*Z01 - 0.5*Z02*Z02 - 0.5*Z10*Z10 + 0.5*Z11*Z11 - 0.5*Z12*Z12 + 0.5*Z20*Z20 + 0.5*Z22*Z22 ) + Z22*( Y01*Z02 + Y02*Z01 + Y11*Z12 + Y12*Z11 ) + Z20*( Y00*Z01 + Y01*Z00 + Y10*Z11 + Y11*Z10 ) + Z01*( Y20*Z00 + Y22*Z02 ) + Z11*( Y20*Z10 + Y22*Z12 );
		B.data[84] = Z21*( -W00*Y00 + W01*Y01 - W02*Y02 - W10*Y10 + W11*Y11 - W12*Y12 + W20*Y20 + 3.0*W21*Y21 + W22*Y22 ) + W21*( -Y00*Z00 + Y01*Z01 - Y02*Z02 - Y10*Z10 + Y11*Z11 - Y12*Z12 + Y20*Z20 + Y22*Z22 ) + Y21*( -W00*Z00 + W01*Z01 - W02*Z02 - W10*Z10 + W11*Z11 - W12*Z12 + W20*Z20 + W22*Z22 ) + Y01*( W00*Z20 + W02*Z22 + W20*Z00 + W22*Z02 ) + Z01*( W00*Y20 + W02*Y22 + W20*Y00 + W22*Y02 ) + W01*( Y00*Z20 + Y02*Z22 + Y20*Z00 + Y22*Z02 ) + W11*( Y10*Z20 + Y12*Z22 + Y20*Z10 + Y22*Z12 ) + Z11*( W10*Y20 + W12*Y22 + W20*Y10 + W22*Y12 ) + Y11*( W10*Z20 + W12*Z22 + W20*Z10 + W22*Z12 );
		B.data[85] = W21*( -W00*Y00 + W01*Y01 - W02*Y02 - W10*Y10 + W11*Y11 - W12*Y12 + W20*Y20 + 1.5*W21*Y21 + W22*Y22 ) + 0.5*( Y21*( -W00*W00 + W01*W01 - W02*W02 - W10*W10 + W11*W11 - W12*W12 + W20*W20 + W22*W22 ) ) + W01*( W00*Y20 + W02*Y22 + W20*Y00 + W22*Y02 ) + W11*( W10*Y20 + W12*Y22 + W20*Y10 + W22*Y12 ) + Y01*( W00*W20 + W02*W22 ) + Y11*( W10*W20 + W12*W22 );
		B.data[86] = Z21*( -0.5*Z00*Z00 + 0.5*Z01*Z01 - 0.5*Z02*Z02 - 0.5*Z10*Z10 + 0.5*Z11*Z11 - 0.5*Z12*Z12 + 0.5*Z20*Z20 + 0.5*Z21*Z21 + 0.5*Z22*Z22 ) + Z22*( Z01*Z02 + Z11*Z12 ) + Z20*( Z00*Z01 + Z10*Z11 );
		B.data[87] = Z21*( -W00*Z00 + W01*Z01 - W02*Z02 - W10*Z10 + W11*Z11 - W12*Z12 + W20*Z20 + 1.5*W21*Z21 + W22*Z22 ) + W21*( -0.5*Z00*Z00 + 0.5*Z01*Z01 - 0.5*Z02*Z02 - 0.5*Z10*Z10 + 0.5*Z11*Z11 - 0.5*Z12*Z12 + 0.5*Z20*Z20 + 0.5*Z22*Z22 ) + Z22*( W01*Z02 + W02*Z01 + W11*Z12 + W12*Z11 ) + Z20*( W00*Z01 + W01*Z00 + W10*Z11 + W11*Z10 ) + Z01*( W20*Z00 + W22*Z02 ) + Z11*( W20*Z10 + W22*Z12 );
		B.data[88] = W21*( -W00*Z00 + W01*Z01 - W02*Z02 - W10*Z10 + W11*Z11 - W12*Z12 + W20*Z20 + 1.5*W21*Z21 + W22*Z22 ) + Z21*( -0.5*W00*W00 + 0.5*W01*W01 - 0.5*W02*W02 - 0.5*W10*W10 + 0.5*W11*W11 - 0.5*W12*W12 + 0.5*W20*W20 + 0.5*W22*W22 ) + W01*( W00*Z20 + W02*Z22 + W20*Z00 + W22*Z02 ) + W11*( W10*Z20 + W12*Z22 + W20*Z10 + W22*Z12 ) + Z01*( W00*W20 + W02*W22 ) + Z11*( W10*W20 + W12*W22 );
		B.data[89] = W21*( -0.5*W00*W00 + 0.5*W01*W01 - 0.5*W02*W02 - 0.5*W10*W10 + 0.5*W11*W11 - 0.5*W12*W12 + 0.5*W20*W20 + 0.5*W21*W21 + 0.5*W22*W22 ) + W01*( W00*W20 + W02*W22 ) + W11*( W10*W20 + W12*W22 );
		B.data[90] = Z22*( -X00*Z00 - X01*Z01 + X02*Z02 - X10*Z10 - X11*Z11 + X12*Z12 + X20*Z20 + X21*Z21 + 1.5*X22*Z22 ) + 0.5*( X22*( -Z00*Z00 - Z01*Z01 + Z02*Z02 - Z10*Z10 - Z11*Z11 + Z12*Z12 + Z20*Z20 + Z21*Z21 ) ) + Z21*( X01*Z02 + X02*Z01 + X11*Z12 + X12*Z11 ) + Z20*( X00*Z02 + X02*Z00 + X10*Z12 + X12*Z10 ) + Z02*( X20*Z00 + X21*Z01 ) + Z12*( X20*Z10 + X21*Z11 );
		B.data[91] = Z22*( -W00*X00 - W01*X01 + W02*X02 - W10*X10 - W11*X11 + W12*X12 + W20*X20 + W21*X21 + 3.0*W22*X22 ) + W22*( -X00*Z00 - X01*Z01 + X02*Z02 - X10*Z10 - X11*Z11 + X12*Z12 + X20*Z20 + X21*Z21 ) + X22*( -W00*Z00 - W01*Z01 + W02*Z02 - W10*Z10 - W11*Z11 + W12*Z12 + W20*Z20 + W21*Z21 ) + Z21*( W01*X02 + W02*X01 + W11*X12 + W12*X11 ) + Z20*( W00*X02 + W02*X00 + W10*X12 + W12*X10 ) + Z02*( W00*X20 + W01*X21 + W20*X00 + W21*X01 ) + Z12*( W10*X20 + W11*X21 + W20*X10 + W21*X11 ) + Z01*( W02*X21 + W21*X02 ) + Z00*( W02*X20 + W20*X02 ) + W12*( X20*Z10 + X21*Z11 ) + X12*( W20*Z10 + W21*Z11 );
		B.data[92] = W22*( -W00*X00 - W01*X01 + W02*X02 - W10*X10 - W11*X11 + W12*X12 + W20*X20 + W21*X21 + 1.5*W22*X22 ) + 0.5*( X22*( -W00*W00 - W01*W01 + W02*W02 - W10*W10 - W11*W11 + W12*W12 + W20*W20 + W21*W21 ) ) + W02*( W00*X20 + W01*X21 + W20*X00 + W21*X01 ) + W12*( W10*X20 + W11*X21 + W20*X10 + W21*X11 ) + W20*( W00*X02 + W10*X12 ) + W21*( W01*X02 + W11*X12 );
		B.data[93] = Z22*( -Y00*Z00 - Y01*Z01 + Y02*Z02 - Y10*Z10 - Y11*Z11 + Y12*Z12 + Y20*Z20 + Y21*Z21 + 1.5*Y22*Z22 ) + Y22*( -0.5*Z00*Z00 - 0.5*Z01*Z01 + 0.5*Z02*Z02 - 0.5*Z10*Z10 - 0.5*Z11*Z11 + 0.5*Z12*Z12 + 0.5*Z20*Z20 + 0.5*Z21*Z21 ) + Z21*( Y01*Z02 + Y02*Z01 + Y11*Z12 + Y12*Z11 ) + Z20*( Y00*Z02 + Y02*Z00 + Y10*Z12 + Y12*Z10 ) + Z02*( Y20*Z00 + Y21*Z01 ) + Z12*( Y20*Z10 + Y21*Z11 );
		B.data[94] = Z22*( -W00*Y00 - W01*Y01 + W02*Y02 - W10*Y10 - W11*Y11 + W12*Y12 + W20*Y20 + W21*Y21 + 3.0*W22*Y22 ) + W22*( -Y00*Z00 - Y01*Z01 + Y02*Z02 - Y10*Z10 - Y11*Z11 + Y12*Z12 + Y20*Z20 + Y21*Z21 ) + Y22*( -W00*Z00 - W01*Z01 + W02*Z02 - W10*Z10 - W11*Z11 + W12*Z12 + W20*Z20 + W21*Z21 ) + Y02*( W00*Z20 + W01*Z21 + W20*Z00 + W21*Z01 ) + Z02*( W00*Y20 + W01*Y21 + W20*Y00 + W21*Y01 ) + Z12*( W10*Y20 + W11*Y21 + W20*Y10 + W21*Y11 ) + W12*( Y10*Z20 + Y11*Z21 + Y20*Z10 + Y21*Z11 ) + W02*( Y00*Z20 + Y01*Z21 + Y20*Z00 + Y21*Z01 ) + Y12*( W10*Z20 + W11*Z21 + W20*Z10 + W21*Z11 );
		B.data[95] = W22*( -W00*Y00 - W01*Y01 + W02*Y02 - W10*Y10 - W11*Y11 + W12*Y12 + W20*Y20 + W21*Y21 + 1.5*W22*Y22 ) + 0.5*( Y22*( -W00*W00 - W01*W01 + W02*W02 - W10*W10 - W11*W11 + W12*W12 + W20*W20 + W21*W21 ) ) + W02*( W00*Y20 + W01*Y21 + W20*Y00 + W21*Y01 ) + W12*( W10*Y20 + W11*Y21 + W20*Y10 + W21*Y11 ) + Y02*( W00*W20 + W01*W21 ) + Y12*( W10*W20 + W11*W21 );
		B.data[96] = Z22*( -0.5*Z00*Z00 - 0.5*Z01*Z01 + 0.5*Z02*Z02 - 0.5*Z10*Z10 - 0.5*Z11*Z11 + 0.5*Z12*Z12 + 0.5*Z20*Z20 + 0.5*Z21*Z21 + 0.5*Z22*Z22 ) + Z21*( Z01*Z02 + Z11*Z12 ) + Z20*( Z00*Z02 + Z10*Z12 );
		B.data[97] = Z22*( -W00*Z00 - W01*Z01 + W02*Z02 - W10*Z10 - W11*Z11 + W12*Z12 + W20*Z20 + W21*Z21 + 1.5*W22*Z22 ) + W22*( -0.5*Z00*Z00 - 0.5*Z01*Z01 + 0.5*Z02*Z02 - 0.5*Z10*Z10 - 0.5*Z11*Z11 + 0.5*Z12*Z12 + 0.5*Z20*Z20 + 0.5*Z21*Z21 ) + Z12*( W10*Z20 + W11*Z21 + W20*Z10 + W21*Z11 ) + Z02*( W00*Z20 + W01*Z21 + W20*Z00 + W21*Z01 ) + Z21*( W02*Z01 + W12*Z11 ) + Z20*( W02*Z00 + W12*Z10 );
		B.data[98] = W22*( -W00*Z00 - W01*Z01 + W02*Z02 - W10*Z10 - W11*Z11 + W12*Z12 + W20*Z20 + W21*Z21 + 1.5*W22*Z22 ) + Z22*( -0.5*W00*W00 - 0.5*W01*W01 + 0.5*W02*W02 - 0.5*W10*W10 - 0.5*W11*W11 + 0.5*W12*W12 + 0.5*W20*W20 + 0.5*W21*W21 ) + W02*( W00*Z20 + W01*Z21 + W20*Z00 + W21*Z01 ) + W12*( W10*Z20 + W11*Z21 + W20*Z10 + W21*Z11 ) + Z12*( W10*W20 + W11*W21 ) + Z02*( W00*W20 + W01*W21 );
		B.data[99] = W22*( -0.5*W00*W00 - 0.5*W01*W01 + 0.5*W02*W02 - 0.5*W10*W10 - 0.5*W11*W11 + 0.5*W12*W12 + 0.5*W20*W20 + 0.5*W21*W21 + 0.5*W22*W22 ) + W02*( W00*W20 + W01*W21 ) + W12*( W10*W20 + W11*W21 );
	}

	/**
	 * See equations 11-13
	 *
	 * K = E - z*F
	 * L = G - z*H
	 * M = I - z*J
	 *
	 * (E,F,G,H,I,J) = x*(z^2 + z + 1) +  y*(z^2 + z + 1) + z^3 + z^2 + z + 1
	 *
	 * @param A Solution to linear system
	 */
	public void setDeterminantVectors( DenseMatrix64F A ) {
		int r = A.numCols*4;
		int t = A.numCols*5;

		K00  =             - A.data[t++];
		K01  = A.data[r++] - A.data[t++];
		K02  = A.data[r++] - A.data[t++];
		K03  = A.data[r++];
		K04  =             - A.data[t++];
		K05  = A.data[r++] - A.data[t++];
		K06  = A.data[r++] - A.data[t++];
		K07  = A.data[r++];
		K08  =             - A.data[t++];
		K09  = A.data[r++] - A.data[t++];
		K10 = A.data[r++] - A.data[t++];
		K11 = A.data[r++] - A.data[t];
		K12 = A.data[r];

		r = A.numCols*6;
		t = A.numCols*7;
		L00  =             - A.data[t++];
		L01  = A.data[r++] - A.data[t++];
		L02  = A.data[r++] - A.data[t++];
		L03  = A.data[r++];
		L04  =             - A.data[t++];
		L05  = A.data[r++] - A.data[t++];
		L06  = A.data[r++] - A.data[t++];
		L07  = A.data[r++];
		L08  =             - A.data[t++];
		L09  = A.data[r++] - A.data[t++];
		L10 = A.data[r++] - A.data[t++];
		L11 = A.data[r++] - A.data[t];
		L12 = A.data[r];

		r = A.numCols*8;
		t = A.numCols*9;
		M00  =             - A.data[t++];
		M01  = A.data[r++] - A.data[t++];
		M02  = A.data[r++] - A.data[t++];
		M03  = A.data[r++];
		M04  =             - A.data[t++];
		M05  = A.data[r++] - A.data[t++];
		M06  = A.data[r++] - A.data[t++];
		M07  = A.data[r++];
		M08  =             - A.data[t++];
		M09  = A.data[r++] - A.data[t++];
		M10 = A.data[r++] - A.data[t++];
		M11 = A.data[r++] - A.data[t];
		M12 = A.data[r];
	}

	/**
	 * Extracts the polynomial coefficients from det(B) where B = [K;L;M] and the columns are variables (x,y,1)
	 * see paper for a slightly less confusing description.
	 *
	 * polynomial = coef[0] + z*coef[1] + z^2*coef[2] + z^3*coef[3] ... etc
	 *
	 * @param coefs Array with 11 elements. The index corresponds to the coefficient power.
	 */
	public void extractPolynomial( double coefs[] ) {
		coefs[0] = K12*( L03*M07 - L07*M03 ) + K03*( L07*M12 - L12*M07 ) + K07*( -L03*M12 + L12*M03 );
		coefs[1] = K03*( L06*M12 + L07*M11 - L11*M07 - L12*M06 ) + K12*( L02*M07 + L03*M06 - L06*M03 - L07*M02 ) + K07*( -L02*M12 - L03*M11 + L11*M03 + L12*M02 ) + K02*( L07*M12 - L12*M07 ) + K11*( L03*M07 - L07*M03 ) + K06*( -L03*M12 + L12*M03 );
		coefs[2] = K12*( L01*M07 + L02*M06 + L03*M05 - L05*M03 - L06*M02 - L07*M01 ) + L12*( -K01*M07 - K02*M06 - K03*M05 + K05*M03 + K06*M02 + K07*M01 ) + M12*( K01*L07 + K02*L06 + K03*L05 - K05*L03 - K06*L02 - K07*L01 ) + L03*( -K06*M11 - K07*M10 + K10*M07 + K11*M06 ) + K03*( L06*M11 + L07*M10 - L10*M07 - L11*M06 ) + M03*( K06*L11 + K07*L10 - K10*L07 - K11*L06 ) + K02*( L07*M11 - L11*M07 ) + K11*( L02*M07 - L07*M02 ) + K07*( -L02*M11 + L11*M02 );
		coefs[3] = K12*( L00*M07 + L01*M06 + L02*M05 + L03*M04 - L04*M03 - L05*M02 - L06*M01 - L07*M00 ) + L12*( -K00*M07 - K01*M06 - K02*M05 - K03*M04 + K04*M03 + K05*M02 + K06*M01 + K07*M00 ) + M12*( K00*L07 + K01*L06 + K02*L05 + K03*L04 - K04*L03 - K05*L02 - K06*L01 - K07*L00 ) + K11*( L01*M07 + L02*M06 + L03*M05 - L05*M03 - L06*M02 - L07*M01 ) + L11*( -K01*M07 - K02*M06 - K03*M05 + K05*M03 + K06*M02 + K07*M01 ) + M11*( K01*L07 + K02*L06 + K03*L05 - K05*L03 - K06*L02 - K07*L01 ) + L03*( -K06*M10 - K07*M09 + K09*M07 + K10*M06 ) + K03*( L06*M10 + L07*M09 - L09*M07 - L10*M06 ) + M03*( K06*L10 + K07*L09 - K09*L07 - K10*L06 ) + M07*( -K02*L10 + K10*L02 ) + K07*( -L02*M10 + L10*M02 ) + L07*( K02*M10 - K10*M02 );
		coefs[4] = K11*( L00*M07 + L01*M06 + L02*M05 + L03*M04 - L04*M03 - L05*M02 - L06*M01 - L07*M00 ) + L11*( -K00*M07 - K01*M06 - K02*M05 - K03*M04 + K04*M03 + K05*M02 + K06*M01 + K07*M00 ) + M11*( K00*L07 + K01*L06 + K02*L05 + K03*L04 - K04*L03 - K05*L02 - K06*L01 - K07*L00 ) + K12*( L00*M06 + L01*M05 + L02*M04 - L04*M02 - L05*M01 - L06*M00 ) + K10*( L01*M07 + L02*M06 + L03*M05 - L05*M03 - L06*M02 - L07*M01 ) + L10*( -K01*M07 - K02*M06 - K03*M05 + K05*M03 + K06*M02 + K07*M01 ) + L12*( -K00*M06 - K01*M05 - K02*M04 + K04*M02 + K05*M01 + K06*M00 ) + M10*( K01*L07 + K02*L06 + K03*L05 - K05*L03 - K06*L02 - K07*L01 ) + M12*( K00*L06 + K01*L05 + K02*L04 - K04*L02 - K05*L01 - K06*L00 ) + K03*( L06*M09 + L07*M08 - L08*M07 - L09*M06 ) + K07*( -L02*M09 - L03*M08 + L08*M03 + L09*M02 ) + K09*( L02*M07 + L03*M06 - L06*M03 - L07*M02 ) + K02*( L07*M09 - L09*M07 ) + K06*( -L03*M09 + L09*M03 ) + K08*( L03*M07 - L07*M03 );
		coefs[5] = K10*( L00*M07 + L01*M06 + L02*M05 + L03*M04 - L04*M03 - L05*M02 - L06*M01 - L07*M00 ) + L10*( -K00*M07 - K01*M06 - K02*M05 - K03*M04 + K04*M03 + K05*M02 + K06*M01 + K07*M00 ) + M10*( K00*L07 + K01*L06 + K02*L05 + K03*L04 - K04*L03 - K05*L02 - K06*L01 - K07*L00 ) + K11*( L00*M06 + L01*M05 + L02*M04 - L04*M02 - L05*M01 - L06*M00 ) + K09*( L01*M07 + L02*M06 + L03*M05 - L05*M03 - L06*M02 - L07*M01 ) + L11*( -K00*M06 - K01*M05 - K02*M04 + K04*M02 + K05*M01 + K06*M00 ) + M09*( K01*L07 + K02*L06 + K03*L05 - K05*L03 - K06*L02 - K07*L01 ) + L09*( -K01*M07 - K02*M06 - K03*M05 + K05*M03 + K06*M02 + K07*M01 ) + M11*( K00*L06 + K01*L05 + K02*L04 - K04*L02 - K05*L01 - K06*L00 ) + K12*( L00*M05 + L01*M04 - L04*M01 - L05*M00 ) + L12*( -K00*M05 - K01*M04 + K04*M01 + K05*M00 ) + M08*( K02*L07 + K03*L06 - K06*L03 - K07*L02 ) + L08*( -K02*M07 - K03*M06 + K06*M03 + K07*M02 ) + K08*( L02*M07 + L03*M06 - L06*M03 - L07*M02 ) + M12*( K00*L05 + K01*L04 - K04*L01 - K05*L00 );
		coefs[6] = M04*( -K00*L12 - K01*L11 - K02*L10 - K03*L09 + K09*L03 + K10*L02 + K11*L01 + K12*L00 ) + M05*( -K00*L11 - K01*L10 - K02*L09 - K03*L08 + K08*L03 + K09*L02 + K10*L01 + K11*L00 ) + M00*( K04*L12 + K05*L11 + K06*L10 + K07*L09 - K09*L07 - K10*L06 - K11*L05 - K12*L04 ) + M01*( K04*L11 + K05*L10 + K06*L09 + K07*L08 - K08*L07 - K09*L06 - K10*L05 - K11*L04 ) + M09*( K00*L07 + K01*L06 + K02*L05 + K03*L04 - K04*L03 - K05*L02 - K06*L01 - K07*L00 ) + M06*( -K00*L10 - K01*L09 - K02*L08 + K08*L02 + K09*L01 + K10*L00 ) + M02*( K04*L10 + K05*L09 + K06*L08 - K08*L06 - K09*L05 - K10*L04 ) + M08*( K01*L07 + K02*L06 + K03*L05 - K05*L03 - K06*L02 - K07*L01 ) + M10*( K00*L06 + K01*L05 + K02*L04 - K04*L02 - K05*L01 - K06*L00 ) + M11*( K00*L05 + K01*L04 - K04*L01 - K05*L00 ) + M07*( -K00*L09 - K01*L08 + K08*L01 + K09*L00 ) + M03*( K04*L09 + K05*L08 - K08*L05 - K09*L04 ) + M12*( K00*L04 - K04*L00 );
		coefs[7] = M04*( -K00*L11 - K01*L10 - K02*L09 - K03*L08 + K08*L03 + K09*L02 + K10*L01 + K11*L00 ) + M00*( K04*L11 + K05*L10 + K06*L09 + K07*L08 - K08*L07 - K09*L06 - K10*L05 - K11*L04 ) + M08*( K00*L07 + K01*L06 + K02*L05 + K03*L04 - K04*L03 - K05*L02 - K06*L01 - K07*L00 ) + M05*( -K00*L10 - K01*L09 - K02*L08 + K08*L02 + K09*L01 + K10*L00 ) + M01*( K04*L10 + K05*L09 + K06*L08 - K08*L06 - K09*L05 - K10*L04 ) + M09*( K00*L06 + K01*L05 + K02*L04 - K04*L02 - K05*L01 - K06*L00 ) + K00*( L04*M11 + L05*M10 - L08*M07 - L09*M06 ) + K04*( -L00*M11 - L01*M10 + L08*M03 + L09*M02 ) + K08*( L00*M07 + L01*M06 - L04*M03 - L05*M02 ) + K01*( L04*M10 - L08*M06 ) + K05*( -L00*M10 + L08*M02 ) + K09*( L00*M06 - L04*M02 );
		coefs[8] = M04*( -K00*L10 - K01*L09 - K02*L08 + K08*L02 + K09*L01 + K10*L00 ) + M00*( K04*L10 + K05*L09 + K06*L08 - K08*L06 - K09*L05 - K10*L04 ) + M08*( K00*L06 + K01*L05 + K02*L04 - K04*L02 - K05*L01 - K06*L00 ) + K00*( L04*M10 + L05*M09 - L08*M06 - L09*M05 ) + K04*( -L00*M10 - L01*M09 + L08*M02 + L09*M01 ) + K08*( L00*M06 + L01*M05 - L04*M02 - L05*M01 ) + K01*( L04*M09 - L08*M05 ) + K05*( -L00*M09 + L08*M01 ) + K09*( L00*M05 - L04*M01 );
		coefs[9] = K00*( L04*M09 + L05*M08 - L08*M05 - L09*M04 ) + K04*( -L00*M09 - L01*M08 + L08*M01 + L09*M00 ) + K08*( L00*M05 + L01*M04 - L04*M01 - L05*M00 ) + K01*( L04*M08 - L08*M04 ) + K05*( -L00*M08 + L08*M00 ) + K09*( L00*M04 - L04*M00 );
		coefs[10] = K00*( L04*M08 - L08*M04 ) + K04*( -L00*M08 + L08*M00 ) + K08*( L00*M04 - L04*M00 );
	}
}
