/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2008 jOpenDocument, by ILM Informatique. All rights reserved.
 * 
 * The contents of this file are subject to the terms of the GNU
 * General Public License Version 3 only ("GPL").  
 * You may not use this file except in compliance with the License. 
 * You can obtain a copy of the License at http://www.gnu.org/licenses/gpl-3.0.html
 * See the License for the specific language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each file.
 * 
 */

package org.jopendocument.dom;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import org.jopendocument.dom.ODPackage.RootElement;
import org.jopendocument.dom.spreadsheet.SheetTest;
import org.jopendocument.dom.spreadsheet.Table;
import org.jopendocument.dom.style.PageLayoutStyle;
import org.jopendocument.dom.style.SideStyleProperties.Side;
import org.jopendocument.dom.text.Heading;
import org.jopendocument.dom.text.Paragraph;
import org.jopendocument.dom.text.ParagraphStyle;
import org.jopendocument.dom.text.TextDocument;
import org.jopendocument.util.CollectionUtils;
import org.jopendocument.util.CompareUtils;
import org.jopendocument.util.TimeUtils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Calendar;
import java.util.List;
import java.util.TimeZone;

import javax.xml.datatype.Duration;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.xpath.XPath;

public class ODSingleXMLDocumentTest extends TestCase {

    // necessary since JUnit only accept classes and not instances
    static private XMLVersion staticVersion;

    public static Test suite() {
        final TestSuite suite = new TestSuite("Test for ODSingleXMLDocument");
        for (final XMLVersion v : XMLVersion.values()) {
            staticVersion = v;
            // works since an instance is created by this call and thus uses staticVersion
            suite.addTest(new TestSuite(ODSingleXMLDocumentTest.class, v.toString()));
        }

        return suite;
    }

    static public final void assertEquals(final BigDecimal o1, final BigDecimal o2) {
        assertTrue(CompareUtils.equalsWithCompareTo(o1, o2));
    }

    private final XMLVersion version;

    public ODSingleXMLDocumentTest() {
        this(staticVersion);
    }

    public ODSingleXMLDocumentTest(XMLVersion version) {
        super();
        this.version = version;
    }

    private ODPackage createPackage() throws IOException {
        return createPackage("test");
    }

    private ODPackage createPackage(final String name) throws IOException {
        return new ODPackage(this.getClass().getResourceAsStream(name + "." + ContentType.TEXT.getVersioned(this.version).getExtension()));
    }

    protected void setUp() throws Exception {
        super.setUp();
    }

    private void assertValid(final ODSingleXMLDocument single) {
        SheetTest.assertValid(single.getPackage());
    }

    private void assertValid(final ODDocument doc) {
        SheetTest.assertValid(doc.getPackage());
    }

    public void testCreate() throws Exception {
        final ODSingleXMLDocument single = createPackage().toSingle();
        assertValid(single);
        assertTrue(single.getPackage().isSingle());

        // test createFromDocument()
        final ODPackage pkg1 = ODPackage.createFromDocuments((Document) single.getDocument().clone(), null);
        assertTrue(pkg1.isSingle());
        assertEquals(single.getPackage().getContentType(), pkg1.getContentType());

        final ODPackage emptyPkg = new ODPackage();
        final ContentType[] types = ContentType.values();
        final ContentTypeVersioned[] versTypes = ContentTypeVersioned.values();
        for (final ContentTypeVersioned ct : versTypes) {
            assertEquals(ct.isTemplate(), ct.getTemplate() == ct);
            assertEquals(!ct.isTemplate(), ct.getNonTemplate() == ct);
            if (ct.isTemplate())
                assertSame(ct, ct.getNonTemplate().getTemplate());
            else if (ct.getTemplate() != null)
                assertSame(ct, ct.getTemplate().getNonTemplate());

            final ODPackage pkg = ct.createPackage(OOXML.getLast(ct.getVersion()).getFormatVersion());
            assertEquals(ct, pkg.getContentType());
            assertEquals(0, pkg.validateSubDocuments().size());

            // OOo has no templates and only one version
            if (ct.getVersion() == XMLVersion.OD) {
                pkg.setTemplate(true);
                assertTrue(pkg.isTemplate());
                pkg.setTemplate(false);
                assertFalse(pkg.isTemplate());

                try {
                    pkg.putFile(RootElement.META.getZipEntry(), RootElement.META.createDocument(XMLFormatVersion.getOOo()));
                    fail("should throw since XML version is not compatible");
                } catch (Exception e) {
                    assertTrue(e.getMessage().contains("Cannot change version"));
                }
                try {
                    pkg.putFile(RootElement.META.getZipEntry(), RootElement.META.createDocument(XMLFormatVersion.get(ct.getVersion(), "1.0")));
                    fail("should throw since office version is not compatible");
                } catch (Exception e) {
                    assertTrue(e.getMessage().contains("Cannot change format version"));
                }
            }
            // no-op
            pkg.setContentType(pkg.getContentType());
            try {
                // same version but different type
                pkg.setContentType(types[(ct.getType().ordinal() + 3) % types.length].getVersioned(ct.getVersion()));
                fail("should throw since type is not compatible");
            } catch (Exception e) {
                assertTrue(e.getMessage().contains("Cannot change type"));
            }

            // an empty package can be changed to anything
            emptyPkg.setContentType(ct);
            assertEquals(ct, emptyPkg.getContentType());
        }

        // TextDocument
        {
            final TextDocument emptyTextDoc = TextDocument.createEmpty("PPPPP");
            assertValid(emptyTextDoc);
            assertEquals(ContentType.TEXT, emptyTextDoc.getPackage().getContentType().getType());
            assertEquals("PPPPP", emptyTextDoc.getParagraph(0).getElement().getText());
            try {
                emptyTextDoc.getParagraph(1);
                fail("Only 1 paragraph");
            } catch (IndexOutOfBoundsException e) {
                // ok
            }
        }
    }

    public void testSplit() throws Exception {
        final ODPackage pkg = createPackage();
        SheetTest.assertValid(pkg);
        final ODSingleXMLDocument single = pkg.toSingle();
        assertSingle(pkg, single);

        // a valid package must not contain a single xml document
        final ByteArrayOutputStream out = new ByteArrayOutputStream(32000);
        pkg.save(out);
        assertFalse(new ODPackage(new ByteArrayInputStream(out.toByteArray())).isSingle());

        // split
        assertTrue(pkg.split());
        // the second time nothing happens
        assertFalse(pkg.split());
        assertTrue(single.isDead());
        assertNull(single.getPackage());
        assertFalse(pkg.isSingle());
        assertEquals(0, pkg.validateSubDocuments().size());

        // we can convert once more to single
        assertSingle(pkg, pkg.toSingle());
    }

    private void assertSingle(final ODPackage pkg, final ODSingleXMLDocument single) {
        assertValid(single);
        assertFalse(single.isDead());
        assertSame(pkg, single.getPackage());
        assertTrue(pkg.isSingle());
    }

    public void testAdd() throws Exception {
        final ODSingleXMLDocument single = new ODPackage(this.getClass().getResourceAsStream("empty.odt")).toSingle();
        // really empty
        single.getBody().removeContent();
        final ODSingleXMLDocument single2 = new ODPackage(this.getClass().getResourceAsStream("styles.odt")).toSingle();
        single.add(single2);
        assertValid(single);
    }

    public void testAddParagraph() throws Exception {
        final TextDocument single = new ODPackage(this.getClass().getResourceAsStream("styles.odt")).getTextDocument();

        final Paragraph p = new Paragraph();
        p.setStyle("testPragraph");
        p.addContent("Hello");
        p.addTab();
        p.addStyledContent("World", "testChar");
        // add at the end
        assertNull(p.getElement().getDocument());
        p.setDocument(single);
        assertSame(single.getContentDocument(), p.getElement().getDocument());

        final Heading h = new Heading();
        h.setStyle("inexistantt");
        h.addContent("Heading text");
        try {
            h.setDocument(single);
            fail("should throw since style doesn't exist");
        } catch (Exception e) {
            // ok
        }
        h.setStyle("testPragraph");
        // add before p
        final Element pParent = p.getElement().getParentElement();
        single.add(h, pParent, pParent.indexOf(p.getElement()));

        assertValid(single);

        // rm
        p.setDocument(null);
        assertNull(p.getElement().getDocument());
    }

    public void testTable() throws Exception {
        final TextDocument textDoc = createPackage().getTextDocument();
        assertValid(textDoc);
        final ODXMLDocument single = textDoc.getPackage().getContent();
        assertNull(single.getDescendantByName("table:table", "inexistant"));
        final Element table = single.getDescendantByName("table:table", "JODTestTable");
        assertNotNull(table);
        final Table<TextDocument> t = new Table<TextDocument>(textDoc, table);

        assertEquals(1, t.getHeaderRowCount());
        assertEquals(0, t.getHeaderColumnCount());

        final Calendar c = Calendar.getInstance();
        c.clear();
        c.set(2005, 0, 12, 12, 35);
        assertEquals(c.getTime(), t.getValueAt(2, 1));

        // 11.91cm
        assertEquals(119.06f, t.getWidth());
        t.setColumnCount(6, -1, true);
        t.setValueAt(3.14, 5, 0);
        assertTableWidth(t, 119.06f);
        final float ratio = t.getColumn(0).getWidth() / t.getColumn(1).getWidth();
        t.setColumnCount(2, 1, true);
        // ratio is kept
        assertEquals(ratio, t.getColumn(0).getWidth() / t.getColumn(1).getWidth());
        assertTableWidth(t, 119.06f);
        // table changes width
        final float width = t.getColumn(0).getWidth();
        t.setColumnCount(4, 0, false);
        assertTableWidth(t, 119.06f + 2 * width);
        t.setColumnCount(1, 123, false);
        assertEquals(1, t.getColumnCount());
        assertTableWidth(t, width);

        assertValid(textDoc);

        t.detach();
        assertNull(single.getDescendantByName("table:table", "JODTestTable"));
    }

    private void assertTableWidth(Table<?> t, float w) {
        assertEquals(w, t.getWidth());
        float total = 0;
        for (int i = 0; i < t.getColumnCount(); i++) {
            total += t.getColumn(i).getWidth();
        }
        assertEquals(round(w), round(total));
    }

    private long round(float w) {
        return Math.round(w * 100.0) / 100;
    }

    public void testFrame() throws Exception {
        final ODPackage pkg = createPackage();
        final Element frameElem = pkg.getContent().getDescendantByName(pkg.getFormatVersion().getXML().getFrameQName(), "Cadre1");

        final ODFrame<ODDocument> frame = new ODFrame<ODDocument>(pkg.getODDocument(), frameElem);
        // for some reason OO converted the 72 to 73 during export
        final BigDecimal width = this.version == XMLVersion.OOo ? new BigDecimal("42.73") : new BigDecimal("42.72");
        assertEquals(width, frame.getWidth());
        // height depends on the content
        assertNull(frame.getHeight());

        assertEquals("right", frame.getStyle().getGraphicProperties().getHorizontalPosition());
        assertEquals("paragraph", frame.getStyle().getGraphicProperties().getHorizontalRelation());

        assertEquals(asList("position"), frame.getStyle().getGraphicProperties().getProtected());
        assertTrue(frame.getStyle().getGraphicProperties().isContentPrinted());
    }

    public void testStyle() throws Exception {
        final StyleDesc<ParagraphStyle> pDesc = Style.getStyleDesc(ParagraphStyle.class, this.version);
        {
            final ODPackage pkg = createPackage();

            // in OOo format there's no name attribute
            if (pkg.getVersion() != XMLVersion.OOo) {
                // test that multiple attributes may refer to paragraph styles
                final XPath ellipseXPath = pkg.getContent().getXPath("//draw:ellipse[@draw:name = 'Ellipse']");
                final Element ellipse = (Element) ellipseXPath.selectSingleNode(pkg.getContent().getDocument());
                final String drawTextStyleName = ellipse.getAttributeValue("text-style-name", ellipse.getNamespace("draw"));
                final ParagraphStyle ellipseTextStyle = pDesc.findStyleWithName(pkg, pkg.getContent().getDocument(), drawTextStyleName);
                assertEquals(singletonList(ellipse), ellipseTextStyle.getReferences());
            }

            // P1 exists both in content and styles but doesn't denote the same style
            final ODXMLDocument styles = pkg.getXMLFile("styles.xml");
            final XPath headerXPath = styles.getXPath("//text:p[string() = 'Header']");
            final Element headerElem = (Element) headerXPath.selectSingleNode(styles.getDocument());
            final String styleName = new Paragraph(headerElem).getStyleName();

            final Element styleP1 = pkg.getStyle(pkg.getDocument("styles.xml"), pDesc, styleName);
            final Element contentP1 = pkg.getStyle(pkg.getDocument("content.xml"), pDesc, styleName);
            assertNotNull(styleP1);
            assertNotNull(contentP1);
            assertNotSame(styleP1, contentP1);

            // styleP1 can only be referenced from styles.xml
            assertEquals(singletonList(headerElem), StyleStyle.warp(pkg, styleP1).getReferences());
            // contentP1 can only be referenced from content.xml
            assertFalse(StyleStyle.warp(pkg, contentP1).getReferences().contains(headerElem));

            // test non-StyleStyle
            final PageLayoutStyle pm1Style = Style.getStyle(pkg, PageLayoutStyle.class, "pm1");
            assertNull(pm1Style.getBackgroundColor());
            assertNull(pm1Style.getPageLayoutProperties().getBorder(Side.TOP));
            assertEquals(new BigDecimal(2), pm1Style.getPageLayoutProperties().getMargin(Side.TOP, LengthUnit.CM));
            // only style:master-page points to pm1
            assertSame(pkg.getXMLFile("styles.xml").getChild("master-styles").getChildren().get(0), CollectionUtils.getSole(pm1Style.getReferences()));

            // when merging styles.xml
            pkg.toSingle();
            // the content doesn't change
            assertSame(contentP1, pkg.getStyle(pkg.getDocument("content.xml"), pDesc, styleName));
            final String mergedStyleName = new Paragraph((Element) headerXPath.selectSingleNode(pkg.getContent().getDocument())).getStyleName();
            assertNotNull(mergedStyleName);
            assertFalse(styleName.equals(mergedStyleName));
        }

        final ODPackage pkg = createPackage();
        final Element heading = (Element) pkg.getContent().getXPath("//text:h[string() = 'Titre 2']").selectSingleNode(pkg.getContent().getDocument());
        final String styleName = heading.getAttributeValue("style-name", heading.getNamespace());
        // common styles are not in content.xml
        assertNull(pkg.getContent().getStyle(pDesc, styleName));
        // but in styles.xml
        testStyleElem(pkg.getXMLFile("styles.xml").getStyle(pDesc, styleName));
        testStyleElem(pkg.getStyle(pDesc, styleName));
        // except if we merge the two
        pkg.toSingle();
        testStyleElem(pkg.getContent().getStyle(pDesc, styleName));
        testStyleElem(pkg.getStyle(pDesc, styleName));

        // test that we can use StyleStyle instances to warp default-style
        // (was causing problems since the element name isn't the normal one, e.g. style:style)
        final ParagraphStyle defaultPStyle = Style.getStyleStyleDesc(ParagraphStyle.class, this.version).findDefaultStyle(pkg);
        assertEquals(StyleStyleDesc.ELEMENT_DEFAULT_NAME, defaultPStyle.getElement().getName());
        assertEquals("Times New Roman", defaultPStyle.getTextProperties().getAttributeValue("font-name", this.version.getSTYLE()));
    }

    private void testStyleElem(final Element styleElem) {
        assertNotNull(styleElem);
        // OOo has text:level="2" on text:h
        if (this.version != XMLVersion.OOo)
            assertEquals("2", styleElem.getAttributeValue("default-outline-level", styleElem.getNamespace()));
        assertEquals("Heading", styleElem.getAttributeValue("parent-style-name", styleElem.getNamespace()));
    }

    public void testMeta() throws Exception {
        final ODPackage pkg = createPackage();
        final ODMeta meta = pkg.getMeta();
        assertEquals("firstInfo", meta.getUserMeta("Info 1").getValue());
        assertEquals("", meta.getUserMeta("secondName").getValue());

        final List<String> expected = Arrays.asList("Info 1", "secondName", "Info 3", "Info 4");
        assertEquals(expected, meta.getUserMetaNames());

        // does not exist
        assertNull(meta.getUserMeta("toto"));
        // now it does
        assertNotNull(meta.getUserMeta("toto", true));
        meta.removeUserMeta("toto");
        // now it was removed
        assertNull(meta.getUserMeta("toto"));
        final ODUserDefinedMeta toto = meta.getUserMeta("toto", true);
        toto.setValue(3.5);
        assertEquals(ODValueType.FLOAT, toto.getValueType());
        assertEquals(3.5f, ((BigDecimal) toto.getValue()).floatValue());
        final TimeZone marquisesTZ = TimeZone.getTimeZone("Pacific/Marquesas");
        final TimeZone pstTZ = TimeZone.getTimeZone("PST");
        final Calendar cal = Calendar.getInstance(pstTZ);
        final int hour = cal.get(Calendar.HOUR_OF_DAY);
        final int minute = cal.get(Calendar.MINUTE);

        {
            toto.setValue(cal, ODValueType.TIME);
            Duration actual = (Duration) toto.getValue();
            assertEquals(hour, actual.getHours());
            assertEquals(minute, actual.getMinutes());
            // for TIME the time zone is important, the same date has not the same value
            cal.setTimeZone(marquisesTZ);
            toto.setValue(cal, ODValueType.TIME);
            actual = (Duration) toto.getValue();
            // +1h30
            assertFalse(hour == actual.getHours());
            assertFalse(minute == actual.getMinutes());
            // test that we used the time part of the calendar
            final Calendar startInstant = (Calendar) cal.clone();
            startInstant.clear();
            // don't use set() otherwise it can moves to the next day
            startInstant.set(cal.get(Calendar.YEAR), cal.get(Calendar.MONTH), cal.get(Calendar.DAY_OF_MONTH));
            assertEquals(0, startInstant.get(Calendar.HOUR_OF_DAY));
            assertEquals(cal.getTimeInMillis() - startInstant.getTimeInMillis(), actual.getTimeInMillis(startInstant));

            final Duration duration = TimeUtils.getTypeFactory().newDurationDayTime(false, 2, 2, 2, 2);
            toto.setValue(duration);
            assertEquals(ODValueType.TIME, toto.getValueType());
            assertEquals(duration, toto.getValue());
        }

        {
            cal.setTimeZone(pstTZ);
            toto.setValue(cal, ODValueType.DATE);
            assertEquals(cal.getTime(), toto.getValue());
            // on the contrary for DATE the timezone is irrelevant
            cal.setTimeZone(marquisesTZ);
            toto.setValue(cal, ODValueType.DATE);
            assertEquals(cal.getTime(), toto.getValue());
        }
    }
}
